/*
* ============================================================================
*  Name     : CTaskManagerAppView from TaskManagerAppView.h
*  Part of  : TaskManager
*  Created  : 08/31/2005 by Forum Nokia
*  Version  : 1.0
*  Copyright: Nokia Corporation
* ============================================================================
*/

// INCLUDE FILES
#include <coemain.h>
#include <eiklbx.h>
#include <eiktxlbm.h> // CTextListBoxModel
#include <eikclb.h>			// CEikColumnListBox
#include <barsread.h>			// TResourceReader
#include <cknflash.h>	// CCknFlashingDialog
#include <cknconf.h>	// CCknConfirmationDialog
#include <eikclbd.h>
#include <SocketTaskManager.rsg>
#include <stringloader.h>
#include <e32std.h>
#include "TaskManager.hrh"
#include "TaskManagerAppView.h"
#include "TaskManagerAppUi.h"
#include "Response.h"
#include "TaskManagerIapDialog.h"

// CONSTANTS
#define KListPosition TPoint(0,0)
_LIT(KTab, "\t");
_LIT(KError, "Error: %d");
_LIT(KNoTasks, "No Tasks!");
_LIT(KLoadingTasks, "Loading tasks...");
_LIT(KCompletingTask, "Completing task...");
_LIT(KTaskCompleted, "\n\nTask completed?");
_LIT(KInvalidTask, "Invalid task. Cannot complete.");
_LIT(KTaskFormat, "%d\t%S");
_LIT(KOpeningConnection, "Opening connection...");
_LIT(KTitle, "TaskManager");
const TInt KMaxErrorLength = 30;


// ================= MEMBER FUNCTIONS =======================

// constructor
CTaskManagerAppView::CTaskManagerAppView(CTaskManagerAppUi& aAppUi)
	:	iAppUi(aAppUi)
	{
	iStatusText = KNoTasks;
	}

// destructor
CTaskManagerAppView::~CTaskManagerAppView()
	{
	delete iTaskList;
	iTaskList = 0;
	}

// ----------------------------------------------------
// CTaskManagerAppView::NewL()
// Two-phased constructor.
// ----------------------------------------------------
//
CTaskManagerAppView *CTaskManagerAppView::NewL(const TRect& aRect, CTaskManagerAppUi& aAppUi)
	{
    CTaskManagerAppView *self = new(ELeave) CTaskManagerAppView(aAppUi);
    CleanupStack::PushL(self);
    self->ConstructL(aRect);
    CleanupStack::Pop(self);
    return self;
	}

// ----------------------------------------------------
// CTaskManagerAppView::ConstructL()
// Symbian OS default constructor can leave.
// ----------------------------------------------------
//	
void CTaskManagerAppView::ConstructL(const TRect& aRect)
	{
    // Create a window for this application view
    CreateWindowL();
	
	CreateListL();
	
    // Set the windows size
    SetRect(aRect);

    // Activate the window, which makes it ready to be drawn
    ActivateL();
	}

// ----------------------------------------------------
// CTaskManagerAppView::ConstructL()
// Creates a listbox that is used for showing the tasks
// to the user.
// ----------------------------------------------------
//
void CTaskManagerAppView::CreateListL()
	{
	iTaskList = new(ELeave)CEikColumnListBox; 
	iTaskList->ConstructL(this, CEikListBox::ENoExtendedSelection);
	iTaskList->CreateScrollBarFrameL(ETrue);
	iTaskList->ScrollBarFrame()->SetScrollBarVisibilityL(
        CEikScrollBarFrame::EOff, CEikScrollBarFrame::EAuto);
	iTaskList->SetContainerWindowL(*this);
	iTaskList->MakeVisible(EFalse);
	}


// ----------------------------------------------------
// CTaskManagerAppView::Draw()
// This function is used for window server-initiated 
// redrawing of controls, and for some 
// application-initiated drawing. Here we show the 
// status text to the user.
// ----------------------------------------------------
//
void CTaskManagerAppView::Draw(const TRect& /*aRect*/) const
	{
    // Get the standard graphics context 
    CWindowGc &gc = SystemGc();
    
    // Gets the control's extent
    TRect rect = Rect();
    
    // Clears the screen
    gc.Clear(rect);
    
    // status text is showed only if iTaskList isn't visible
    if( !iTaskList->IsVisible() )
    	{
   		gc.UseFont( iCoeEnv->NormalFont() );
   		
   		// This is done to center the text
   		TInt pointX = rect.Width() / 2 -
   					  iCoeEnv->NormalFont()->TextWidthInPixels( iStatusText ) / 2;

   		TInt pointY = rect.Height() / 2 -
   					  iCoeEnv->NormalFont()->HeightInPixels() / 2;
   					  
		gc.DrawText( iStatusText, TPoint( pointX, pointY ) );
    	}

	}

// ----------------------------------------------------
// CTaskManagerAppView::CompleteTaskL()
// This function marks the selected task as completed in 
// the server database.
// ----------------------------------------------------
//	
void CTaskManagerAppView::CompleteTaskL()
	{
		const MDesCArray* items = iTaskList->Model()->MatchableTextArray();
		
		if (iTaskList->CurrentItemIndex() < 0)
			{
			return;
			}
		
		// We get the currently selected item's text to print it in a query
		TPtrC pointer = items->MdcaPoint( iTaskList->CurrentItemIndex() );
		
		TInt tabOffSet = pointer.Find(KTab);
		// id of the task was not found.
		if (tabOffSet == KErrNotFound || tabOffSet == 0)
			{
			CCknFlashingDialog::RunDlgLD(KTitle, KInvalidTask);
			return;
			}
		
		TLex lex(pointer.Left(tabOffSet));
		TInt taskId;
		User::LeaveIfError(lex.Val(taskId));
		
		// (KMaxTaskLength +20) space for task and KTaskCompleted
		TBuf<KMaxTaskLength + 20> message = pointer.Mid(tabOffSet+1);
		message.Append( KTaskCompleted ); 
		
		iAppUi.SetViewBusy(ETrue);
		if (CCknConfirmationDialog::RunDlgLD(KTitle, message))
			{
			// show 'Completing task' to the user.
			ShowStatus(KCompletingTask);
			
			iAppUi.ShowConnectingCbaL(ETrue);
			iAppUi.Model().MarkTaskDoneL(taskId);
			iTransactionStatus = EMarkingTaskDone;
			}
		
		iAppUi.SetViewBusy(EFalse);
	}

// ----------------------------------------------------
// CTaskManagerAppView::DeleteSelectedTaskL()
// Removes the selected task from the listbox.
// ----------------------------------------------------
//
void CTaskManagerAppView::DeleteSelectedTaskL()
	{
	CTextListBoxModel* model = iTaskList->Model();
	CDesCArray* itemArray = static_cast<CDesCArray*>( model->ItemTextArray() );
	
	TInt currentItem = iTaskList->CurrentItemIndex();
	
	itemArray->Delete( currentItem );
	
	// no more tasks, show 'No tasks' to user.													 
	if( model->NumberOfItems() == 0 )
		{
		iTaskList->MakeVisible( EFalse );
		}
	else
		{
		if (currentItem >= model->NumberOfItems()) 
			{
			iTaskList->SetCurrentItemIndex(currentItem-1);
			}
		else
			{
			iTaskList->SetCurrentItemIndex(currentItem);
			}
		}
	iTaskList->DrawNow();		
	}

// ----------------------------------------------------
// CTaskManagerAppView::ReadTasksL()
// Reads tasks from aResponse and adds them to the listbox.
// ----------------------------------------------------
//
void CTaskManagerAppView::ReadTasksL( const CResponse& aResponse )
	{
	CTextListBoxModel* model = iTaskList->Model();
	
	model->SetOwnershipType( ELbmOwnsItemArray ); // Just to underline the relation
	CDesCArray* itemArray = static_cast<CDesCArray*>( model->ItemTextArray() );
										 
	itemArray->Reset();
	
	TInt taskCount = aResponse.TaskCount();
	TBuf<KMaxTaskLength+10> taskDesc;
	for (TInt i = 0; i < taskCount; i++)
		{
		TBuf<KMaxTaskLength> desc = aResponse.TaskDescription(i);
		taskDesc.Format(KTaskFormat, aResponse.TaskId(i), &desc);
		itemArray->AppendL(taskDesc);
		}
	
	// If there are items in the listbox, make the listbox visible.
	if( model->NumberOfItems() > 0 )
		{
		iTaskList->HandleItemAdditionL();
		iTaskList->DrawNow();
		iTaskList->MakeVisible( ETrue );
		iTaskList->SetFocus(ETrue);
		}
	}

// ----------------------------------------------------
// CTaskManagerAppView::CountComponentControls()
// Gets the number of controls contained in a compound 
// control. 
// ----------------------------------------------------
//
TInt CTaskManagerAppView::CountComponentControls() const
	{
	TInt count = 0;
	if (iTaskList)
		{
		count++;
		}
		
	return count;
	}

// ----------------------------------------------------
// CTaskManagerAppView::ComponentControl()
// Gets the specified component of a compound control.
// ----------------------------------------------------
//
CCoeControl* CTaskManagerAppView::ComponentControl( TInt aIndex ) const
	{
	switch( aIndex )
		{
		case 0:
			return iTaskList;
		default:
			return 0;
		};
	}

// ----------------------------------------------------
// CTaskManagerAppView::SizeChanged()
// Responds to size changes to sets the size and 
// position of the contents of this control. 
// ----------------------------------------------------
//	
void CTaskManagerAppView::SizeChanged()
	{
	iTaskList->SetRect(Rect());
	
    CColumnListBoxData* data = iTaskList->ItemDrawer()->ColumnData();
    TRAPD(err, data->SetColumnWidthPixelL(0, 0);) // hide the id column
    TRAP(err, data->SetColumnWidthPixelL(1, Rect().Width());) // show the task description

    // Update scroll bar to match the view size.
    TRAP(err,iTaskList->UpdateScrollBarsL();)
   	}

// ----------------------------------------------------
// CTaskManagerAppView::OfferKeyEventL()
// When a key event occurs, the control framework calls 
// this function for each control on the control stack, 
// until one of them can process the key event 
// (and returns EKeyWasConsumed).
// ----------------------------------------------------
//	
TKeyResponse CTaskManagerAppView::OfferKeyEventL( const TKeyEvent& aKeyEvent,
												  TEventCode aType )
	{
	if( iTaskList && iTaskList->IsVisible() )
		{
		return iTaskList->OfferKeyEventL( aKeyEvent, aType );
		}
	else
		{
		return EKeyWasNotConsumed;
		}
	}

// ----------------------------------------------------
// CTaskManagerAppView::OpeningConnectionL()
// Called when a GPRS connection is opened.
// ----------------------------------------------------
//	
void CTaskManagerAppView::OpeningConnectionL()
	{
	ShowStatus(KOpeningConnection);
	iAppUi.ShowConnectingCbaL(ETrue);
	}
	
// ----------------------------------------------------
// CTaskManagerAppView::ConnectingL()
// Called when a transaction is initiated. 
// ----------------------------------------------------
//
void CTaskManagerAppView::ConnectingToServerL(const TBool& aLoadingTasks)
	{
	if (aLoadingTasks)
		{
		ShowStatus(KLoadingTasks);
		}
	else
		{
		ShowStatus(KCompletingTask);
		}

	// show cancel button.
	iAppUi.ShowConnectingCbaL(ETrue);
	}

// ----------------------------------------------------
// CTaskManagerAppView::SuccessL()
// Called when the transaction is successfully 
// finished. 
// ----------------------------------------------------
//	
void CTaskManagerAppView::SuccessL(const CResponse& aResponse)
	{
	// even though the transaction was successful, errors might have occurred
	// in the server database.
	if (aResponse.HasError())
		{
		CCknFlashingDialog::RunDlgLD(KTitle, aResponse.Error());
		}
	// no errors.
	else
		{
		// we were completing a task, remove it from the listbox.
		if (iTransactionStatus == EMarkingTaskDone)
			{
			iTransactionStatus = EFetchingTasks;
			DeleteSelectedTaskL();
			}
		// we were loading tasks, show them in the listbox.
		else
			{
			ReadTasksL(aResponse);
			}
		}
	
	// will show 'No tasks' or if tasks exist, a list of tasks is shown.
	ShowStatus(KNoTasks);
	
	iAppUi.ShowConnectingCbaL(EFalse);
	}

// ----------------------------------------------------
// CTaskManagerAppView::FailedL()
// Called when the transaction failed. 
// ----------------------------------------------------
//	
void CTaskManagerAppView::FailedL(const TInt& aError)
	{
	TBuf<KMaxErrorLength> error;
	error.Format(KError, aError);
	CCknFlashingDialog::RunDlgLD(KTitle, error);
	
	// will show 'No tasks' or if tasks exist, a list of tasks is shown.
	ShowStatus(KNoTasks);

	iAppUi.ShowConnectingCbaL(EFalse);
	}
	
// ----------------------------------------------------
// CTaskManagerAppView::CancelledL()
// Called when the transaction was cancelled by 
// the user. 
// ----------------------------------------------------
//	
void CTaskManagerAppView::CancelledL()
	{
	// will show 'No tasks' or if tasks exist, a list of tasks is shown.
	ShowStatus(KNoTasks);

	iAppUi.ShowConnectingCbaL(EFalse);
	}

// ----------------------------------------------------
// CTaskManagerAppView::ErrorL()
// Called when connection settings are invalid and the 
// transaction cannot be initiated. This occurs 
// e.g. when username and/or password is not set.
// ----------------------------------------------------
//	
void CTaskManagerAppView::ErrorL(const TDesC& aErrorMsg)
	{
	CCknFlashingDialog::RunDlgLD(KTitle, aErrorMsg);
	
	// will show 'No tasks' or if tasks exist, a list of tasks is shown.
	ShowStatus(KNoTasks);

	iAppUi.ShowConnectingCbaL(EFalse);
	}

// ----------------------------------------------------
// CTaskManagerAppView::QueryIapL()
// Opens up a dialog containing all IAPs. User 
// selects from the list the IAP that he/she wants to use.
// ----------------------------------------------------
//		
TBool CTaskManagerAppView::QueryIapL(TUint32& aId, const TUint32& aDefaultId)
	{
	TBool retval = EFalse;
	RArray<TIap>& iaps = iAppUi.Model().Iaps();
	TInt iapCount = iaps.Count();
	
	CDesCArrayFlat* iapArray = new (ELeave) CDesCArrayFlat(iapCount);
	CleanupStack::PushL(iapArray);
	
	TInt selectedIndex = 0;
	
	// Load all IAPs to the list.
	for (TInt i = 0; i < iapCount; i++)
		{
		if (iaps[i].iId == aDefaultId)
			{
			selectedIndex = i;
			}			
		iapArray->AppendL(iaps[i].iName);
		}
		
	TInt index(selectedIndex);
	CTaskManagerIapDialog* dlg = new (ELeave) CTaskManagerIapDialog(*iapArray, index);
	if (dlg->ExecuteLD(R_TASKMANAGER_IAP_SELECTION_DIALOG))
		{
		aId = iaps[index].iId;
		retval = ETrue;
		}
		
	CleanupStack::PopAndDestroy(iapArray);	
	return retval;
	}

// ----------------------------------------------------
// CTaskManagerAppView::ShowStatus()
// Shows proper status text to the user.
// ----------------------------------------------------
//	
void CTaskManagerAppView::ShowStatus(const TDesC& aStatus)
	{
	iStatusText = aStatus;
	
	// Show 'No tasks' or if tasks exist, show list of tasks.
	if (aStatus == KNoTasks)
		{
		if (iTaskList->Model()->NumberOfItems() > 0)
			{
			iTaskList->MakeVisible(ETrue);
			iAppUi.TasksExist(ETrue);
			}
		else
			{
			iAppUi.TasksExist(EFalse);
			}
		}
	// Show only status text.
	else
		{
		iTaskList->MakeVisible(EFalse);
		}
		
	DrawNow();
	}
	
// End of file