/*
 * Copyright 2005 Nokia. All rights reserved.
 */
package com.forum.nokia.taskmanager;

/**
 * This class handles parsing of the client's message into a more handlable format.
 * It divides the message into variables by splitting the message according the
 * separator constant. All the work is done in the constructor, which takes the
 * client message as a parameter. All the other functions are merely getters for
 * the parsed data.
 * 
 */
public class ClientPackage
{
    /**
     * "Enumerations" of possible commands.
     */
    public static final int NOT_SET = -1;
    public static final int FETCH = 0;
    public static final int MARK_DONE = 1;
    public static final int ERROR = 2;
    
    /**
     * Constants used in parsing the client's message.
     */
    public static final String SEPARATOR = "#";
    public static final String FETCH_STRING = "fetch";
    public static final String MARK_DONE_STRING = "mark";
    
    /**
     * These constants determine the order in which these parameters are
     * expected to be found in the client message.
     */
    public static final int USERNAME = 0;
    public static final int PASSWORD = 1;
    public static final int OPERATION = 2;
    public static final int ID = 3;
    
    /**
     * These constants define how many tokens are associated with each command.
     */
    private static final int FETCH_TOKEN_AMOUNT = 3;
    private static final int MARK_DONE_TOKEN_AMOUNT = 4;
    
    // member variables
    private int commandType = NOT_SET;
    private String username = "";
    private String password = "";
    private int messageId = NOT_SET;
    
    /**
     * The constructor parses the client's message in to an object.
     * 
     * @param clientMessage Message received from client.
     * @throws IllegalArgumentException Thrown if the message is in illegal format.
     */
    public ClientPackage( String clientMessage ) throws IllegalArgumentException
    {
        // The received message is in the format
        // "username|password|operation|[id|]" where the last argument's existence
        // depends on wether the operation is "fetch" or "mark"

        // Message string is tokenized
        String[] tokenized = clientMessage.split( SEPARATOR );

        // We assume the first token is username and the second is password
        this.username = tokenized[USERNAME];
        this.password = tokenized[PASSWORD];
        
        // operation is the third token, but it needs to be converted into
        // another format
        String operation = tokenized[OPERATION];
        if( operation.equals(FETCH_STRING) )
        {
            this.commandType = FETCH;
        }
        else if( operation.equals(MARK_DONE_STRING) )
        {
            this.commandType = MARK_DONE;
        }
        else
        {
            throw new IllegalArgumentException("Unknown command");
        }
        // if the command is identified as "mark task done", we expect there
        // to be a fourth token, representing the id of the task being marked.
        if( commandType == MARK_DONE )
        {
            try
            {
                this.messageId = Integer.parseInt( tokenized[ID] );
            }
            catch( NumberFormatException e )
            {
                throw new IllegalArgumentException( "Illegal task id" );
            }
        }
        
        // Finally, we check that the amount of tokens found is correct
        switch( this.commandType )
        {
            case FETCH:
            {
                if( tokenized.length != FETCH_TOKEN_AMOUNT )
                {
                    throw new IllegalArgumentException("Too many tokens");
                }
                break;
            }
            case MARK_DONE:
            {
                if( tokenized.length != MARK_DONE_TOKEN_AMOUNT )
                {
                    throw new IllegalArgumentException("Too many tokens");
                }
                break;
            }
        }
    }
    
    /**
     * Function returns the type of command parsed from the client message.
     * @return Command type as integer.
     */
    public int getCommandType() { return this.commandType; }
    
    /**
     * Function returns the username parsed from the client message. 
     * @return Username in string form.
     */
    public String getUsername() { return this.username; }
    
    /**
     * Function returns the password parsed from the client message.
     * @return Password in string form.
     */
    public String getPassword() { return this.password; }
    
    /**
     * Function returns the message id, which is needed when marking a certain
     * task done.
     * @return Task id number as integer or -1 (NOT_SET) if not set.
     */
    public int getMessageId() { return this.messageId; }
}
