/*
 * Copyright 2005 Nokia. All rights reserved.
 */
package com.forum.nokia.taskmanager;

import java.util.*;
import com.forum.nokia.taskmanager.Task;

/**
 * This class handles the parsing of a server-to-client message. You only need to
 * specify the type of response, tasks to be sent back (in case response type is
 * "fetch" e.g. we are responding to a "fetch tasks" -command from the client) and
 * a possible error message (in the event we want to report an error to the client).
 * 
 */
public class ServerPackage
{
    // Member variables
    private int commandType;
    private ArrayList tasks = null;
    private String errorMessage = "";
    
    /**
     * Constructor for responses FETCH and MARK_DONE. Checks that the response type
     * is ok and establishes the variables.
     * 
     * @param commandType Type of response.
     * @throws IllegalArgumentException
     */
    public ServerPackage( int commandType ) throws IllegalArgumentException
    {
        // Response type must be either "Fetch tasks" or "Mark task done"
        if( commandType != ClientPackage.FETCH &&
            commandType != ClientPackage.MARK_DONE )
        {
            throw new IllegalArgumentException("Illegal response type");
        }
        
        this.commandType = commandType;
        tasks = new ArrayList();
    }
    
    /**
     * Constructor for an error response. Takes an error message to be
     * relayed back to the client.
     * 
     * @param errorMessage
     * @throws IllegalArgumentException
     */
    public ServerPackage( String errorMessage ) throws IllegalArgumentException
    {
        if( errorMessage == null )
        {
            throw new IllegalArgumentException("Illegal error message");
        }
        this.errorMessage = errorMessage;
        this.commandType = ClientPackage.ERROR;
    }
    
    /**
     * Adds a new task into the list of tasks, but only if the created ServerPackage
     * is a response to a "fetch" client command.
     * 
     * @param task The task to be added.
     */
    public void addTask( Task task ) throws IllegalArgumentException
    {
        if( commandType != ClientPackage.FETCH )
        {
            throw new IllegalArgumentException("Only fetch-response can have tasks");
        }
        
        tasks.add( task );
    }
    
    /**
     * Returns a string that is in valid format for transport. This function parses
     * all of it's member variables into string form that can be read on the
     * client-side.
     * 
     * @return A string in the agreed format.
     */
    public String getMessage()
    {
        String message = "";
        
        switch( commandType )
        {
            // In this case the response is a simple "OK" 
            case ClientPackage.MARK_DONE:
            {
                message = "3#OK";
                break;
            }
            // In this case we return a string with the tag "Error: " at the
            // beginning, followed by a more detailed error message.
            case ClientPackage.ERROR:
            {
                message = "#Error: " + errorMessage;
                message = Integer.toString( message.length() ) + message;
                break;
            }
            // In this case the tasks must be "opened" and separators must be
            // added into right places.
            case ClientPackage.FETCH:
            {
                // If there are no tasks..
                if( tasks.size() == 0 )
                {
                    message = "1#";
                    break;
                }
                Iterator i = tasks.iterator();
                message = "#";
                
                // Tasks are parsed into format "#id#desc#id#desc#"
                while( i.hasNext() )
                {
                    Task t = (Task) i.next();
                    message += t.getId() + "#" + t.getDesc() + "#";
                }
                
                // size (in bytes) is added so that the final message will be:
                // "size|id|desc|id|desc|
                // the size is intentionally calculated with only the tasks and without
                // including the bytes size occupies.
                message = Integer.toString( message.length() ) + message;
                break;
            }
            default:
                break;
        }
        return message;
    }
}
