// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This client executable has all required capabilities to modify a file.
// Demonstrates all functionalities provided by the client side interface of the server.
//



/**
 @file
*/
#include "secureclientandserver.h"
#include "secureclient.h"

LOCAL_D CConsoleBase* console;

void DoExampleL()
	{
	console->Printf(KTxtTestingCountServer);
	
	// Create a session.
	RSecureSession countserv;
	User::LeaveIfError(countserv.Connect());
	
	// Create a sub session.
	RSecureSubSession counterA;
	counterA.Open(countserv);

	// The user can choose to initialise the counter from the counter.dat file.
	console->Printf(KTxtInitCounterAWith);
	console->Printf(KTxtOption);
	TUint8 inpOption = console->Getch();
	if(inpOption == '1')
		{
		// This call will fail if the client executable does not have required file read capability.
		counterA.SetCounterFromFileL();
		console->Printf(KTxtInitFromFile,counterA.CounterValueL());
		}
	else
		{
		// This is an invalid initialisation string. Hence, the counter will be initialised to zero.
		_LIT(KTxtValueString,"a2");
		console->Printf(KTxtValueString);
		// Set counter from the KTxtValueString string
		TInt err = counterA.SetFromString(KTxtValueString);
		if (err == ENonNumericString)
			{
			// Print an error message as the initialisation string, KTxtValueString is invalid (non numeric format).
			console->Printf(KTxtInitCounterFailed);
			}
		else
			{
			// Initialisation from string succeeded.
			console->Printf(KTxtInitCounterSucceeded);
			}
		}
	RSecureSubSession counterB;

	// Create a sub session.
	counterB.Open(countserv);

	// The user can choose to initialise the counter from the counter.dat file.
	console->Printf(KTxtInitCounterBWith);
	console->Printf(KTxtOption);
	inpOption = console->Getch();
	if(inpOption == '1')
		{
		// This call will fail if the client executable does not have required file read capability.
		counterB.SetCounterFromFileL();
		console->Printf(KTxtInitFromFile,counterB.CounterValueL());
		}
	else
		{
		// A valid numeric string.
		_LIT(KTxtLegalString,"100");
		console->Printf(KTxtLegalString);
		// Set counter from the KTxtLegalString string.
		TInt err = counterB.SetFromString(KTxtLegalString);
		if (err == ENonNumericString)
			{
			// Print an error message as the initialisation string, KTxtValueString is invalid (non numeric format).
			console->Printf(KTxtInitCounterFailed);
			}
		else
			{
			// Initialisation from string succeeded.
			// Counter is initialised to 100.
			console->Printf(KTxtInitCounterSucceeded);
			}
		}
	
	// Wait for a key press.
	console->Printf(KMsgPressAnyKey);
	console->Getch();
	console->ClearScreen();

	// Print values of both counters.
	console->Printf(KTxtGetCounterAValue,counterA.CounterValueL());
	console->Printf(KTxtGetCounterBValue,counterB.CounterValueL());

	// Wait for a key press.
	console->Printf(KMsgPressAnyKey);
	console->Getch();
	console->ClearScreen();

	// Increase the counter value by 1 and print it.
	_LIT(KTxt1,"\nIncrease counterA by default value (i.e. 1)..\n");
	console->Printf(KTxt1);
	counterA.Increase();
	console->Printf(KTxtGetCounterAValue,counterA.CounterValueL());

	// Increase the counter by a value greater than 1 and print it.
	_LIT(KTxt2,"\nIncrease counterA by 2..\n");
	console->Printf(KTxt2);
	// IncreaseByL operation undergoes a custom security check.
	// If the value is greater than 10 and the SID is greater than 0x70fffff5, it fails.
	counterA.IncreaseByL(2);
	console->Printf(KTxtGetCounterAValue,counterA.CounterValueL());

	// Increase the counter value by 1 and print it.
	_LIT(KTxt3,"\nIncrease counterB by default value (i.e. 1)..\n");
	console->Printf(KTxt3);
	counterB.Increase();
	console->Printf(KTxtGetCounterBValue,counterB.CounterValueL());

	// Increase the counter by a value greater than 1 and print it.
	_LIT(KTxt4,"\nIncrease counterA by 17..\n");
	console->Printf(KTxt4);
	// Fails if the SID is greater than 0x70fffff5
	counterA.IncreaseByL(17);
	console->Printf(KTxtGetCounterAValue,counterA.CounterValueL());

	// Wait for a key press.
	console->Printf(KMsgPressAnyKey);
	console->Getch();
	console->ClearScreen();

	// Increase the counter by a value greater than 1 and print it.
	_LIT(KTxt5,"\nIncrease counterB by 5..\n");
	console->Printf(KTxt5);
	counterB.IncreaseByL(5);
	console->Printf(KTxtGetCounterBValue,counterB.CounterValueL());

	// Decrease the counter value by 1 and print it.
	_LIT(KTxt6,"\nDecrease counterA..\n");
	console->Printf(KTxt6);
	counterA.Decrease();
	console->Printf(KTxtGetCounterAValue,counterA.CounterValueL());

	// Decrease the counter by a value greater than 1 and print it.
	_LIT(KTxt7,"\nDecrease counterB by 3..\n");
	console->Printf(KTxt7);
	console->Printf(KTxtGetCounterBValue,counterB.CounterValueL());
	// DecreaseByL operation undergoes a custom security check.
	// If the value is greater than 10 and the SID is greater than 0x70fffff5, it fails.
	counterB.DecreaseByL(3);

	// Print the resource count of the server.
	_LIT(KTxt8,"\nResource count is.. %d \n");
	console->Printf(KTxt8,countserv.ResourceCountL());

	// The user can save the counter value to a file.
	console->Printf(KTxtSaveToFile);
	TUint8 sel = console->Getch();
	if(sel == 'a')
		{
		// Save the value of the counter A.
		console->Printf(KTxtSelectCounter,sel);
		console->Printf(KTxtSaveCounter);
		// This call will fail if the client executable does not have required file write capability.
		counterA.SaveCounterL();
		}
	else if(sel == 'b')
		{
		// Save the value of the counter B.
		console->Printf(KTxtSelectCounter,sel);
		console->Printf(KTxtSaveCounter);
		// This call will fail if the client executable does not have required file write capability.
		counterB.SaveCounterL();
		}

	// Close both sub sessions and the main session.
	_LIT(KTxt9,"\nClosing counterA and then CounterB..\n");
	console->Printf(KTxt9);
	counterA.Close();
	counterB.Close();
	countserv.Close();
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();

	TRAPD(createError, console = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen)));
	if (createError)
		return createError;

	TRAPD(mainError, DoExampleL());
	if (mainError)
		console->Printf(KTextFailed, mainError);
	console->Printf(KTextPressAnyKey);
	console->Getch();

	delete console;
	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}
