// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Defines CSecureServer, CSecureServerSession and CSecureServerSubSession classes.
// It also defines the constants that form the security policy for the server.
//



/**
 @file
*/

#ifndef __SECURESERVER_H__
#define __SECURESERVER_H__

#include <e32svr.h>
#include <e32uid.h>
#include <e32math.h>
#include <f32file.h>

/**
 The size of the heap to be allocated for the thread that runs the server. The constant is passed to the StartThread() function defined in secureclientandserver.h and implemented in secureserver.cpp.
@see RThread::Create()
*/
const TUint KDefaultHeapSize=0x10000;

/**
A set of panic numbers used when the example raises panics in response to programming errors. All panics raised by the example have the category SecureServer.
@see KTxtServer
*/

enum TSecureServerPanic
	{
	/**
	An undefined op code.
	*/
	EBadRequest = 1,
	/**
	A descriptor containing non-numeric characters.
	*/
	EDescriptorNonNumeric,
	/**
	The Active Scheduler can not be instantiated.
	*/
	EMainSchedulerError,
	/**
	The server can not be created.
	*/
	EServerCreateServer,
	/**
	The server can not be added to the active scheduler to receive requests.
	*/
	EServerStartServer,
	/**
	The cleanup stack can not be created.
	*/
	ECreateTrapCleanup,
	/**
	The object index of the sub session returns an error.
	*/
	EBadSubsessionHandle,
	/**
	A custom security policy violation.
	*/
	ESecureServerSecurityPolicy
	};

/**
The server specifies 5 ranges of security policy. Each range undergoes a different security check.
@see CpolicyServer::TPolicy::iRangeCount
*/
const TInt KSecureServerRangeCount = 5;

/**
Ordered ranges of request numbers.
@see CpolicyServer::TPolicy::iRanges
*/
const TInt KSecureServerRanges[KSecureServerRangeCount] =
	{
	/**
	Security Policy for op codes 0-4 (ESecureServerCreateSubSession to ESecureServerValue).
	*/
	0,
	/**
	Security Policy for op code 5 (ESecureServerResourceCount).
	*/
	5,
	/**
	Security Policy for op codes 6-10 (ESecureServerIncrease to ESecureServerReset).
	*/
	6,
	/**
	Security Policy for op codes 11-12 (ESecureServerSaveCounter to ESecureServerSetCounterFromFile).
	*/
	11,
	/**
	Security Policy for op codes 13 and above.
	*/
	13
	};

/**
An array containing the unique set of policies used in this example.
@see CpolicyServer::TPolicy::iElements
*/
const CPolicyServer::TPolicyElement KSecureServerElements[] =
	{
	/**
	Checks if the SID of the client is equal to 0x70fffff0.
	*/
	_INIT_SECURITY_POLICY_S0(0x70fffff0),
	/**
	Checks whether the client has ReadDeviceData and WriteDeviceData capabilities.
	*/
	{_INIT_SECURITY_POLICY_C1(ECapabilityReadDeviceData | ECapabilityWriteDeviceData), CPolicyServer::EFailClient}
	};

/**
An array of TUint8 values that specify the action to take for each range in KSecureServerRanges array.
@see CpolicyServer::TPolicy::iElementsIndex
*/
const TUint8 KSecureServerElementsIndex[KSecureServerRangeCount] =
		{
		/**
		Maps the 0th range of KSecureServerRanges(0-4) to EAlwaysPass.
		These requests do not undergo a security check.
		*/
		CPolicyServer::EAlwaysPass,
		/**
		Maps the 1st range of KSecureServerRanges(5-7) to 0th element of KSecureServerElements.
		SID is checked for these requests.
		*/
		0,
		/**
		Maps the 2nd range of KSecureServerRanges(8-12) to ECustonCheck.
		These request are passed to the CustomSecurityCheckL method implemented in the example.
		*/
		CPolicyServer::ECustomCheck,
		/**
		Maps the 3rd range of KSecureServerRanges(13 and 14) to 1st element of KSecureServerElements.
		ReadDeviceData and WriteDeviceData capabilities are checked for these requests.
		*/
		1,
		/**
		Maps the 4th range of KSecureServerRanges(16 and above) to ENotSupported.
		Returns KErrNotSupported for requests with op code >= 15.
		*/
		CPolicyServer::ENotSupported
		};

/**
The security policy of the server.
This is passed to the constructor of the policy server. In this example, this is the constructor of the CSecureServer class.
@see CSecureServer
*/
const CPolicyServer::TPolicy KSecureServerPolicy =
	{
	/**
	Specifies all connect attempts should pass.
	*/
	CPolicyServer::EAlwaysPass,
	KSecureServerRangeCount,
	KSecureServerRanges,
	KSecureServerElementsIndex,
	KSecureServerElements
	};

/**
The class that represents the secure server.

The server starts when the first client connects.
When the server starts, it prepares:
- the active scheduler
- the server active object
- the object container index which produces object containers for each session.

@see CSecureServer
*/
class CSecureServer : public CPolicyServer
	{
public:
	CSession2* NewSessionL(const TVersion &aVersion,const RMessage2& aMessage) const;
	static CSecureServer* NewL(CActive::TPriority aActiveObjectPriority);
	static TInt ThreadFunction(TAny* aStarted);
	static void PanicServer(TSecureServerPanic aPanic);
	CPolicyServer::TCustomResult CustomSecurityCheckL(const RMessage2 &aMsg, TInt &aAction, TSecurityInfo &aMissing);
	CObjectCon* NewContainerL();
	void RemoveContainer(CObjectCon *aCon);
	~CSecureServer();
private :
	CSecureServer(CActive::TPriority aActiveObjectPriority);
	void ConstructL();

private:
	/**
	The object container index that creates an object container for each session.
	*/
	CObjectConIx* iContainerIndex;
	};

class CSecureServerSubSession;

/**
The class that represents a session with the server.
When a client connects to the server, a session object is created by the server.
The class provides functions that respond to client messages.
A session can own any number of subsession objects.
*/
class CSecureServerSession : public CSession2
	{
public:
	static CSecureServerSession* NewL();
	void CreateL();
	void ServiceL(const RMessage2& aMessage);
	void NewCounterL(const RMessage2& aMessage);
	void CloseSession();
	void NumResourcesL(const RMessage2& aMessage);
	CSecureServerSubSession* CounterFromHandle(const RMessage2& aMessage,TInt aHandle);
	void DeleteCounter(TInt aHandle);
	TInt CountResources();
	void PanicClient(const RMessage2& aMessage,TInt aPanic) const;
	CSecureServerSession();
private:
	/**
	Object container for this session.
	*/
	CObjectCon* iContainer;
	/**
	Object index which stores objects (CCountSubSession instances) for this session.
	*/
	CObjectIx* iCountersObjectIndex;
	/**
	Total number of resources.
	A resource count is the number of CSecureServerSubSession objects.
	*/
	TInt iResourceCount;
	};

/**
The class that represents a subsession.
When a client wants more than one connection to a server at the same time, it can use a subsession. A subsession is more efficient than a session, but an application needs to provide more code.
A subsession object is created by a session. A session can own any number of subsession objects.
The class provides functions that respond to client messages.
*/
class CSecureServerSubSession : public CObject
	{
public:
	static CSecureServerSubSession* NewL(CSecureServerSession* aSession);
	CSecureServerSubSession(CSecureServerSession* aSession);
	void ConstructL(CSecureServerSession* aSession);
	void SetFromStringL(const RMessage2& aMessage);
	void Increase();
	void IncreaseBy(const RMessage2& aMessage);
	void Decrease();
	void DecreaseBy(const RMessage2& aMessage);
	void Reset();
	void CounterValueL(const RMessage2& aMessage);
	void SaveCounterValueL();
	void SetCounterValueFromFileL();

protected:
	/**
	The parent session.
	*/
	CSecureServerSession* iSession;

private:
	/**
	The counter value.
	*/
	TInt iCount;
	};

#endif
