// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Demonstrates the speculative pattern of subscribing to a byte-array property
//



/**
 @file
*/

#include "subscribe.h"
LOCAL_D CConsoleBase* console;

/**
Constructor
*/
CArrayPropertyWatch::CArrayPropertyWatch():CActive(EPriority)
	{
	}

/**
Adds the active object to the active scheduler and creates a handle to the property
@param aConsole The console interface
*/
void CArrayPropertyWatch::ConstructL(CConsoleBase* aConsole)
	{
	User::LeaveIfError(iProperty.Attach(KMyPropertyCat,KMyPropertyName));
	iConsole = aConsole;
	CActiveScheduler::Add(this);
	}

/**
Constructs a CArrayPropertyWatch object
@param aConsole The console interface
@return A CArrayPropertyWatch object
*/
CArrayPropertyWatch* CArrayPropertyWatch::NewL(CConsoleBase* aConsole)
	{
	CArrayPropertyWatch* self = new (ELeave) CArrayPropertyWatch;
	CleanupStack::PushL(self);
	self->ConstructL(aConsole);
	CleanupStack::Pop(self);
	return self;
	}

/**
Destructor
*/
CArrayPropertyWatch::~CArrayPropertyWatch()
	{
	iProperty.Close();
	// Delete the property as it is no longer required by the subscriber
	RProperty::Delete(KMyPropertyCat, KMyPropertyName);
	Cancel();
	}

/**
Cancels any outstanding request
*/
void CArrayPropertyWatch::DoCancel()
	{
	iProperty.Cancel();
	}

/**
Handles the subscription to the property
Stops the subscription if the string 'STOP' is received
*/
void CArrayPropertyWatch::RunL()
	{
	// Get the value of the property
	TBuf16<KArraySize> buf;
	TInt err = iProperty.Get(buf);
	if(err == KErrNotFound)
		{
		// Leave the function if the property is not defined
		CActiveScheduler::Stop();
		User::Leave(err);
		}
	else
		{
		// Print the value of the property
		PrintProperty(buf);
		if(buf == KStop)
			{
			CActiveScheduler::Stop();
			}
		else
			{
			// Re-subscribe to the property
			iProperty.Subscribe(iStatus);
			SetActive();
			}
		}
	}

/**
Prints the value of byte-array property
@param aBuf The descriptor containing the changed byte-array property
*/
void CArrayPropertyWatch::PrintProperty(TDes16& aBuf)
	{
	iConsole->Printf(KTxtArray);
	TInt bufLength = aBuf.Length();
	for(TInt ix = 0; ix < bufLength;ix++)
		{
		iConsole->Printf(KTxtArrayElement,aBuf[ix]);
		}
	iConsole->Printf(KTxtNewLine);
	}

/**
Defines and subscribes to the property
*/
void CArrayPropertyWatch::DefinePropertyL()
	{
	iConsole->Printf(KTxtDefine);
	// Define the property
	TInt err = RProperty::Define(KMyPropertyCat, KMyPropertyName,RProperty::EByteArray,KAllowAllPolicy,KAllowAllPolicy,KArraySize);
	if(err == KErrAlreadyExists || err == KErrNone)
		{
		// Ignore the KErrAlreadyExists error code
		TBuf16<KArraySize> buf;
		// Returns the default value of the property
		// Ignore this value
		err = iProperty.Get(buf);
		User::LeaveIfError(err);
		// Subscribe to the property and maintain an outstanding request
		iProperty.Subscribe(iStatus);
		SetActive();
		}
	else
		{
		// Leave in case of errors other than KErrAlreadyExists
		User::Leave(err);
		}
	}

void DoExampleL()
	{
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);

	console->Printf(KTxtSpecSub);

	// Create the byte-array property watch active object
	CArrayPropertyWatch* obj = CArrayPropertyWatch::NewL(console);
	CleanupStack::PushL(obj);

	// Subscribe to the property and start the scheduler
	obj->DefinePropertyL();
	CActiveScheduler::Start();

	CleanupStack::PopAndDestroy(obj);
	CleanupStack::PopAndDestroy(scheduler);
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();

	TRAPD(createError, console = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen)));
	if (createError)
		return createError;

	TRAPD(mainError, DoExampleL());
	if (mainError)
		console->Printf(KTextFailed, mainError);
	console->Printf(KTextPressAnyKey);
	console->Getch();

	delete console;
	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}
