// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Demonstrates the standard state pattern of subscribing to a byte-array property
//



/**
 @file
*/

#include "subscribe.h"
LOCAL_D CConsoleBase* console;

/**
Constructor
*/
CArrayPropertyWatch::CArrayPropertyWatch():CActive(EPriority)
	{
	}

/**
Adds the active object to the active scheduler and creates a handle to the byte-array property
@param aConsole The console interface
*/
void CArrayPropertyWatch::ConstructL(CConsoleBase* aConsole)
	{
	User::LeaveIfError(iProperty.Attach(KMyPropertyCat,KMyPropertyName));
	iConsole = aConsole;
	CActiveScheduler::Add(this);
	}

/**
Constructs a CArrayPropertyWatch object
@param aConsole The console interface
@return A CArrayPropertyWatch object
*/
CArrayPropertyWatch* CArrayPropertyWatch::NewL(CConsoleBase* aConsole)
	{
	CArrayPropertyWatch* self = new (ELeave) CArrayPropertyWatch;
	CleanupStack::PushL(self);
	self->ConstructL(aConsole);
	CleanupStack::Pop(self);
	return self;
	}

/**
Destructor
*/
CArrayPropertyWatch::~CArrayPropertyWatch()
	{
	iProperty.Close();
	Cancel();
	}

/**
Cancels any outstanding request
*/
void CArrayPropertyWatch::DoCancel()
	{
	iProperty.Cancel();
	}

/**
Handles the subscription to the property
Stops the subscription if the string 'STOP' is received
*/
void CArrayPropertyWatch::RunL()
	{
	// Get the value of the property
	TBuf16<KArraySize> buf;
	TInt err = iProperty.Get(buf);
	if(err == KErrNotFound)
		{
		// Leave the function if the property is not defined
		CActiveScheduler::Stop();
		User::Leave(err);
		}
	else
		{
		if(buf == KStop)
			{
			CActiveScheduler::Stop();
			}
		else
			{
			// Re-subscribe to the property
			WatchL();
			}
		}
	}

/**
Prints the value of the byte-array property
@param aBuf Descriptor containing the byte-array property
*/
void CArrayPropertyWatch::PrintProperty(TDes16& aBuf)
	{
	iConsole->Printf(KTxtArray);
	TInt bufLength = aBuf.Length();
	for(TInt ix = 0; ix < bufLength;ix++)
		{
		iConsole->Printf(KTxtArrayElement,aBuf[ix]);
		}
	iConsole->Printf(KTxtNewLine);
	}

/**
Issues an outstanding request by subscribing to the property
*/
void CArrayPropertyWatch::WatchL()
	{
	// Get the value of the property
	TBuf16<KArraySize> buf;
	TInt err = iProperty.Get(buf);
	if(err == KErrNotFound)
		{
		// Leave if the property is not found
		User::Leave(err);
		}
	PrintProperty(buf);
	// Subscribe to the property
	iProperty.Subscribe(iStatus);
	// Issue an outstanding request
	SetActive();
	}

void DoExampleL()
	{
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);

	console->Printf(KTxtSpecSt);

	// Create the byte-array property watch active object
	CArrayPropertyWatch* obj = CArrayPropertyWatch::NewL(console);
	CleanupStack::PushL(obj);

	// Subscribe to the property and start the scheduler
	obj->WatchL();
	CActiveScheduler::Start();

	CleanupStack::PopAndDestroy(obj);
	CleanupStack::PopAndDestroy(scheduler);
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();

	TRAPD(createError, console = Console::NewL(KTextConsoleTitle, TSize(KConsFullScreen,KConsFullScreen)));
	if (createError)
		return createError;

	TRAPD(mainError, DoExampleL());
	if (mainError)
		console->Printf(KTextFailed, mainError);
	console->Printf(KTextPressAnyKey);
	console->Getch();

	delete console;
	delete cleanup;
	__UHEAP_MARKEND;
	return KErrNone;
	}
