/*
 * Copyright  2008 Nokia Corporation.
 */

#include "DescriptorExAppUi.h"
#include "DescriptorExContainer.h"
#include <DescriptorEx.rsg>
#include "descriptorex.hrh"
#include <aknutils.h>

#include <avkon.hrh>
#include "DescriptorExamples.h"

// -----------------------------------------------------------------------------
// defines a type that is pointer to a example method in CDescriptorExamples
// -----------------------------------------------------------------------------
typedef void (CDescriptorExamples::*TExampleMethodPtr)();

// -----------------------------------------------------------------------------
// defines an array of pointers pointing to example methods
// -----------------------------------------------------------------------------
static const TExampleMethodPtr ExampleMethods[] =
    {
    &CDescriptorExamples::ToStack,
    &CDescriptorExamples::ToHeapL,
    &CDescriptorExamples::Literals,

    &CDescriptorExamples::NonModifyingMethods,
    &CDescriptorExamples::ModifyingMethodsL,
    &CDescriptorExamples::CharacterConversionsL,
    &CDescriptorExamples::LexicalAnalysis,

    &CDescriptorExamples::CircularBuffersL,
    &CDescriptorExamples::FlatDynamicBuffersL,
    &CDescriptorExamples::SegmentedDynamicBuffersL,
    &CDescriptorExamples::PackageBuffers,
    &CDescriptorExamples::RBufDemonstrations
    };
static const TInt KNumberOfExampleMethods = sizeof(ExampleMethods) / 
                                            sizeof(TExampleMethodPtr);

// -----------------------------------------------------------------------------
// Executes requested method. If aNumber indexes out of bounds of method pointer
// array, nothing is done.
// -----------------------------------------------------------------------------
LOCAL_C void ExecuteExampleMethodL(TInt aNumber, CDescriptorExamples &aExample)
    {
    if( aNumber >= 0 && aNumber < KNumberOfExampleMethods )
        {
        TExampleMethodPtr mPtr = ExampleMethods[aNumber];
        (aExample.*mPtr)(); // call example method through method pointer
        }
    }

// -----------------------------------------------------------------------------
// Second phase constructor of class CDescriptorExAppUi.
// -----------------------------------------------------------------------------
void CDescriptorExAppUi::ConstructL()
    {
    BaseConstructL(EAknEnableSkin);

    iAppContainer = new (ELeave) CDescriptorExContainer;
    iAppContainer->SetMopParent(this);
    iAppContainer->ConstructL( ClientRect() );
    AddToStackL( iAppContainer );
    iBuffer = HBufC::NewL(4096);
    iExamples = new (ELeave) CDescriptorExamples( this );
    }

// -----------------------------------------------------------------------------
// This destructor frees reserved resources
// -----------------------------------------------------------------------------
CDescriptorExAppUi::~CDescriptorExAppUi()
    {
    delete iExamples;
    iExamples = NULL;
    delete iBuffer;
    iBuffer = NULL;
    if (iAppContainer)
        {
        RemoveFromStack( iAppContainer );
        delete iAppContainer;
        iAppContainer = NULL;
        }
   }

// -----------------------------------------------------------------------------
// Run all example methods and write results to the file.
// -----------------------------------------------------------------------------
void CDescriptorExAppUi::RunAllL()
    {
    _LIT( KFileName, "c:\\Data\\DescriptorEx.log.txt" );

    CFileForwarder *fileWriter = new (ELeave) CFileForwarder();
    CleanupStack::PushL( fileWriter );
    fileWriter->ConstructL(KFileName);

    CDescriptorExamples *example = new (ELeave) CDescriptorExamples(fileWriter);
    CleanupStack::PushL( example );

    for( TInt i=0; i < KNumberOfExampleMethods; i++ )
        {
        ExecuteExampleMethodL(i, *example);
        }
    GetViewBuffer().AppendFormat( _L(
        "Results of example methods written to file\n\"%S\"\n"), &KFileName );
    UpdateView();

    CleanupStack::PopAndDestroy(2); // examples, fileWriter
    }

// -----------------------------------------------------------------------------
// Handles commands from framework. Expecting to receive only menu commands and
// selecting back from right menu (EAknSoftkeyBack).
// -----------------------------------------------------------------------------
void CDescriptorExAppUi::HandleCommandL(TInt aCommand)
    {
    if( aCommand - TDescriptorExCommandFirstID < KNumberOfExampleMethods )
        {
        // reset internal buffer before example
        iBuffer->Des().SetLength(0);

        // call one of the example methods. Since commands start from
        // TDescriptorExCommandFirstID, it has to be decreased from aCommand
        // to get correct example method number (since indexing to array
        // starts from zero).
        ExecuteExampleMethodL( aCommand - TDescriptorExCommandFirstID,
                              *iExamples );
        }

    switch ( aCommand )
        {
        case EAknSoftkeyExit:
        case EEikCmdExit:
            {
            Exit();
            break;
            }
        case ECmd_RunAll:
            {
            //The RBuf demonstration (RBufDemonstrations) consists of code 
            //examples in manipulating RBuf objects and doesn't write any 
            //results to the file so it isn't executed here.
            //iExamples->RBufDemonstrations();
            RunAllL();
            break;
            }
        default:
            {
            break;      
            }
        };
    }

// -----------------------------------------------------------------------------
// Handle key events
// -----------------------------------------------------------------------------
TKeyResponse CDescriptorExAppUi::HandleKeyEventL(const TKeyEvent& aKeyEvent,
                                                 TEventCode aType)
{
    // forward key event to container
    return iAppContainer->OfferKeyEventL( aKeyEvent, aType );
}

// -----------------------------------------------------------------------------
// Return modifiable descriptor pointer to our internal buffer so that examples
// can write their results to it.
// -----------------------------------------------------------------------------
TPtr CDescriptorExAppUi::GetViewBuffer()
{
    return iBuffer->Des();
}

// -----------------------------------------------------------------------------
// Update our edvin control with data in our internal buffer. If unicode
// build, replace any ascii '\n' character with unicode variant since edwin
// understands only unicode paragraphs when unicode build.
// -----------------------------------------------------------------------------
void CDescriptorExAppUi::UpdateView()
    {
    // replace any '\n' characters from buffer with UCS paragraph separator if
    // unicode build.
#ifdef _UNICODE
    TText16* ptr=(TText16*)iBuffer->Ptr();
    TText16* endPtr=ptr + iBuffer->Length();
    for(;ptr < endPtr; ptr++ )
    {
        if( *ptr == '\n' )
        {
            *ptr = 0x2029; // UCS_PARASEP = 0x2029
        }
    }
#endif
    // MResultViewer declares this method as non leaving (no suffix L). If
    // text editor can't set the text, we ignore the error.
    TRAPD(ignored, iAppContainer->SetTextL(*iBuffer));
    }

void CDescriptorExAppUi::HandleResourceChangeL(TInt aType)
    {
    CAknAppUi::HandleResourceChangeL(aType);

    if ( aType == KEikDynamicLayoutVariantSwitch )
        {
        TRect rect;
        AknLayoutUtils::LayoutMetricsRect(AknLayoutUtils::EMainPane, rect);
        iAppContainer->SetRect(rect);
        }
    }
