/*
 * Copyright  2008 Nokia Corporation.
 */

#include <e32std.h>
#include <e32base.h>
#include <charconv.h>
#include <collate.h>
#include "DescriptorExamples.h"
#include "StringRenderer.h"

// -----------------------------------------------------------------------------
// These macros are shorter versions for Rendering macros. They assume variable
// 'output' derived from TDes to be in the scope where used.
// -----------------------------------------------------------------------------
#define RenderVariableString(aVariableName) \
    RenderVariableFormatted(aVariableName, output, KRenderDefault);

#define RenderVariableBinary(aVariableName) \
    RenderVariableFormatted(aVariableName, output, KRenderContentAsBinary);

#define RenderResult(aDeclaration) \
    RenderResultFormatted(aDeclaration, output, KRenderDefault);

#define ExecuteAndRenderVariable(aDeclaration, aVariable) \
    ExecuteAndRenderVariableFormatted(aDeclaration, aVariable, output, \
                                      KRenderDefault );

// -----------------------------------------------------------------------------
// This example method is documented in header file "DescriptorExamples.h"
// -----------------------------------------------------------------------------
void CDescriptorExamples::NonModifyingMethods()
    {
    TPtr output( iViewer->GetViewBuffer() );
    RenderHeader( _L( "NonModifyingMethods" ), output );

    // ---------- Initialize an example string ----------
    // this declares a literal with a few characters
    _LIT(KCharacters, "abcdefghijklmnopqrstuvwxyz0123456789");
    // this casts the literal to non modifiable descriptor
    const TDesC &str = KCharacters();
    RenderVariableString(str);

    // ---------- Capacity ----------
    output.Append( _L( "\nCapacity...\n" ) );

    // Length() returns number of characters in descriptor
    RenderResult( str.Length() ); // 36

    // Size() returns number of bytes in descriptor - in unicode buid it is
    //        2*Length() and in non-unicode build it is same as Length()
    RenderResult( str.Size() ); // 72

    // ---------- Extracting portions ----------
    output.Append( _L( "\nExtracting portions...\n" ) );

    // Left(count) returns a string witn "count" number of characters
    // calculated from left side of the string
    RenderResult( str.Left(2) ); // "ab"
    RenderResult( str.Left(5) ); // "abcde"

    // Right(count) returns a string witn "count" number of characters
    // calculated from right side of the string
    RenderResult( str.Right(2) ); // "89"
    RenderResult( str.Right(12) ); // "yz0123456789"

    // Mid(index, count) returns a portion from string starting
    // from "index" and having "count" number of characters
    RenderResult( str.Mid(2,6) ); // "cdefgh"
    RenderResult( str.Mid(33,3) ); // "789"

    // if "count" is omitted the rest of the string from "index" is returned
    RenderResult( str.Mid(28) ); // "23456789"
    RenderResult( str.Mid(35) ); // "9"

    // ---------- Locating character ----------
    output.Append( _L( "\nLocating character...\n" ) );

    // locate index of given character
    RenderResult( str.Locate('d') ); // 3
    RenderResultFormatted( _L("1ABAD").Locate('A'),
        output, KRenderDefault); // "1"
    RenderResultFormatted( _L("1ABAD").LocateReverse('A'),
        output, KRenderDefault); // "1"

    // ---------- Compare ----------
    // Comparison with method Compare() uses the binary comparison approach to
    // compare each character in string. For example value of character
    // 'A' has unicode value of 65 and 'a' 97.
    //
    // Both strings could have been converted to lower or upper case to perform
    // case insensitive comparison. However, there exists also CompareC() method
    // below that introduces a method to compare in case insensitive manner.
    output.Append( _L( "\nCompare()\n" ) );

    // examplerun: str.Compare(_L("aaa"))=1
    RenderResultFormatted( str.Compare( _L("aaa") ),
        output, KRenderDefault); // >0 : str is greater

    // examplerun: str.Compare(_L("zzz"))=-25
    RenderResultFormatted( str.Compare( _L("zzz") ),
        output, KRenderDefault ); // <0 : str is less

    // examplerun: str.Compare(str)=0
    RenderResult( str.Compare(str) );       // 0  : str is equal

    // examplerun: str.Left(2).Compare(_L("ab"))=0
    RenderResultFormatted( str.Left(2).Compare(_L("ab")),
        output, KRenderDefault); // 0 : equal

    // examplerun: str.Left(2).Compare(_L("AB"))=32
    // 'A' (65) > 'a' (97)
    RenderResultFormatted( str.Left(2).Compare(_L("AB")),
        output, KRenderDefault ); // 0 : equal

    // ---------- Comparing (Collated) ----------
    // To perform collated comparisons, method
    //
    //   CompareC(TDesC&, TInt aMaxLevel, const TCollationMethod*)
    //
    // is used. The aMaxLevel parameter can be used to perform specific rules:
    //
    // 0 - only test the character identity; accents and case are ignored
    // 1 - test the character identity and accents; case is ignored
    // 2 - test the character identity, accents and case
    // 3 - test the Unicode value as well as the character identity,
    //     accents and case.
    //
    output.Append( _L( "\nCompareC()...\n" ) );

    // To perform case insensitive comparison, level 1 is used
    RenderResultFormatted( _L("ab").CompareC(_L("AB"),1,NULL),
        output, KRenderDefault ); // 0 : equal

    // note that when using collated comparison as case sensitive manner
    // the string "ab" is less than "AB" while using standars Compare()
    // the "ab" is greater than "AB"!!!
    RenderResultFormatted( _L("ab").CompareC(_L("AB"),2,NULL),
        output, KRenderDefault );
    RenderResultFormatted( _L("ab").Compare(_L("AB")),
        output, KRenderDefault );

    // ----------
    // The natural comparison ignores white space differences.
    // This and other specific behaviour can be changed by modifying
    // the standard behaviour.
    TCollationMethod stdMethod = *Mem::CollationMethodByIndex(0); // std impl

    // dont ignore punctuation and spaces
    TCollationMethod strictMethod = stdMethod; // copy std impl
    strictMethod.iFlags |= TCollationMethod::EIgnoreNone; // modify copy

    // use level 1 (case ignored) in both cases to compare. Then use
    // standard method and modified one. Examplerun:
    //
    //    _L("a  b").CompareC(_L("A B"),1,&stdMethod)=0
    //    _L("a  b").CompareC(_L("A B"),1,&strictMethod)=-2
    //
    RenderResultFormatted( _L("a   b").CompareC(_L("A B"),1,&stdMethod),
        output, KRenderDefault );
    RenderResultFormatted( _L("a   b").CompareC(_L("A B"),1,&strictMethod),
        output, KRenderDefault );

    iViewer->UpdateView();
    }

// -----------------------------------------------------------------------------
// This example method is documented in header file "DescriptorExamples.h"
// -----------------------------------------------------------------------------
void CDescriptorExamples::ModifyingMethodsL()
    {
    // this is used to point to correct position in view buffer
    TPtr output( iViewer->GetViewBuffer() );
    RenderHeader( _L( "ModifyingMethods" ), output );
    TBuf<10> buf10;

    // ---------- Setting and viewing characteristics ----------
    output.Append( _L("\nCharacteristics of buffer after initialization:\n") );
    RenderResult( buf10.MaxLength() ); // max number of chars
    RenderResult( buf10.MaxSize() ); // max number of bytes
    RenderResult( buf10.Length() ); // number of chars
    RenderResult( buf10.Size() ); // number of bytes

    TBuf<20> example(_L("Hello"));
    RenderVariableFormatted( example, output, KRenderCharacteristics );

    // first init the data in buffer with asterix
    output.Append( _L("\nLet's fill the buffer\n") );
    buf10.Fill( '*', buf10.MaxLength() );
    RenderVariableString( buf10 ); // max number of bytes
    RenderResult( buf10.Length() ); // number of chars
    RenderResult( buf10.Size() ); // number of bytes

    output.Append( _L("\nNow manipilate the length:\n") );
    // set the current length to three. The data that exist in the
    // buffer is as is
    ExecuteAndRenderVariable( buf10.SetLength(3), buf10 );
    ExecuteAndRenderVariable( buf10.SetMax(), buf10 );
    ExecuteAndRenderVariable( buf10.Zero(), buf10 );

    // ---------- Copying ----------
    output.Append( _L("\nCopying...\n") );
    TPtrC8 narrowHello( _L8( "hello" ) );
    RenderVariableString( narrowHello );
    TPtrC16 wideHello( _L16( "unIc heLLo" ) );
    RenderVariableString( wideHello );
    // copy contents of 8 bit descriptor to 16 bit descriptor
    ExecuteAndRenderVariable( buf10.Copy(narrowHello), buf10 );
    // copy contents of 16 bit descriptor to 16 bit descriptor
    ExecuteAndRenderVariable( buf10.Copy(wideHello), buf10 );
    // copy contents of 16 bit descriptor to 16 bit descriptor
    // and capitalize the content
    ExecuteAndRenderVariable( buf10.CopyCP(wideHello), buf10 );
    // copy and set to lower case
    ExecuteAndRenderVariable( buf10.CopyLC(wideHello), buf10 );
    // copy and set to Uppper case
    ExecuteAndRenderVariable( buf10.CopyUC(wideHello), buf10 );

    // ---------- Repeating ----------
    output.Append( _L("\nRepeating...\n") );
    TPtrC abc( _L( "aBc" ) );
    RenderVariableString( abc );
    // write abs twise to the descriptor
    ExecuteAndRenderVariable( buf10.Repeat((TText*)abc.Ptr(),2), buf10 );
    // write abs until to the maximum size
    ExecuteAndRenderVariable( buf10.Repeat(abc), buf10 );

    // ---------- Justifying ----------
    output.Append( _L("\nJustifying...\n") );
    RenderVariableString( abc );
    // show aligning of string "abs" to our buffer
    ExecuteAndRenderVariable( buf10.Justify(abc,8,ELeft,'_'), buf10 );
    ExecuteAndRenderVariable( buf10.Justify(abc,8,ECenter,'*'), buf10 );
    ExecuteAndRenderVariable( buf10.Justify(abc,10,ERight,'.'), buf10 );

    // ---------- Rendering numbers ----------
    output.Append( _L("\nRendering numbers...\n") );
    // decimal number
    ExecuteAndRenderVariable( buf10.Num(12345), buf10 );
    // hexadecimal number. characters in lover case
    ExecuteAndRenderVariable( buf10.Num(65300,EHex), buf10 );
    // binary number
    ExecuteAndRenderVariable( buf10.Num(55,EBinary), buf10 );
    // hexadecimal number. characters in upper case
    ExecuteAndRenderVariable( buf10.NumUC(65300,EHex), buf10 );
    // fixed width decimal number
    ExecuteAndRenderVariable( buf10.NumFixedWidth(25,EDecimal,6), buf10 );
    // fixed width hexadecimal number
    ExecuteAndRenderVariable( buf10.NumFixedWidth(0xf05,EHex,6), buf10 );
    // fixed width binary number
    ExecuteAndRenderVariable( buf10.NumFixedWidth(3,EBinary,8), buf10 );
    // fixed width hexadecimal number, upper case
    ExecuteAndRenderVariable( buf10.NumFixedWidthUC(0xf05,EHex,6), buf10 );

    // declare 64 bit number that has now only lower 32 bits set
    TInt64 longNum(0x12345678);
    ExecuteAndRenderVariable( buf10.Num(longNum,EHex), buf10 );
    // multiply the result so that it needs more than 32 bits and show results.
    // The result 123456780 > FFFFFFFF that is the maximum size 32 bit
    // integer can store
    ExecuteAndRenderVariable( buf10.Num(longNum*16,EHex), buf10 );

    TRealFormat realFormatter(8);
    ExecuteAndRenderVariable( buf10.Num(3.5,realFormatter), buf10 );
    ExecuteAndRenderVariable( buf10.Num(10.0/3,realFormatter), buf10 );
    ExecuteAndRenderVariable( buf10.Num(123.0*1000*1000*1000,realFormatter),
                              buf10 );

    // ---------- Formatting ----------
    output.Append( _L("\nFormatting data types\n") );
    TBuf<64> buf;
    RenderVariableFormatted( buf, output, KRenderCharacteristics );

    // binary format using %b
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("binary=%b" ), 33),
        buf,output, KRenderDefault );

    // integer format using %d
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("decimal=%d" ), 33),
        buf,output, KRenderDefault );

    // hexadecimal format using %x
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("hexadecimal=%x" ), 33),
        buf,output, KRenderDefault );

    // real as exponent format
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("real (exp)=%e" ), 33.43),
        buf,output, KRenderDefault );

    // real as fixed format
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("real (fixed)=%f" ), 33.43),
        buf,output, KRenderDefault );

    // pointer to descriptor
    TPtrC str = _L("hello");
    RenderVariableString( str );
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%S'" ), &str),
        buf,output, KRenderDefault );

    // pointer to null terminated character array
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%s'" ), str.Ptr()),
        buf,output, KRenderDefault );

    // ---------- Formatting align and padding ----------
    output.Append( _L("\nFormatting align and padding\n") );

    // padding width: 10, default pad character, default alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("binary=%10b" ), 33),
        buf,output, KRenderDefault );

    // padding width: 10, '0' as pad character, default alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("binary=%010b" ), 33),
        buf,output, KRenderDefault );

    // padding width: 8, '0' as pad character, default alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("binary=%08d" ), 12345),
        buf,output, KRenderDefault );

    RenderVariableString( str );
    // padding width: 20, default pad character, default alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%20S'" ), &str),
        buf,output, KRenderDefault );

    // padding width: 20, '0' as pad character, default alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%020S'" ), &str),
        buf,output, KRenderDefault );

    // padding width: 20, default (space) as pad character, center alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%=20S'" ), &str),
        buf,output, KRenderDefault );

    // padding width: 20, default (space) as pad character, left alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%-20S'" ), &str),
        buf,output, KRenderDefault );

    // padding width: 20, default (space) as pad character, right alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%+20S'" ), &str),
        buf,output, KRenderDefault );

    // padding width: 20, custom pad character, center alignment
    ExecuteAndRenderVariableFormatted(
        buf.Format( _L("str='%=*20S'" ), '.', &str),
        buf,output, KRenderDefault );

    // ---------- Modifying content in other means ----------
    output.Append( _L("\nOther modifications...\n") );
    buf.Copy(_L("abcd"));
    RenderVariableString( buf );

    // insert a string to middle of buffer
    ExecuteAndRenderVariableFormatted(
        buf.Insert( 2, _L("____") ),
        buf, output, KRenderDefault );

    // replaces a portion with given string
    ExecuteAndRenderVariableFormatted(
        buf.Replace( 3, 2, _L("****") ),
        buf, output, KRenderDefault );

    // replaces but since portion size is 0, this is same
    // as insert( 1, _L("AA") );
    ExecuteAndRenderVariableFormatted(
        buf.Replace( 1, 0, _L("AA") ),
        buf, output, KRenderDefault );

    // replaces a portion but since no data empty, this is same
    // as delete( 1, 9 );
    ExecuteAndRenderVariableFormatted(
        buf.Replace( 2, 9, _L("") ),
        buf, output, KRenderDefault );

    // delete from index 1, 2 characters
    buf.Copy( _L("Hello!" ));
    RenderVariableString(buf);
    ExecuteAndRenderVariableFormatted(
        buf.Delete( 1, 2 ),
        buf, output, KRenderDefault );

    // ---------- Trimming ----------
    output.Append( _L("\nTrimming. The buf in each case is\n") );
    buf.Copy( _L( "    Hello   World!  " ) );
    RenderVariableString(buf);

    // trim from left
    ExecuteAndRenderVariableFormatted(
        buf.TrimLeft(),
        buf, output, KRenderDefault );

    // trim from right
    buf.Copy( _L( "    Hello   World!  " ) );
    ExecuteAndRenderVariableFormatted(
        buf.TrimRight(),
        buf, output, KRenderDefault );

    // trim from left & right
    buf.Copy( _L( "    Hello   World!  " ) );
    ExecuteAndRenderVariableFormatted(
        buf.Trim(),
        buf, output, KRenderDefault );

    // trim from left & right & from middle
    buf.Copy( _L( "    Hello   World!  " ) );
    ExecuteAndRenderVariableFormatted(
        buf.TrimAll(),
        buf, output, KRenderDefault );

    // ---------- Filling ----------
    output.Append( _L("\nFilling...\n") );
    buf.Copy( _L( "abcd" ) );
    RenderVariableString(buf);
    // fill chars from zero index to current length with given char
    ExecuteAndRenderVariableFormatted(
        buf.Fill('*'),
        buf, output, KRenderCharacteristics );

    // fill chars from zero index to 11 with given char
    ExecuteAndRenderVariableFormatted(
        buf.Fill('.', 12),
        buf, output, KRenderCharacteristics );

    // fill chars from zero index to 6 with binary zero
    ExecuteAndRenderVariableFormatted(
        buf.FillZ(6),
        buf, output, KRenderCharacteristics | KRenderContentAsBinary);

    // change content of buffer (so length changes)
    ExecuteAndRenderVariableFormatted(
        buf.Copy( _L("Hey!") ),
        buf, output, KRenderDefault);
    // fill chars from zero index to current length binary zero
    ExecuteAndRenderVariableFormatted(
        buf.FillZ(),
        buf, output, KRenderCharacteristics | KRenderContentAsBinary);

    // ---------- Filling ----------
    output.Append( _L("\nText conversions...\n") );
    buf.Copy( _L( "Hello World" ) );
    RenderVariableString(buf);
    ExecuteAndRenderVariableFormatted(
        buf.LowerCase(),
        buf, output, KRenderDefault);

    ExecuteAndRenderVariableFormatted(
        buf.UpperCase(),
        buf, output, KRenderDefault);

    ExecuteAndRenderVariableFormatted(
        buf.Capitalize(),
        buf, output, KRenderDefault);

    // ---------- Swapping ----------
    output.Append( _L("\nSwapping...\n") );
    TBuf<20> buf1( _L("buf1!") );
    TBuf<20> buf2( _L("buf2**") );
    RenderVariableString( buf1 );
    RenderVariableString( buf2 );
    output.Append( _L("After buf1.Swap(buf2)\n") );
    buf1.Swap(buf2);
    RenderVariableString( buf1 );
    RenderVariableString( buf2 );

    // ---------- AppendXXX ----------
    // AppendXXX() methods are similar like other methods in descriptor API.
    // They just append to the end of the descriptor buffer instead of
    // replacing the current content. That's why appending methods are not
    // described here.

    iViewer->UpdateView();
    }

// -----------------------------------------------------------------------------
// This example method is documented in header file "DescriptorExamples.h"
// -----------------------------------------------------------------------------
void CDescriptorExamples::CharacterConversionsL()
    {
{
    // Create session to the file server
    RFs fsSession;
    User::LeaveIfError( fsSession.Connect() );
    CleanupClosePushL(fsSession);

    // Create character converter that can convert between
    // SMS encoding and UCS-2
    CCnvCharacterSetConverter *converter;
    converter = CCnvCharacterSetConverter::NewLC();
    converter->PrepareToConvertToOrFromL(KCharacterSetIdentifierSms7Bit,
                                         fsSession );

    // Declare descriptor containing a string encoded with
    // 7-bit SMS encoding.
    const TUint8 SMSSourceBytes[] =
        {
        54,      // character '6'  (takes one byte)
        2,       // character '$'  (takes one byte)
        27, 61,  // character '~'  (takes two bytes)
        53,      // character '5'  (takes one byte)
        27, 101, // euro character (takes two bytes)
        };
    TPtrC8 SMSEncodedStr(SMSSourceBytes,sizeof(SMSSourceBytes));

    // Convert the SMS encoded message to the UCS-2 encoding
    TInt state = CCnvCharacterSetConverter::KStateDefault;
    TBuf<64> UCS2Buf;
    converter->ConvertToUnicode(UCS2Buf, SMSEncodedStr, state);

    // remove objects from the cleanup stack
    CleanupStack::PopAndDestroy(2); // converter, fsSession
}
    // this is used to point to correct position in view buffer
    TPtr output( iViewer->GetViewBuffer() );
    CArrayFix<CCnvCharacterSetConverter::SCharacterSet>* sets;
    RFs fsSession;
    CCnvCharacterSetConverter *converter;

    // ---------- List available converters ----------
    // This example lists all character sets converters available in the
    // system.
    RenderHeader( _L( "CharacterConversions: List converters" ), output );

    // open connection to file server and push the instance to the cleanup
    // stack
    User::LeaveIfError(fsSession.Connect());
    CleanupClosePushL(fsSession);
    // get list of character sets and push it to cleanup stack
    sets = CCnvCharacterSetConverter::CreateArrayOfCharacterSetsAvailableL(fsSession);
    CleanupStack::PushL(sets);
    // show names
    output.Append( _L( "\nList of available character converters:\n" ) );
    for( int i=sets->Count()-1; i>=0; i-- )
    {
        TPtrC name = sets->At(i).Name();
        output.AppendFormat( _L("  %S\n"), &name );
    }

    // ---------- Convert from 7-bit SMS to Unicode ----------
    // This example converts SMS to Unicode string.
    //
    // SMS character set consist of characters whose value is specified with 7
    // bits. So 128 characters are available in 7-bit SMS character set (default
    // alphabet). However, character set is extended so that if a character in
    // message is ESC (0x1B) the next character is treated as extended character
    // outside the default alphabet. So 7-bit SMS encoding can represent 255
    // different characters when using extended behaviour.
    //
    // The default alphabet is specified in ETSI "GSM 03.38". However, to
    // quickly view standard characters and extended ones, refer to
    // (http://www.ozeki.hu/index.phtml?ow_page_number=137) or to
    // (http://www.csoft.co.uk/character_sets/gsm.htm) or google with search
    // string "GSM character set" or "SMS character set".
    //
    // GSM 03.38 specification specifies that SMS message can be encoded with
    // four character encodings. This example codes the string with default
    // 7-bit SMS alphabet. SMS can consist of maximum of 160 characters. Packing
    // mechanism specified in GSM 03.38 makes it possible to pack 160 7-bit
    // characters to 140 bytes.
    //
    // Symbian provices character conversion for SMS 7-bit SMS data. However, it
    // doesn't use packed format specified in GSM 03.38. Instead, it treats the
    // binary array (of 8 bit numbers) to store 7 bit SMS characters.
    //
    // Lets assume that we have a SMS that consist of string "6$~5e" (where the
    // 'e' is meant to be an euro sign). The SMS and Unicode character codes for
    // the characters are:
    //
    // char | SMS value (7bit) | Unicode value (16 bit)
    // ------------------------------------------------
    // '6'  |  54              |   54
    // '$'  |   2              |   36
    // '~'  |  27              |  126 // Extended character coded
    //      |  61              |      // with two 7 bit values: ESC+code
    // '5'  |  53              |   53
    // 'e'  |  27              | 8364 // "Euro sign": Extended character
    //      | 101              |      // coded with two 7 bit values: ESC+code
    //
    const TUint8 SMSSourceBytes[] =
        {
        54,2,27,61,53,27,101
        };

    // create converter that converts character data between 7-bit GSM data and
    // Unicode
    converter = CCnvCharacterSetConverter::NewLC();
    converter->PrepareToConvertToOrFromL( KCharacterSetIdentifierSms7Bit,
                                          fsSession );

    // declare output buffer and declare a 8 bit descriptor to describe our
    // 7-bit SMS data.
    TInt state = CCnvCharacterSetConverter::KStateDefault;
    TBuf16<10> unicodeConverted;
    TPtrC8 SMSSource( SMSSourceBytes, sizeof( SMSSourceBytes ) );

    // convert the 7-bit SMS data to Unicode characters and store results to
    // variable "unicodeConverted"
    converter->ConvertToUnicode( unicodeConverted, SMSSource, state );

    // show unicode string converted from SMS ("6$~5");
    RenderHeader( _L( "CharacterConversions: SMS to Unicode" ), output );
    output.Append( _L("SMS data consist of 7 bit character values (where euro and tilde char consist of two numbers):\n") );
    RenderVariableBinary( SMSSource );
    output.Append( _L("\nWhile unicode uses 16 bits to store each character:\n") );
    RenderVariableBinary( unicodeConverted );
    RenderVariableString( unicodeConverted );

    // ---------- Unicode To SMS ----------
    // Declare unicode string that consist "normal" and special characters
    RenderHeader( _L( "CharacterConversions: Unicode to SMS" ), output );
    
    //nice characters but can't be compiled with GCCE ;)
    //TBuf<64> unicodeSource(_L("Some nice chars: ,,, \x20AC") ); // 0x20AC = euro sign, desicmal 8364;
    TBuf<64> unicodeSource(_L("Some nice chars: \x00F6,\x00E4,\x004F,\x00C4 \x20AC") );
    
    // == 00F6 
    // == 004F
    // == 00E4
    // == 00C4
    
    RenderVariableString( unicodeSource ); 
    
    RenderVariableBinary( unicodeSource );

    // lets convert the unicode to 7 bit SMS
    TBuf8<128> convertedToSMS;
    converter->ConvertFromUnicode( convertedToSMS, unicodeSource );
    RenderVariableBinary( convertedToSMS );
    RenderVariableString( convertedToSMS );

    output.Append( _L("\nEuro character is espaced in SMS encoding so number of 'chars' differ:\n") );
    RenderResult( unicodeSource.Length() );
    RenderResult( convertedToSMS.Length() );

    output.Append( _L("\nUnicode consist of 16 bit chars while SMS consist of 8 bit ones. Number of bytes in encodings:\n") );
    RenderResult( unicodeSource.Size() );
    RenderResult( convertedToSMS.Size() );

    iViewer->UpdateView();
    CleanupStack::PopAndDestroy(3); // fsSession, sets, converter
    }

// -----------------------------------------------------------------------------
// This example method is documented in header file "DescriptorExamples.h"
// -----------------------------------------------------------------------------
void CDescriptorExamples::LexicalAnalysis()
    {
    TPtr output( iViewer->GetViewBuffer() );
    RenderHeader( _L( "LexicalAnalysis" ), output );

    // TLex provides methods for parsin text. Its usage is quite well
    // documented in symbian documentation "Using TLex". However, converting
    // strings to numbers is not explained. So this example explains them.
    TLex lex;

    TInt num1;
    lex.Assign( _L("-254 is number") );
    RenderVariableString(lex.Remainder());
    // parse integer number
    ExecuteAndRenderVariableFormatted(
        lex.Val(num1),
        num1, output, KRenderDefault );

    TUint32 num2;
    lex.Assign( _L("2ff is hex number") );

    RenderVariableString(lex.Remainder());
    // parse hex number
    ExecuteAndRenderVariableFormatted(
        lex.Val(num2, EHex),
        num2, output, KRenderDefault );

    // parse float
    TReal32 num3;
    lex.Assign( _L("234.678 is real number") );
    RenderVariableString(lex.Remainder());
    // parses the value as 234.677993774414!
    lex.Val(num3, '.');
    output.Append( _L( "lex.Val(num3, '.') -> num3=" ) );
    TRealFormat realFormat;
    output.AppendNum( num3, realFormat );

    iViewer->UpdateView();
    }
