/*
 * Copyright  2008 Nokia Corporation.
 */

#include "StringRenderer.h"

// -----------------------------------------------------------------------------
// Append() documented in StringRenderer.h.
// -----------------------------------------------------------------------------
void Append( const TDesC8 &aSrc, TDes16 &aDst )
    {
    for( TInt i=0; i < aSrc.Length(); i++ )
        {
        aDst.Append(aSrc[i]);
        }
    }

void Append( const TDesC16 &aSrc, TDes16 &aDst )
    {
    aDst.Append( aSrc );
    }

// -----------------------------------------------------------------------------
// RenderAsNumbers() documented in StringRenderer.h.
// -----------------------------------------------------------------------------
template <class TDESC>
LOCAL_C void RenderAsNumbersTemplated( const TDESC &anArray, TDes16 &aOutput )
    {
    aOutput.Append('{');
    for( TInt i=0; i < anArray.Length(); i++ )
        {
        if( i>0 )
            {
            aOutput.Append(',');
            }
        aOutput.AppendNum( (TInt) anArray[i] );
        }
    aOutput.Append('}');
    }

void RenderAsNumbers( const TDesC8 &anArray, TDes16 &aOutput )
{
    RenderAsNumbersTemplated(anArray, aOutput);
}

void RenderAsNumbers( const TDesC16 &anArray, TDes16 &aOutput )
{
    RenderAsNumbersTemplated(anArray, aOutput);
}

// -----------------------------------------------------------------------------
// Renders contents of 8 or 16 bit descriptor either as numbers or characters.
// The example result is "Hello" when aRenderFormat == KRenderDefault and
// {72,101,108,108,111} when aRenderFormat == KRenderContentAsBinary
// -----------------------------------------------------------------------------
template <class DESC>
LOCAL_C void RenderDescriptorContent( const DESC &aVariable,
                                      TDes16 &aOutput,
                                      TInt aRenderFormat=KRenderDefault )
    {
    if( aRenderFormat & KRenderContentAsBinary )
        { // render as numbers
        RenderAsNumbers( aVariable, aOutput );
        }
    else
        { // render as string
        aOutput.Append('"');
        Append(aVariable, aOutput);
        aOutput.Append('"');
        }
    }

// -----------------------------------------------------------------------------
// RenderObject() documented in StringRenderer.h.
// -----------------------------------------------------------------------------
_LIT( KTDesCCharacteristicsFormat, "(len=%d)" );
_LIT( KTDesCharacteristicsFormat,  "(len=%d, maxlen=%d)" );

void RenderObject(const TDesC8 &aVariable, TDes16 &aOutput, TInt aRenderFormat)
    {
    RenderDescriptorContent( aVariable, aOutput, aRenderFormat );
    if( aRenderFormat & KRenderCharacteristics )
        {
        aOutput.Append(' ');
        aOutput.AppendFormat( KTDesCCharacteristicsFormat, aVariable.Length() );
        }
    }

void RenderObject(const TDes8 &aVariable, TDes16 &aOutput, TInt aRenderFormat)
    {
    RenderDescriptorContent( aVariable, aOutput, aRenderFormat );
    if( aRenderFormat & KRenderCharacteristics )
        {
        aOutput.Append(' ');
        aOutput.AppendFormat( KTDesCharacteristicsFormat, aVariable.Length(),
            aVariable.MaxLength() );
        }
    }

void RenderObject(const TDesC16 &aVariable, TDes16 &aOutput, TInt aRenderFormat)
    {
    RenderDescriptorContent( aVariable, aOutput, aRenderFormat );
    if( aRenderFormat & KRenderCharacteristics )
        {
        aOutput.Append(' ');
        aOutput.AppendFormat( KTDesCCharacteristicsFormat, aVariable.Length() );
        }
    }

void RenderObject(const TDes16 &aVariable, TDes16 &aOutput, TInt aRenderFormat)
    {
    RenderDescriptorContent( aVariable, aOutput, aRenderFormat );
    if( aRenderFormat & KRenderCharacteristics )
        {
        aOutput.Append(' ');
        aOutput.AppendFormat( KTDesCharacteristicsFormat,
                              aVariable.Length(),
                              aVariable.MaxLength() );
        }
    }

void RenderObject(const TInt &aVariable,
                  TDes16 &aOutput,
                  TInt /*aRenderFormat*/ )
    {
    aOutput.AppendNum( aVariable );
    }

// -----------------------------------------------------------------------------
// RenderHeader() documented in StringRenderer.h.
// -----------------------------------------------------------------------------
void RenderHeader(const TDesC &aHeader, TDes &aOutput)
    {
    _LIT( KFormat, "\n--- %S ---\n" );
    TInt spaceAvailable = aOutput.MaxLength() - aOutput.Length();
    if( spaceAvailable >=
            aHeader.Length() +
            KFormat().Length()-2 ) // "%S" is replaced with header text
        {
        aOutput.AppendFormat( KFormat, &aHeader );
        }
    }
