// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// The following class describes the Inverter class which recieves input 
// through InverterInq and sends it to InverterOutQ after inverting the recieved words.
// It implements a periodic timer which opens the InverterInq every 10 seconds.
//



/**
 @file
*/

#include "CInverter.h"
#include "e32keys.h"     // for EKeyEscape.

//Periodic timer interval of 10 seconds.
const TInt KPeriodicTimerInterval10Sec(10000000);

//Error code to indicate Stop command.
const TInt KErrStop = -100; 

/**
Destructor, cancels the periodic timer, close the message queues.
*/
CInverter::~CInverter()
	{
    if(iTimer)
        {
        // Calling Cancel without checking if the timer is active is safe
        iTimer->Cancel();
        }
    delete iTimer;
	iInMsgQ.Close();
	iOutMsgQ.Close();
	}

CInverter* CInverter::NewL()
	{
	CInverter* self=new(ELeave)CInverter();
	CleanupStack::PushL(self);
	self->ConstructL();
	CleanupStack::Pop(self);
	return self;
	}
/**
Construct a periodic timer and starts the timer function.
*/
void CInverter::ConstructL()
    {
    // Initialize the periodic timer.
    iTimer = CPeriodic::NewL(CActive::EPriorityIdle);
    // Start the periodic timer, when ever the time elapsed
    // the StaticWake() will get called.

    iTimer->Start(KPeriodicTimerInterval10Sec, KPeriodicTimerInterval10Sec,TCallBack(StaticWake, this));
    
    }

/**
The following function is called periodically at an interval specified in the Start function
of the CPeriodic class.
*/
TInt CInverter::StaticWake(TAny* aAny)
	{
	CInverter* self = static_cast<CInverter*>( aAny );
	TInt ret = self->RecieveMsg();
	if(ret!= KErrStop) //KErrStop indicates that system has to be stopped and no more messages to be sent.
	self->SendMsg();
	return KErrNone; // Return value ignored by CPeriodic
	}

/**
The following function opens the InveterInQ message queue and recieves data from it. If 
the recieved message indicates stop command given by the first process, it stops the timer
and closes the message queue etc. This function is called at a regular interval of 5 secs, so 
every five seconds data is recieved by this process.
*/
TInt CInverter::RecieveMsg()
    {

    //Opens the message queue input to the inverter.
    _LIT(KGlobalInverterInQ, "InverterInQ");
    TInt ret = iInMsgQ.OpenGlobal(KGlobalInverterInQ); 
    	
    	if (ret == KErrNone)
    	{
    	//Recieve the message coming from the first process.
    	iMsgQData.Zero();
    	iInMsgQ.ReceiveBlocking(iMsgQData);
    	
    	//If the recieved data indicates stop command, stop the system.
    	if(iMsgQData[0]==EKeyEscape)
			{
			this->Stop();
			return KErrStop;
			}
    	}
    	return KErrNone;
    }

/**
The following function will send the message to the message queue after inverting the words.
*/
void CInverter::SendMsg()
	{
	//Opens the global message Queue named InverterOutQ.
	_LIT(KGlobalInverterOutQ, "InverterOutQ");
	TInt ret = iOutMsgQ.OpenGlobal(KGlobalInverterOutQ); 
	
	//Sends the inverted words to the opened message queue.
	if (ret == KErrNone)
	{
	DoInvert();
	iOutMsgQ.SendBlocking(idestMsg);
	}
	}

/**
The following function inverts words in the sentence, recieved by the message queue. 
*/
void CInverter::DoInvert()
{
idestMsg.Zero();
TInt i=0; 
while(iMsgQData[i]!= '\r')
{
	while((iMsgQData[i]!= ' ')&&(iMsgQData[i]!= '\r'))
	{
	iwords.Append(iMsgQData[i]);
	i++; 
	}
	ReverseWord(iwords);
	idestMsg.Append(itmpWord); 
	idestMsg.Append(' ');
	if(iMsgQData[i]== '\r')
	{
	i--;
	}
	i++;
	iwords.Zero();
    }
}

/**
Reverse the individual words of the sentence, filtered in the function above.
*/
void CInverter::ReverseWord(TBuf<20> buf)
	{
	itmpWord.Zero();
	TInt size = (buf.Size())/2;
	for(int i=1; i<= size; i++)
		{
		itmpWord.Append(buf[size -i]);
		}
	}

/**
This function is called when user gives stop command, it closes the message queue, stops the
scheduler and cancels the timer.
*/
void CInverter::Stop()
	{
	iTimer->Cancel();
	CActiveScheduler::Stop();
	}

LOCAL_C void DoStartL()
	{

	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);	
	
	CInverter* invert = CInverter::NewL();
	CleanupStack::PushL(invert);
	
	//Start the scheduler for the periodic timer
	CActiveScheduler::Start();
	
	CleanupStack::PopAndDestroy(invert);
	CleanupStack::PopAndDestroy(scheduler);
	}

GLDEF_C TInt E32Main()
	{
	__UHEAP_MARK;
	CTrapCleanup* cleanup = CTrapCleanup::New();
	if(cleanup == NULL)
		{
		return KErrNoMemory;
		}
	

	TRAPD(mainError, DoStartL());
	if(mainError != KErrNone)
		{
		_LIT(KUserPanic,"Failed to complete");	
		User::Panic(KUserPanic, mainError);
		}

		delete cleanup;
		__UHEAP_MARKEND;
		return KErrNone;
	}
