// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Implements the write operation on a shared memory block.
//



/**
 @file
*/
#include "sharedmem.h"
#include "adder.h"

/**
Performs the two-phase construction of an object of the CAdder class.
@param aConsole The console object.
@return A CAdder object.
*/
CAdder* CAdder::NewL(CConsoleBase* aConsole)
	{
	CAdder* self = new (ELeave) CAdder;
	CleanupStack::PushL(self);
	self->ConstructL(aConsole);
	CleanupStack::Pop(self);
	return self;
	}

/**
Handles the key press events from the console.
It stops the active scheduler for any key press event.
*/
void CAdder::RunL()
	{
	// Get the key code.
	TUint8 option = iConsole->KeyCode();
	// Print the selected option.
	_LIT(KTextFormat,"%c\n");
	iConsole->Printf(KTextFormat,option);
	// Stop the timer and the active scheduler.
	StopTimer();
	CActiveScheduler::Stop();
	}

/**
Cancel any outstanding request.
*/
void CAdder::DoCancel()
	{
	if(IsActive())
		{
		// Cancel any outstanding read requests.
		iConsole->ReadCancel();
		}
	}

/**
Destructor.
*/
CAdder::~CAdder()
	{
	// Cancel all outstanding requests.
	DoCancel();

	// Delete the timer object.
	iPeriodic->Cancel();
	delete iPeriodic;

	// Close all resource handles.
	iChunk.Close();
	iMutex.Close();
	iCondVar.Close();
	}

/**
Constructor.
*/
CAdder::CAdder():CActive(EPriorityUserInput)
	{
	}

/**
The second phase constructor of the CAdder class.
It creates the following member objects of the class:
- iCondVar
- iChunk
- iMutex
It also initialises the value of chunk to 0.
@param aConsole The console object.
*/
void CAdder::ConstructL(CConsoleBase* aConsole)
	{
	// Create the global condition variable.
	User::LeaveIfError(iCondVar.CreateGlobal(KCondVarName));
	// Create the global chunk.
	User::LeaveIfError(iChunk.CreateGlobal(KChunkName,KChunkSize,KChunkSize));
	// Create the global mutex variable.
	User::LeaveIfError(iMutex.CreateGlobal(KMutexName));

	// Create the CPeriodic object.
	iPeriodic = CPeriodic::NewL(CActive::EPriorityUserInput);
	iConsole = aConsole;

	// Get the address of the chunk.
	TUint8 *ptr = iChunk.Base();
	// Initialise the value of the chunk to 0.
	*ptr = 0;

	// Add the object to the active scheduler.
	CActiveScheduler::Add(this);
	}

/**
Issues an outstanding request to get a keystroke from the console.
*/
void CAdder::ReadFunction()
	{
	_LIT(KTextMessage,"Press a key to exit...\n");
	iConsole->Printf(KTextMessage);
	// Wait for a key press event.
	iConsole->Read(iStatus);
	SetActive();
	}

/**
Starts the timer object.
@see CPeriodic::Start().
*/
void CAdder::StartTimer()
	{
	// Start generating periodic events.
	iPeriodic->Start(0,3000000,TCallBack(AddFunction,this));
	}

/**
Stops the timer object.
*/
void CAdder::StopTimer()
	{
	// Cancel the  outstanding request.
	iPeriodic->Cancel();
	}

/**
The call back function for the CAdder::iPeriodic object.
@param aPtr Contains the address of the object passed as an argument to the AddFunction().
@return KErrNone.
*/
TInt CAdder::AddFunction(TAny* aPtr)
	{
	CAdder* ptr = static_cast<CAdder*> (aPtr);
	_LIT(KTxtPanic,"Pointer is NULL");
	__ASSERT_ALWAYS(ptr,User::Panic(KTxtPanic,-1));
	// Invoke the Add() function.
	ptr->Add();
	return KErrNone;
	}

/**
Adds a random integer value to the global chunk.
*/
void CAdder::Add()
	{
	// Acquire the mutex.
	iMutex.Wait();

	// Get a random number.
	TInt randVal = Math::Random() % 10;

	// Get the address of the chunk.
	TUint8 *ptr = iChunk.Base();

	// Print the value of the chunk before addition.
	iConsole->Printf(_L("Value read from the shared memory: %d\n"),*ptr);

	// Add the random value to the shared memory variable.
	*ptr += randVal;

	while(*ptr > KUpperThreshold)
		{
		// Wait on the condition variable if the result is greater than 100.
		_LIT(KBlockMessage,"Adder blocked by condVar until Subtractor signals that value has been decreased.\nIntermediate value of the chunk = %d\n");
		iConsole->Printf(KBlockMessage,*ptr);
		iCondVar.Wait(iMutex);
		}
	// Print the updated value of the chunk.
	iConsole->Printf(_L("Value of the shared memory increased to : %d\n"),*ptr);

	// Signal the mutex and the condition variable.
	if(*ptr > KLowerThreshold)
		{
		// Signal that the level is safe for subtraction to (re)start.
		iCondVar.Signal();
		}
	iMutex.Signal();
	}

LOCAL_D CConsoleBase* console;
LOCAL_C void DoExampleL();
LOCAL_C void callExampleL();

LOCAL_C void DoExampleL()
	{
	// Create and install the active scheduler.
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);

	// Create the CAdder object.
	CAdder* adder = CAdder::NewL(console);
	CleanupStack::PushL(adder);

	// Start the timer of the CAdder object.
	adder->StartTimer();
	// Issue an asynchronous read request.
	adder->ReadFunction();
	// Start the active scheduler.
	CActiveScheduler::Start();

	CleanupStack::PopAndDestroy(2,scheduler);
	}

GLDEF_C TInt E32Main() // main function called by E32
    {
	__UHEAP_MARK;
	CTrapCleanup* cleanup=CTrapCleanup::New(); // get clean-up stack
	TRAPD(error,callExampleL()); // more initializations, then do example
	delete cleanup; // destroy clean-up stack
	__ASSERT_ALWAYS(!error,User::Panic(KTxtEPOC32EX,error));
	__UHEAP_MARKEND;
	return 0; // and return
    }

LOCAL_C void callExampleL() // initialise and call example code under cleanup stack
    {
	console=Console::NewL(KTxtExampleCode,TSize(KConsFullScreen,KConsFullScreen));
	CleanupStack::PushL(console);
	TRAPD(error,DoExampleL()); // perform example function
	if (error)
		console->Printf(KFormatFailed, error);
	else
		console->Printf(KTxtOK);
	console->Printf(KTxtPressAnyKey);
	console->Getch(); // get and ignore character
	CleanupStack::PopAndDestroy(); // close console
    }
