// Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Implements the write operation on a shared memory block.
//



/**
 @file
*/
#include "sharedmem.h"
#include "subtractor.h"

/**
Performs the two-phase construction of an object of the CAdder class.
@param aConsole The console object.
@return A CSubtractor object.
*/
CSubtractor* CSubtractor::NewL(CConsoleBase* aConsole)
	{
	CSubtractor* self = new (ELeave) CSubtractor;
	CleanupStack::PushL(self);
	self->ConstructL(aConsole);
	CleanupStack::Pop(self);
	return self;
	}

/**
Handles the key press events from the console.
It stops the active scheduler for any key press event.
*/
void CSubtractor::RunL()
	{
	// Get the key code.
	TUint8 option = iConsole->KeyCode();
	// Print the selected option.
	_LIT(KTextFormat,"%c\n");
	iConsole->Printf(KTextFormat,option);
	// Stop the timer and the active scheduler.
	StopTimer();
	CActiveScheduler::Stop();
	}

/**
Cancel any outstanding request.
*/
void CSubtractor::DoCancel()
	{
	if(IsActive())
		{
		// Cancel any outstanding read requests.
		iConsole->ReadCancel();
		}
	}

/**
Destructor.
*/
CSubtractor::~CSubtractor()
	{
	// Cancel all outstanding requests.
	DoCancel();
	// Delete the timer object.
	iPeriodic->Cancel();
	delete iPeriodic;
	}

/**
Constructor.
*/
CSubtractor::CSubtractor():CActive(EPriorityUserInput)
	{
	}

/**
The second phase constructor of the CSubtractor class.
It creates the CPeriodic object and adds the object to the active scheduler.
@param aConsole The console object.
*/
void CSubtractor::ConstructL(CConsoleBase* aConsole)
	{
	// Open the global condition variable.
	User::LeaveIfError(iCondVar.OpenGlobal(KCondVarName));
	// Open the global chunk variable.
	User::LeaveIfError(iChunk.OpenGlobal(KChunkName,EFalse));
	// Open the global mutex variable.
	User::LeaveIfError(iMutex.OpenGlobal(KMutexName));

	// Create the CPeriodic object.
	iPeriodic = CPeriodic::NewL(CActive::EPriorityUserInput);
	iConsole = aConsole;

	// Add the object to the active scheduler.
	CActiveScheduler::Add(this);
	}

/**
Issues an outstanding request to get a keystroke from the console.
*/
void CSubtractor::ReadFunction()
	{
	_LIT(KTextMessage,"Press a key to exit...\n");
	iConsole->Printf(KTextMessage);
	// Wait for a key press event.
	iConsole->Read(iStatus);
	SetActive();
	}

/**
Starts the timer object.
@see CPeriodic::Start().
*/
void CSubtractor::StartTimer()
	{
	iPeriodic->Start(0,3000000,TCallBack(SubtractFunction,this));
	}

/**
Stops the timer object.
*/
void CSubtractor::StopTimer()
	{
	// Cancel the  outstanding request.
	iPeriodic->Cancel();
	}

/**
The call back function for the CSubtractor::iPeriodic object.
@param aPtr Contains the address of the object passed as an argument to the SubtractFunction().
@return KErrNone.
*/
TInt CSubtractor::SubtractFunction(TAny* aPtr)
	{
	CSubtractor* ptr = static_cast<CSubtractor*> (aPtr);
	_LIT(KTxtPanic,"Unexpected datatype\n");
	__ASSERT_ALWAYS(ptr,User::Panic(KTxtPanic,-1));
	// Invoke the Subtract() function.
	ptr->Subtract();
	return KErrNone;
	}

/**
Subtracts a random integer value from the global chunk.
*/
void CSubtractor::Subtract()
	{
	// Acquire the mutex.
	iMutex.Wait();

	// Get a random number.
	TInt randVal = Math::Random() % 10;

	// Get the address of the chunk.
	TUint8 *ptr = iChunk.Base();

	// Print the value of the chunk before subtraction.
	iConsole->Printf(_L("Value read from the shared memory: %d\n"),*ptr);

	// Subtract the random value from the shared memory variable.
	*ptr -= randVal;

	while(*ptr < KLowerThreshold)
		{
		// Wait on the condition variable if the result is lesser than 0.
		_LIT(KBlockMessage,"Subtractor blocked by condVar until Adder signals that value has been increased.\nIntermediate value of the chunk = %d\n");
		iConsole->Printf(KBlockMessage,*ptr);
		iCondVar.Wait(iMutex);
		}
	// Print the updated value of the chunk.
	iConsole->Printf(_L("Value of the shared memory decreased to : %d\n"),*ptr);

	// Signal the mutex and the condition variable.
	if(*ptr < KUpperThreshold)
		{
		// Signal that the level is safe for addition to (re)start.
		iCondVar.Signal();
		}
	iMutex.Signal();
	}

LOCAL_D CConsoleBase* console;
LOCAL_C void DoExampleL();
LOCAL_C void callExampleL();

LOCAL_C void DoExampleL()
	{
	// Create and install the active scheduler.
	CActiveScheduler* scheduler = new (ELeave) CActiveScheduler();
	CleanupStack::PushL(scheduler);
	CActiveScheduler::Install(scheduler);

	// Create the CSubtractor object.
	CSubtractor* subtractor = CSubtractor::NewL(console);
	CleanupStack::PushL(subtractor);

	// Start the timer of the CSubtractor object.
	subtractor->StartTimer();
	// Issue an asynchronous read request.
	subtractor->ReadFunction();
	// Start the active scheduler.
	CActiveScheduler::Start();

	CleanupStack::PopAndDestroy(2,scheduler);
	}

GLDEF_C TInt E32Main() // main function called by E32
    {
	__UHEAP_MARK;
	CTrapCleanup* cleanup=CTrapCleanup::New(); // get clean-up stack
	TRAPD(error,callExampleL()); // more initializations, then do example
	delete cleanup; // destroy clean-up stack
	__ASSERT_ALWAYS(!error,User::Panic(KTxtEPOC32EX,error));
	__UHEAP_MARKEND;
	return 0; // and return
    }

LOCAL_C void callExampleL() // initialise and call example code under cleanup stack
    {
	console=Console::NewL(KTxtExampleCode,TSize(KConsFullScreen,KConsFullScreen));
	CleanupStack::PushL(console);
	TRAPD(error,DoExampleL()); // perform example function
	if (error)
		console->Printf(KFormatFailed, error);
	else
		console->Printf(KTxtOK);
	console->Printf(KTxtPressAnyKey);
	console->Getch(); // get and ignore character
	CleanupStack::PopAndDestroy(); // close console
    }
