/*
 * Copyright  2008 Nokia Corporation.
 */


#ifndef __CALENDAR_API_EXAMPLE_ENGINE_H
#define __CALENDAR_API_EXAMPLE_ENGINE_H

// INCLUDES
#include <e32base.h>    //CBase
#include <calsession.h>
#include <calentry.h>
#include <calinstanceview.h>
#include <calentryview.h>
#include <CalProgressCallBack.h>
#include <AknProgressDialog.h>
#include <eikprogi.h>   //CEikProgressInfo
#include "CalendarHelperEntry.h" //CCalHelperEntry

// CONSTANTS
const TInt KMaxSync = 2;
const TInt KDefaultAlarmHour = 8;
const TInt KUndefinedModifyIndex = -1;

enum TSearchType
    {
    EWeek,
    EMonth,
    ESixMonths,
    EYear
    };

//commands to observer
class MCalenderEngineObserverUI
    {
    public:
        virtual void ActivateView(TUid aViewId) = 0;
    };

//Commands called from views:
class MCalendarEngineCommandsInterface 
    {
    public:
       virtual void DoSearchL(  const TSearchType& aType) = 0;
       virtual void DoAddL() = 0;
       virtual void DoSaveL() = 0;
       virtual void CreateEntryForModificationL(TBool& aModify)=0;
       virtual TInt EntryCount() const = 0;
       
       virtual void DeleteEntryL(const TInt& aIndex) = 0;
       virtual void ExecuteDeletionL() = 0;
       
       virtual CCalHelperEntry& Entry(const TInt& aIndex) = 0;
       virtual void SetModifyIndex(const TInt& aIndex) = 0;
       virtual TInt ModifyIndex() const = 0;
       virtual TBool SetValuesToNewEntry( const TDesC& aName,
                        const TDateTime& aDate,
                        const TBool& aAlarm,
                        const TDateTime& aAlarmTime,
                        const TInt& aSynchronizationMethod) = 0; 
       virtual void GetValuesToSet(TDes& aName, TTime& aDate,
                       TBool& aAlarm, TTime& aAlarmTime,
                                TInt& aSync) = 0 ;
    
    };

// CLASS DECLARATION

/*!
*  CCalendarAPIexampleEngine engine class.
* Provides support for the following features:
* - Add anniversaries
* - Update anniversaries
* - Delete anniversaries
* - Finds anniversaries
* 
*/
class CCalendarAPIexampleEngine :   public CBase,
                                    public MCalProgressCallBack,
                                    public MCalendarEngineCommandsInterface
{
public: // Constructors and destructor

   /*!
    * Two-phased constructor.
    */
    static CCalendarAPIexampleEngine* NewL(MCalenderEngineObserverUI& aAppUi);

   /*!
    * Destructor.
    */
    virtual ~CCalendarAPIexampleEngine();

public: // Other functions
    
   /*!
    * AddAnniversaryL()
    * 
    * Adds an anniversary to the agenda file.
    * param aAnniv - CCalEntry object that is added to the agenda file
    */    
    void AddAnniversaryL(   CCalEntry* aAnniv) ;

   /*!
    * UpdateAnniversaryL()
    * 
    * Updates an anniversarys data to the agenda file.
    * param aAnniv - CCalEntry object that is updated
    */    
    void UpdateAnniversaryL(CCalEntry* aAnniv) ;

   /*!
    * DeleteAnniversaryL()
    * 
    * Deletes an anniversary from the agenda file.
    * param aAnniv - CCalEntry object that is deleted from the agenda file
    */    
    void DeleteAnniversaryL(CCalEntry* aAnniv) ;

   /*!
    * GetAnniversariesL()
    * 
    * Finds all anniversaries within given time interval.
    * param aFrom - date where the searching begins
    * param aTo - date where the search ends
    * param aAnniversaries - found anniversaries are put to this array.
    */    
    void GetAnniversariesL( const TDateTime& aFrom, 
                            const TDateTime& aTo,                           
                            RPointerArray<CCalInstance>& aAnniversaries);

   /*!
    * EntryCount()
    * 
    * Returns the number of found anniversaries.
    */
    TInt EntryCount() const;

   /*!
    * Entry()
    * 
    * Returns a reference to an entry.
    * param aIndex - index of the wanted entry.
    */
    CCalHelperEntry& Entry(const TInt& aIndex);
    
   /*!
    * DoSearchL()
    * 
    * Finds all anniversaries in given time interval.
    * param aType - search interval (week, month...)
    */    
    void DoSearchL( const TSearchType& aType);
    
   /*!
    * DoAddL()
    *
    * Activates the proper view for adding entries.
    */      
    void DoAddL();                     
    
   /*!
    * ModifyIndex()
    * 
    * Returns the modify index. 
    */                      
    TInt ModifyIndex() const;
    
   /*!
    * SetModifyIndex()
    * 
    * Sets the modify index to given index. Modify index is used when calling
    * CreateEntryForModificationL. 
    * param aIndex - modify index is set to this index
    */                          
    void SetModifyIndex(const TInt& aIndex);
    
   /*!
    * CreateEntryForModificationL()
    * 
    * Returns a reference to an entry. If iModifyIndex is negative, a new entry is created,
    * otherwise a reference to an existing entry (defined by iModifyIndex) is returned.
    * param aModify - is set to EFalse if a new entry was created and ETrue if existing
    *                 entry was returned.
    */                              
    void CreateEntryForModificationL(TBool& aModify);
    
   /*!
    * SaveL()
    * 
    * Adds or updates the current entry to the agenda file. Current entry is defined by
    * calling CreateEntryForModificationL().
    */                              
    void SaveL();
    
    /*!
     * DoSaveL()
     * Called from the views when there's a need to save data.  
     */
    void DoSaveL();
    
   /*!
    * DeleteEntryL()
    * 
    * Deletes an entry from the agenda file.
    * param aIndex - index of which entry is deleted.
    */                                  
    void DeleteEntryL(const TInt& aIndex);
    
    /*!
     * SetValuesToNewEntry()
     *
     * Called by the view when there are values to be set to the
     * new entry
     *
     */
    TBool SetValuesToNewEntry( const TDesC& aName,
                        const TDateTime& aDate,
                        const TBool& aAlarm,
                        const TDateTime& aAlarmTime,
                        const TInt& aSynchronizationMethod);
                        
    /*!
     * GetValuesToSet()
     *
     * Called by the view to get the values to be set
     *
     */
    void GetValuesToSet(TDes& aName, TTime& aDate,
                   TBool& aAlarm, TTime& aAlarmTime,
                            TInt& aSync);   
                        
    /*!
     * Does the actual deletion of selected entry.
     * 
     *
     */
    void ExecuteDeletionL();
    
private:
   /*!
    * Symbian OS default constructor.
    */
    CCalendarAPIexampleEngine(MCalenderEngineObserverUI& aAppUi);
    void ConstructL();  
    
    /** Progress callback.
    
    This calls the observing class with the percentage 
    complete of the current operation.
    
    @param aPercentageCompleted The percentage complete. */
    virtual void Progress(TInt aPercentageCompleted);
    
    /** Progress callback.
    
    This calls the observing class when the current operation is finished.
    
    @param aError The error if the operation failed, or KErrNone if successful.
    */
    virtual void Completed(TInt aError);
    
    /** Asks the observing class whether progress callbacks are required.
    
    @return If the observing class returns EFalse, then the Progress() function
    //will not be called. */
    virtual TBool NotifyProgress(); 
    
private: // Data members

    CActiveSchedulerWait* iWait;
    CCalSession* iCalSession;
    CCalEntryView* iEntryView;
    CCalInstanceView* iInstanceView;
    TBool iCreatingEntryView;
    CAknProgressDialog* iProgressDialog;
    CEikProgressInfo* iProgressInfo;
    
    // Here are all CCalEntrys hold inside CCalHelperEntry
    // the results are put here in DoSearchL
    RPointerArray<CCalHelperEntry> iEntries;
    
    TSearchType iCurrentSearchType;
    
    //Used in creating new entries. Created in CreateEntryForModificationL
    CCalHelperEntry* iEntry;
    
    //Used in knowing which CCalEntry we're modifying
    TInt iModifyIndex;
    
    //A reference to appui, used in updating views.
    MCalenderEngineObserverUI& iAppUi;
    
    };

#endif
