/*
 * Copyright  2008 Nokia Corporation.
 */

// INCLUDE FILES
#include <calsession.h> //CCalSession
#include <calentryview.h>
#include <CalInstance.h>
#include <calinstanceview.h>
#include <txtfmlyr.h>   //CParaFormatLayer
#include <calalarm.h>   //CCalAlarm 
#include <CalendarAPIexample.rsg>
#include "CalendarAPIexampleEngine.h"
#include "CalendarAPIexample.hrh"

#include "CalendarAPIexampleEntryView.h"
#include "CalendarAPIexampleEntryContainer.h"
#include "CalendarAPIexampleEntriesView.h"
#include "CalendarAPIexampleSearchView.h" 

#include "CalendarAPIexample.pan"

#include "CalendarAPIexampleEntryView.h"

// CONSTANTS
const TInt KProgressFinalValue = 100;
_LIT(KDefaultTodoListName, "TODO");

#define KDefaultEventDisplayTime TTimeIntervalMinutes(660)
#define KDefaultAnnivDisplayTime TTimeIntervalMinutes(900)
#define KDefaultDayNoteDisplayTime TTimeIntervalMinutes(1000)

// ================= MEMBER FUNCTIONS =======================

// constructor
CCalendarAPIexampleEngine::CCalendarAPIexampleEngine(
                                            MCalenderEngineObserverUI& aAppUi)
    : iAppUi(aAppUi)
    {
    
    
    }
    
// destructor
CCalendarAPIexampleEngine::~CCalendarAPIexampleEngine()
    {
    
    if( iWait )
        {
        if( iWait->IsStarted() )
            {
            iWait->AsyncStop();
            //it's not propably safe to delete right after this
            }
        delete iWait;
        iWait = NULL;   
        }
    if (iEntryView)
        {
        delete iEntryView;
        iEntryView = 0;
        }
        
    if (iInstanceView)
        {
        delete iInstanceView;
        iInstanceView = 0;
        }

    iEntries.ResetAndDestroy();
    
    delete iEntry;
    iEntry = NULL;

    delete iProgressDialog;     
    iProgressDialog = NULL;
         
    if (iCalSession)
        {
        delete iCalSession;
        iCalSession = 0;
        }
    }
    
    
// Two-phased constructor.
CCalendarAPIexampleEngine* CCalendarAPIexampleEngine::NewL(
                                            MCalenderEngineObserverUI& aAppUi)
    {
    CCalendarAPIexampleEngine* self = 
        new (ELeave) CCalendarAPIexampleEngine(aAppUi);
    CleanupStack::PushL(self);
    self->ConstructL();
    CleanupStack::Pop(self);
    return self;
    }
    
// Symbian OS default constructor can leave.    
void CCalendarAPIexampleEngine::ConstructL()
    {
    iModifyIndex = KUndefinedModifyIndex;
    
    // allocate and construct server
    iCalSession = CCalSession::NewL();      
    
    const TDesC& file = iCalSession->DefaultFileNameL();
    // connect to the cal server
    TRAPD(err,iCalSession->OpenL(file) );   
    if( err == KErrNotFound)
        {
        iCalSession->CreateCalFileL(file);
        iCalSession->OpenL(file);
        }
    else
        {
        User::LeaveIfError(err);   
        }

    iWait = new (ELeave) CActiveSchedulerWait;
    }

/*******************OBSERVER functions***********************/
    
// ----------------------------------------------------
// CCalendarAPIexampleSearchContainer::Progress()
// Monitors the progress of agenda model operations.
// Creates a progress dialog when first called and 
// updates its status on later calls. Note that if the 
// agenda models operation completes quickly, for 
// instance when opening a small file, then Progress() 
// will probably not be called at all.
// ----------------------------------------------------
//                  
void CCalendarAPIexampleEngine::Progress(TInt aPercentageCompleted)
    {
    if (!iProgressDialog)
        {
        
        TRAPD
            (   
            error,
            iProgressDialog = new (ELeave) CAknProgressDialog(
                    REINTERPRET_CAST(CEikDialog**, &iProgressDialog), ETrue);
            iProgressDialog->PrepareLC(R_CALENDARAPIEXAMPLE_PROGRESS_NOTE);
            iProgressInfo = iProgressDialog->GetProgressInfoL();
            iProgressDialog->RunLD();
            )
        if (KErrNone == error)
            {
            iProgressInfo->SetFinalValue(KProgressFinalValue);
            iProgressInfo->SetAndDraw(aPercentageCompleted);
            }
        else
            {
            delete iProgressDialog;
            iProgressDialog = NULL;
            }
        }
    else
        {
        iProgressInfo->SetAndDraw(aPercentageCompleted);
        }
    }

/** Progress callback.

This calls the observing class when the current operation is finished.

@param aError The error if the operation failed, or KErrNone if successful. */
void CCalendarAPIexampleEngine::Completed(TInt )
    {
    if (iCreatingEntryView)
        {       
        if( iWait->IsStarted() )
            {
            iWait->AsyncStop();
            }
        else
            {
            Panic(EWaitNotStarted);
            }
        iCreatingEntryView = EFalse;        
        }
        
    if (iProgressDialog)
        {
        iProgressInfo->SetAndDraw(100);
        TRAPD
            (
            error,
            iProgressDialog->ProcessFinishedL();
            )
        iProgressInfo = NULL;
        iProgressDialog = NULL;
        }
    }

/** Asks the observing class whether progress callbacks are required.

@return If the observing class returns EFalse, then the Progress() function 
        will not be called. */
TBool CCalendarAPIexampleEngine::NotifyProgress()
    {       
    return ETrue;
    }
/****************END OF OBSERVER functions***********************/


/************************HELPER functions***********************/

// ----------------------------------------------------
// CCalendarAPIexampleEngine::AddAnniversaryL()
// adds an anniversary to the agenda file
// ----------------------------------------------------
//  
//A Helper, only called from SaveL
//
void CCalendarAPIexampleEngine::AddAnniversaryL(CCalEntry* aAnniv) 
    {
    if (!iEntryView)
        {
        iEntryView = CCalEntryView::NewL(*iCalSession, *this); 
        iCreatingEntryView = ETrue;
        
        //After scheduler starting, the call back will be 
        //called when the task is completed.
        if( !iWait->IsStarted() )
            {
            iWait->Start();
            }
        else
            {
            Panic(EWaitAlreadyStarted);
            }
        }

    RPointerArray<CCalEntry> calEntryList;
    calEntryList.Append(aAnniv);
    TInt num;

    iEntryView->StoreL(calEntryList, num);
    
    calEntryList.Reset();
    
    }

// ----------------------------------------------------
// CCalendarAPIexampleEngine::UpdateAnniversaryL()
// updates an anniversary
// ----------------------------------------------------
//  A helper, only called from SaveL
//
void CCalendarAPIexampleEngine::UpdateAnniversaryL( CCalEntry* aAnniv) 
    {
    if (!iEntryView)
        {
        iEntryView = CCalEntryView::NewL(*iCalSession, *this); 
        iCreatingEntryView = ETrue;
        
        //After scheduler starting, the call back will be called when the task is completed.
        //CActiveScheduler::Start();
        if( !iWait->IsStarted() )
            {
            iWait->Start();
            }
        else
            {
            Panic(EWaitAlreadyStarted);
            }
        }
    RPointerArray<CCalEntry> calEntryList;
    calEntryList.Append(aAnniv);
    TInt num;

    iEntryView->UpdateL(calEntryList, num);

    calEntryList.Reset();
    }

/************MCalendarEngineCommandsInterface functions***********************/

// ----------------------------------------------------
// CCalendarAPIexampleEngine::DoSearchL()
// Searches all anniversaries in given time interval.
// ----------------------------------------------------
// Called from CCalendarAPIexampleSearchView::DoSearchL()
// and also from DoSaveL if anniversaries are modified
//
void CCalendarAPIexampleEngine::DoSearchL(const TSearchType& aType)
    {
    iEntries.ResetAndDestroy(); 

    iCurrentSearchType = aType;
    TTime now;
    now.HomeTime();
    TDateTime from = now.DateTime();
    User::LeaveIfError(from.SetHour(0));
    User::LeaveIfError(from.SetMinute(0));
    User::LeaveIfError(from.SetSecond(0));
    User::LeaveIfError(from.SetMicroSecond(0));

    //set the time interval
    switch (aType)
        {
        case EWeek:
            now += TTimeIntervalDays(7);
            break;
        case EMonth:
            now += TTimeIntervalMonths(1);
            break;
        case ESixMonths:
            now += TTimeIntervalMonths(6);
            break;
        case EYear:
            now += TTimeIntervalYears(1);
            break;
        }

    TDateTime to = now.DateTime();

    if (!iInstanceView)
        {
        iInstanceView = CCalInstanceView::NewL(*iCalSession, *this); 
        iCreatingEntryView = ETrue;
        
        //After scheduler starting, the call back will be 
        //called when the task is completed.
        if( !iWait->IsStarted() )
            {
            iWait->Start();
            }
        else
            {
            Panic(EWaitAlreadyStarted);
            }
        }

    RPointerArray<CCalInstance> instanceArray;
    TTime nextDate(from);
    // decrease 'from' date by one day, so that 'from' date 
    //will be included in search.
    nextDate -= TTimeIntervalDays(1);
    TTime toDate(to);
    TCalTime fromCalTime;
    fromCalTime.SetTimeLocalL(from);
    TCalTime toCalTime;
    toCalTime.SetTimeLocalL(to);
    CalCommon::TCalTimeRange timeRange(fromCalTime, toCalTime);
    
    iInstanceView->FindInstanceL(instanceArray, CalCommon::EIncludeAnnivs, timeRange);

    //create new CCalHelperEntry for each CCalEntry.
    for (TInt i = 0; i < instanceArray.Count(); i++)
        {
        CCalHelperEntry* entry = CCalHelperEntry::NewLC(instanceArray[i]);
        User::LeaveIfError(iEntries.Append(entry));
        CleanupStack::Pop(entry);
        }
    //now all the instances are in iEntries List
    
    instanceArray.Reset();
    }

// ----------------------------------------------------
// CCalendarAPIexampleEngine::DeleteEntryL()
// deletes an anniversary from the agenda file
//
// Called from CCalendarAPIexampleEntriesView when
// the delete command has been selected
// And from CCalendarAPIexampleEntryView
// ----------------------------------------------------

void CCalendarAPIexampleEngine::DeleteEntryL(const TInt& aIndex)
    {
    //This is the case when user is adding an entry and desides
    //to cancel the addition
    if( aIndex == -1)
        {
        delete iEntry;
        iEntry = NULL;
        return;
        }
        
    if (aIndex < 0 || aIndex >= iEntries.Count())
        {
        Panic(KInvalidEntryIndex);
        }

    CCalHelperEntry* entry = iEntries[aIndex];
    
    CCalEntry* anniv = entry->Anniv();

    if (!iEntryView)
        {
        iEntryView = CCalEntryView::NewL(*iCalSession, *this); 
        iCreatingEntryView = ETrue;
        
        //After scheduler starting, the call back will be called 
        //when the task is completed.
        //CActiveScheduler::Start();
        if( !iWait->IsStarted() )
            {
            iWait->Start();
            }
        else
            {
            Panic(EWaitAlreadyStarted);
            }
        }

    iEntryView->DeleteL(*anniv);
        
    iEntries.Remove(aIndex);

    delete entry; //now the CCalinstance is deleted if there's one
    }

// ----------------------------------------------------
// CCalendarAPIexampleSearchView::DoAddL()
// Resets models modify index, so that next call to 
// models EntryForModifycation will create a new entry.
// Activates the entry view.
//
// Called from CCalendarAPIexampleSearchView when
// the menu option for adding has been selected
// ----------------------------------------------------
//      
void CCalendarAPIexampleEngine::DoAddL()
    {
    //Reset the modify index
    iModifyIndex = KUndefinedModifyIndex;
    
    iAppUi.ActivateView(KEntryViewId);  
    }

// ----------------------------------------------------
// CCalendarAPIexampleEngine::SaveL()
// Adds or updates the current entry to the agenda file.
//
// Called from CCalendarAPIexampleEntryView when the
// view is closed.
// ---------------------------------------------------- 
void CCalendarAPIexampleEngine::DoSaveL()
    {
    //If iEntry is created in CreateEntryForModificationL then
    //there's a new anniversary being added. 
    //add a new anniversary
    
    if (iEntry) //this has been created in CreateEntryForModificationL
                //and the data has been stored by a call to iEntry.SetValuesL
        {
        if (iEntry->Modified())
            {
            iEntry->SaveValuesL();
            
            CCalEntry* anniv = iEntry->Anniv();

            if (!iEntryView)
                {
                iEntryView = CCalEntryView::NewL(*iCalSession, *this); 
                iCreatingEntryView = ETrue;
                
                //After scheduler starting, the call back will be called 
                //when the task is completed.
                if( !iWait->IsStarted() )
                    {
                    iWait->Start();
                    }
                else
                    {
                    Panic(EWaitAlreadyStarted);
                    }
                }

            RPointerArray<CCalEntry> calEntryList;
            calEntryList.Append(anniv);
            TInt num;

            iEntryView->StoreL(calEntryList, num);
            
            calEntryList.Reset();
            
            }
        }
    //modify existing anniversary
    else
        {
        if (0 > iModifyIndex || iEntries.Count() <= iModifyIndex)
            {
            Panic(KInvalidModifyIndex);
            }
        //Get the entry which is being modified
        CCalHelperEntry* entry = iEntries[iModifyIndex];

        if (entry->Modified()) //are there any changes
            {
            if (entry->DateHasChanged())
                {                
                CCalEntry* updateAnniv = entry->Anniv();
                TTime startDate(entry->Date());
                TCalTime time;
                time.SetTimeLocalL(startDate);
                updateAnniv->SetStartAndEndTimeL(time,time);
                UpdateAnniversaryL(updateAnniv);
                }
            //the date of the entry hasn't changed, the entry is just updated.
            else 
                {
                entry->SaveValuesL();
                CCalEntry* updateAnniv = entry->Anniv();
                UpdateAnniversaryL(updateAnniv);
                }
            DoSearchL(iCurrentSearchType);
            }
        }
    }   
     
// ----------------------------------------------------
// CCalendarAPIexampleEngine::SetModifyIndex()
// Sets the modify index. Panics with KInvalidModifyIndex
// if parameter aIndex is negative or greater than the
// number of entries.
//
// Called only from CCalendarAPIexampleEntriesView when the
// Edit command has been selected
// ----------------------------------------------------
//
void CCalendarAPIexampleEngine::SetModifyIndex(const TInt& aIndex)
    {
    if (aIndex < 0 || aIndex >= iEntries.Count())
        {
        Panic(KInvalidModifyIndex);
        }

    iModifyIndex = aIndex;
    }     

// ----------------------------------------------------
// CCalendarAPIexampleEngine::CreateEntryForModificationL()
// Returns an entry for modification. Creates a new
// entry if iModifyIndex hasn't been set. Panics with
// KInvalidModifyIndex if iModifyIndex is greater than
// the number of entries.
//
// Called from CCalendarAPIexampleEntryView when
// the container is created. 
// ----------------------------------------------------
//

void CCalendarAPIexampleEngine::CreateEntryForModificationL(TBool& aModify)
    {
    delete iEntry;
    iEntry = NULL;

    //No selected entry, create a new one for adding an entry.
    if (iModifyIndex < 0)
        {
        aModify = EFalse;
        CCalEntry* nulli = NULL;
        
        iEntry = CCalHelperEntry::NewL(nulli);
        }
    else
        {
        aModify = ETrue;
        }

    if (iModifyIndex >= iEntries.Count())
        {
        Panic(KInvalidModifyIndex);
        }
    }
    
TInt CCalendarAPIexampleEngine::EntryCount() const
    {
    return iEntries.Count();
    }   

TBool CCalendarAPIexampleEngine::SetValuesToNewEntry( const TDesC& aName,
                        const TDateTime& aDate,
                        const TBool& aAlarm,
                        const TDateTime& aAlarmTime,
                        const TInt& aSynchronizationMethod)

    {
    CCalHelperEntry *entry=NULL;
    if( iEntry )
        {
        entry = iEntry;
        }
        
    else 
       entry = iEntries[iModifyIndex];    
    
    return entry->SetValues(aName,aDate,aAlarm,aAlarmTime,aSynchronizationMethod);
       
    }

void CCalendarAPIexampleEngine::GetValuesToSet(TDes& aName, TTime& aDate,
                   TBool& aAlarm, TTime& aAlarmTime,
                            TInt& aSync)
    {
    CCalHelperEntry *entry=NULL;
    if( iEntry )
        {
        entry = iEntry;
        }        
    else 
        entry = iEntries[iModifyIndex];


    aName = entry->Name();
    aDate = entry->Date();
    aAlarm = entry->Alarm();
    aAlarmTime = entry->AlarmTime();
    aSync = entry->SynchronizationMethod();    
    }

// ----------------------------------------------------
// CCalendarAPIexampleEngine::Entry()
// returns a reference to an entry for reading the values
// of the entry. Panics with EOutOfEntriesArray if parameter
// aIndex is negative or greater than the number of entries
//
// Called ONLY from CCalendarAPIexampleEntriesContainer when
// populating the list box
// ----------------------------------------------------
//
CCalHelperEntry& CCalendarAPIexampleEngine::Entry(const TInt& aIndex)
    {
    if (0 > aIndex || iEntries.Count() <= aIndex)
        {
        Panic(EOutOfEntriesArray);
        }

    return *iEntries[aIndex];
    }


// ----------------------------------------------------
// CCalendarAPIexampleEngine::ModifyIndex()
// Returns the modify index.
//
// Called only from CCalendarAPIexampleEntryView
// When deleting the entry.
// ----------------------------------------------------
//

TInt CCalendarAPIexampleEngine::ModifyIndex() const
    {
    return iModifyIndex;
    }
    
void CCalendarAPIexampleEngine::ExecuteDeletionL()
    {       
    DeleteEntryL(ModifyIndex());
    
    // If no more entries exist in the listbox, activate search view.
    if (EntryCount() == 0)
        {
        iAppUi.ActivateView(KSearchViewId);
        }
    // activate entries view
    else
        {
        iAppUi.ActivateView(KEntriesViewId);
        }    
    }
/*******END OF CalendarEngineCommandsInterface functions**********************/

// End of file
