/*
 * Copyright  2008 Nokia Corporation.
 */


// ----------------------------------------------------
//      CCalHelperEntry - a wrapper class for CCalEntry
// ----------------------------------------------------

// ================= MEMBER FUNCTIONS =======================

#include "CalendarHelperEntry.h"

#include <e32math.h>
#include <calalarm.h>
#include <CalInstance.h>
#include "CalendarAPIexample.pan"

#include <calentry.h>

// CONSTANTS
const TInt KMaxSync = 2;
const TInt KDefaultAlarmHour = 8;
const TInt KDefaultAnniversaryHour = 8;
const TInt KMinutesInHour = 60;
const TInt KUndefinedModifyIndex = -1;
const TInt KGuidLength = 30;

// Two-phased constructor.
CCalHelperEntry* CCalHelperEntry::NewL(CCalEntry* aAnniv)
    {
    CCalHelperEntry* self = CCalHelperEntry::NewLC(aAnniv);
    CleanupStack::Pop(self);
    return self;
    }

// Two-phased constructor.
CCalHelperEntry* CCalHelperEntry::NewLC(CCalEntry* aAnniv)
    {
    CCalHelperEntry* self = new (ELeave) CCalHelperEntry;
    CleanupStack::PushL(self);
    self->ConstructL(aAnniv);
    return self;
    }

// Two-phased constructor.
CCalHelperEntry* CCalHelperEntry::NewL(CCalInstance* aAnniv)
    {
    CCalHelperEntry* self = CCalHelperEntry::NewLC(aAnniv);
    CleanupStack::Pop(self);
    return self;
    }

// Two-phased constructor.
CCalHelperEntry* CCalHelperEntry::NewLC(CCalInstance* aAnniv)
    {
    CCalHelperEntry* self = new (ELeave) CCalHelperEntry;
    CleanupStack::PushL(self);
    self->ConstructL(aAnniv);
    return self;
    }


// ----------------------------------------------------
// Standard Symbian OS 2nd phase constructor
// Will initialize entry with values of parameter aAnniv,
// if parameter aAnniv is provided (not NULL).
// ----------------------------------------------------
//
void CCalHelperEntry::ConstructL(CCalInstance* aAnniv)
    {
    iInstance = aAnniv;
    ConstructL(&aAnniv->Entry());
    }

void CCalHelperEntry::ConstructL(CCalEntry* aAnniv)
    {
    // initialize today's date for anniversary date and alarm.
    TTime now;
    now.HomeTime();
    iDate = now.DateTime();
    
    if ( aAnniv )
        {            
        TTime starttime = aAnniv->StartTimeL().TimeLocalL();
        TDateTime time2 = starttime.DateTime();

        TTime endtime = aAnniv->StartTimeL().TimeLocalL();
        TDateTime time3 = endtime.DateTime();
        }
    
    iDate.SetHour(KDefaultAnniversaryHour);
    iDate.SetMinute(0);
    iDate.SetSecond(0);
    iDate.SetMicroSecond(0);
    
    iAlarmTime = now.DateTime();
    User::LeaveIfError(iAlarmTime.SetHour(KDefaultAlarmHour));
    User::LeaveIfError(iAlarmTime.SetMinute(0));
    User::LeaveIfError(iAlarmTime.SetSecond(0));
    User::LeaveIfError(iAlarmTime.SetMicroSecond(0));
    iAlarmDays = TTimeIntervalDays(0);
    
    iAlarmMinutes = TTimeIntervalMinutes(KDefaultAlarmHour*KMinutesInHour);

    if (!aAnniv)
        {
        return;
        }

    iAnniv = aAnniv;

    // read the name for the anniversary.
    TBuf<KMaxNameLength> name;
    name = aAnniv->SummaryL();
    iName = name;

    // read the date for the anniversary.
    iDate = aAnniv->StartTimeL().TimeLocalL().DateTime();

    iAlarmPtr = iAnniv->AlarmL();
    
    iAlarm = iAlarmPtr ? ETrue : EFalse;
    if (iAlarm)
    {

        iAlarmMinutes = -iAlarmPtr->TimeOffset().Int();
        iAlarmTime = iDate;
        TTime alarm = iAlarmTime;
        alarm += iAlarmMinutes;
        iAlarmTime = alarm.DateTime();
    }

    // read how the anniversary is to be synchronized.
    CCalEntry::TReplicationStatus annivReplStatus =
        aAnniv->ReplicationStatusL();
    if (annivReplStatus == CCalEntry::EOpen)
        {
        SetSynchronizationMethod(KSyncPublic);
        }
    else if (annivReplStatus == CCalEntry::EPrivate)
        {
        SetSynchronizationMethod(KSyncPrivate);
        }
    else
        {
        SetSynchronizationMethod(KSyncNo);
        }

    iModified = EFalse;
    }

// constructor
CCalHelperEntry::CCalHelperEntry()
    {
    }
    
// destructor
CCalHelperEntry::~CCalHelperEntry()
    {
    //If there's a CCalInstance then delete it. 
    //It deletes the corresponding CCalEntry object
    if( iInstance )   
        {
        delete iInstance;
        iInstance = NULL;
        }
    else //there only the CCalEntry
        {
        if (iAnniv)
            delete iAnniv;
        iAnniv=NULL;
        }
    if (iAlarmPtr)
        delete iAlarmPtr;
    iAlarmPtr=NULL;
    }

// ----------------------------------------------------
// CCalHelperEntry::Anniv()
// Returns a pointer to the CCalEntry member variable of
// the entry.
// ----------------------------------------------------
//
CCalEntry* CCalHelperEntry::Anniv()
    {
    CCalEntry* anniv = iAnniv;
    return anniv;
    }

// ----------------------------------------------------
// CCalHelperEntry::Name()
// Returns the name of the entry.
// ----------------------------------------------------
//
TBuf<KMaxNameLength> CCalHelperEntry::Name() const
    {
    return iName;
    }

// ----------------------------------------------------
// CCalHelperEntry::Date()
// Returns the date of the entry.
// ----------------------------------------------------
//
TDateTime CCalHelperEntry::Date() const
    {
    return iDate;
    }

// ----------------------------------------------------
// CCalHelperEntry::Alarm()
// Returns the alarm status of the entry.
// ----------------------------------------------------
//
TBool CCalHelperEntry::Alarm() const
    {
    return iAlarm;
    }

// ----------------------------------------------------
// CCalHelperEntry::AlarmTime()
// Returns the alarm time of the entry.
// ----------------------------------------------------
//
TDateTime CCalHelperEntry::AlarmTime() const
    {
    return iAlarmTime;
    }

// ----------------------------------------------------
// CCalHelperEntry::SynchronizationMethod()
// Returns the syncronization method of the entry.
// ----------------------------------------------------
//
TInt CCalHelperEntry::SynchronizationMethod() const
    {
    return iSynchronizationMethod;
    }

// ----------------------------------------------------
// CCalHelperEntry::Modified()
// Returns whether entry has been modified or not.
// ----------------------------------------------------
//
TBool CCalHelperEntry::Modified() const
    {
    return iModified;
    }

// ----------------------------------------------------
// CCalHelperEntry::DateHasChanged()
// Returns whether the date of the entry has been
// modified or not.
// ----------------------------------------------------
//
TBool CCalHelperEntry::DateHasChanged() const
    {
    return iDateModified;
    }

// ----------------------------------------------------
// CCalHelperEntry::ResetTimeL()
// Sets given dates time to midnight.
// ----------------------------------------------------
//
void CCalHelperEntry::ResetTimeL(TDateTime& aDate) const
    {
    User::LeaveIfError(aDate.SetHour(0));
    User::LeaveIfError(aDate.SetMinute(0));
    User::LeaveIfError(aDate.SetSecond(0));
    User::LeaveIfError(aDate.SetMicroSecond(0));
    }

// ----------------------------------------------------
// CCalHelperEntry::GetDaysAndMinutes()
// Calculates how many days and minutes aTime is from
// aFromTime.
// ----------------------------------------------------
//
void CCalHelperEntry::GetDaysAndMinutesL(const TDateTime& aTime,
                                const TDateTime& aFromTime,
                                TTimeIntervalDays& aDays,
                                TTimeIntervalMinutes& aMinutes) const
    {
    // Create modifiable TDateTime (aTime cannot be modified, because it's const).
    TDateTime modifiableTime = aTime;
    // Reset the modifiableTime. This will set the dates time to midnight.
    ResetTimeL(modifiableTime);
    // Create new TTime object. This is a reset aTime in TTime format. (modifiableTime has no more use)
    TTime resetTime(modifiableTime);

    // Create modifiable TDateTime (aFromTime cannot be modified, because it's const).
    TDateTime modifiableFromTime = aFromTime;
    // Reset the modifiableTime. This will set the dates time to midnight.
    ResetTimeL(modifiableFromTime);
    // Create new TTime object. This is a reset aFromTime in TTime format. (modifiableFromTime has no more use)
    TTime resetFromTime(modifiableFromTime);

    // Create a TTime object from aTime that is not reset.
    TTime unresetTime(aTime);

    // calculate the difference of days between aTime and aFromTime when their
    // time is reset and only dates are different.
    aDays = resetFromTime.DaysFrom(resetTime);

    TTimeIntervalMinutes minutes;
    // calucate the difference of minutes between aTime and aFromTime. This is
    // done by just comparing the time of aTime to midnight.
    User::LeaveIfError(unresetTime.MinutesFrom(resetTime, minutes));

    aMinutes = minutes;
    }

// ----------------------------------------------------
// CCalHelperEntry::SetValuesL()
// Sets given values to entry. Return true if values
// are valid and false if not.
// ----------------------------------------------------
// This is called by CCalendarAPIexampleEntryItemList
// SaveL once the modifications are done using
// CCalendarAPIexampleEntryView

TBool CCalHelperEntry::SetValues(   const TDesC& aName,
                            const TDateTime& aDate,
                            const TBool& aAlarm,
                            const TDateTime& aAlarmTime,
                            const TInt& aSync)
    {
    SetDate(aDate); //sets the hours to 8
    SetName(aName);
    SetSynchronizationMethod(aSync);
    SetAlarm(aAlarm);    

    if (aAlarm)
        {
        // Event time from ENTRY
        TTime startTime = iDate;

        // Alarm cannot occur after the entry date.        
        // We have set the anniversary time to 8:00 but actually ignore it
        
        // Alarm time from FORM
        TDateTime alarmDay = iDate;
        //This doesn't work in the last day of month!
        //alarmDay.SetDay( alarmDay.Day() + 1 ); 
        
        alarmDay.SetHour(0);
        alarmDay.SetMinute(0);
        alarmDay.SetMicroSecond(0);
        
        //TDateTime theNextDay = alarmDay 
        TTimeIntervalDays oneDay(1);
        TTime time = alarmDay;
        time += oneDay;
        //oneDay
        alarmDay = time.DateTime();
        
        
        TTime alarmTime = aAlarmTime;
        TTimeIntervalMinutes minutes;
        alarmTime.MinutesFrom( alarmDay, minutes );
        
        if (minutes.Int() > 0)            
            {
            return EFalse;
            }
                        
        //Now we should check that the alarm time is not in the past:
        TTime now;
        now.HomeTime();
        //iDate = now.DateTime();
        now.MinutesFrom(alarmTime, minutes);
        
        if (minutes.Int() > 0)            
            {
            return EFalse;
            }
        
        TTime date = iDate;
        alarmTime = aAlarmTime;
        alarmTime.MinutesFrom(date, minutes);
        
        iAlarmTime = aAlarmTime;
        iAlarmMinutes = minutes;
        
        iModified = ETrue;
        }

    return ETrue;
    }

// ----------------------------------------------------
// CCalHelperEntry::SetName()
// Sets given name to entry. If name has changed,
// iModified is set to true;
// ----------------------------------------------------
//
void CCalHelperEntry::SetName(const TDesC& aName)
    {
    if (aName != iName)
        {
        iName = aName;
        iModified = ETrue;
        }
    }

// ----------------------------------------------------
// CCalHelperEntry::SetDate()
// Sets given date to entry. If date has changed,
// iModified is set to true;
// ----------------------------------------------------
//
void CCalHelperEntry::SetDate(const TDateTime& aDate)
    {
    // TDateTime cannot be used for comparison, local TTime variables needed.
    TTime currentDate(iDate);
    TTime newDate(aDate);
    if (newDate != currentDate)
        {
        iDate = aDate;
        
        iDate.SetHour(KDefaultAnniversaryHour); 
        
        iModified = ETrue;
        iDateModified = ETrue;
        }
    }

// ----------------------------------------------------
// CCalHelperEntry::SetAlarm()
// Sets the alarm state of the entry. If name alarm state
// changed, iModified is set to true.
// ----------------------------------------------------
//
void CCalHelperEntry::SetAlarm(const TBool& aAlarm)
    {
    if (aAlarm != iAlarm)
        {
        iAlarm = aAlarm;
        
        iModified = ETrue;
        }
    }

// ----------------------------------------------------
// CCalHelperEntry::SetSynchronizationMethod()
// Sets the synchronization state of the entry. If state
// has changed, iModified is set to true. Synchronization
// method defines how the anniversary is synchronized
// e.g. with PC.
// ----------------------------------------------------
//
void CCalHelperEntry::SetSynchronizationMethod(const TInt& aSynchronizationMethod)
    {
    if (aSynchronizationMethod != iSynchronizationMethod)
        {
        iSynchronizationMethod = aSynchronizationMethod;
        iModified = ETrue;
        }
    }

// ----------------------------------------------------
// CCalHelperEntry::CreateAnnivL()
// Creates a new CCalEntry object and initializes it with
// the date of the entry.
// ----------------------------------------------------
//
CCalEntry* CCalHelperEntry::CreateAnnivL()
    {
    // Create unique ID.
    TTime now;
    now.HomeTime();
    TInt64 seed = now.Int64();
    TInt randNum = Math::Rand( seed );
    HBufC8* guid = HBufC8::NewLC(KGuidLength);
    guid->Des().Num(  randNum );
    
    //CCalEntry::EMethodNone means that there's no group scheduling
    CCalEntry* anniv = CCalEntry::NewL( CCalEntry::EAnniv, guid,
                                        CCalEntry::EMethodNone, 0 );
    
    CleanupStack::Pop( guid );
                                            
    CleanupStack::PushL(anniv);
    TTime startDate(iDate);
    TCalTime time;
    time.SetTimeLocalL(startDate);
    anniv->SetStartAndEndTimeL(time, time);

    if (iAlarm) //if there is a alarm
    {   
        delete iAlarmPtr;
        iAlarmPtr=NULL;
        iAlarmPtr = CCalAlarm::NewL(); 

        iAlarmPtr->SetTimeOffset(-iAlarmMinutes.Int());
        anniv->SetAlarmL(iAlarmPtr);
    }

    CleanupStack::Pop(anniv);

    return anniv;
    }



// ----------------------------------------------------
// CCalHelperEntry::SaveValuesL()
// Sets the values of the entry to its member CCalEntry object.
// If member CCalEntry object doesn't exist, it is created.
// ----------------------------------------------------
// This is called by DoSaveL of the engine.
void CCalHelperEntry::SaveValuesL()
    {
    TBool created = EFalse;
    if (iModified)
        {
        if (!iAnniv) //This is the case when creating a new entry
            {        //The values are gotten from CCalendarAPIexampleEntryContainer
                     // and CCalendarAPIexampleEntryView.
                     // The engine has called this 
                     // so there's a iEntry created in the engine.
                     // (created by a call to CreateEntryForModificationL)
                     // now we need store the given data.
                     
            //the created CCalHelperEntry goes to engines iEntry.iAnniv
            iAnniv = CreateAnnivL(); 
            
            created = ETrue;
            }
        
        SaveValuesToAnnivL(iAnniv);
        
        if( !created ) 
            {
            delete iAlarmPtr;
            iAlarmPtr = NULL;
            
            if( !iAlarm )
                {
                //remove the alarm
                iAnniv->SetAlarmL(NULL);
                }
            else
                {
                //Create a new alarm
                iAlarmPtr = CCalAlarm::NewL();
                
                iAlarmPtr->SetTimeOffset(-iAlarmMinutes.Int());
                
                iAnniv->SetAlarmL(iAlarmPtr);
                }
            }
        
        iModified = EFalse;
        iDateModified = EFalse;
        
        }
    }

// ----------------------------------------------------
// CCalHelperEntry::SaveValuesToAnnivL()
// Sets the values of the entry to given CCalEntry object.
// ----------------------------------------------------
//  
void CCalHelperEntry::SaveValuesToAnnivL(CCalEntry* aAnniv)
    {
    aAnniv->SetSummaryL(iName);
    
    CCalEntry::TReplicationStatus sync;
    switch (iSynchronizationMethod)
        {
        case KSyncPublic:
            sync=CCalEntry::EOpen;
            break;
        case KSyncPrivate:
            sync=CCalEntry::EPrivate;
            break;
        case KSyncNo:
            sync=CCalEntry::ERestricted;
            break;
        default:
            Panic(KInvalidSyncValue);
            break;
        }
    aAnniv->SetReplicationStatusL(sync);
    
    }

// ----------------------------------------------------
// CCalHelperEntry::NewAnnivLC()
// Creates a new CCalEntry object and initializes it
// with the data of the entry.
// ----------------------------------------------------
//
CCalEntry* CCalHelperEntry::NewAnnivLC()
    {
    CCalEntry* anniv = CreateAnnivL();
    CleanupStack::PushL(anniv);
    SaveValuesToAnnivL(anniv);
    return anniv;
    }

// End of File
