/*
 * Copyright (c) 2009 Nokia Corporation.
 */

#include <aknnotewrappers.h>
#include <coemain.h>
#include <eikedwin.h>
#include <gdi.h>
#include <txtrich.h>
#include "Client.pan"
#include "ClientAppView.h"
#include "ClientApplication.h"
#include "ClientEngine.h"
#include "HTTPClientExample_sc.hlp.hrh"

// Constants for CEikRichTextEditors
const TInt KNumberOfControls = 2;
const TInt KNumberOfLines = 0;
const TInt KTextLimit = 128;

const TInt KBorderThickness = 4;
const TInt KSizeOffset = KBorderThickness * 2;

// Background color
#define KBackgroundColor    TRgb(128, 128, 128)
#define KBorderColor        TRgb(180, 0, 0)

// ----------------------------------------------------------------------------
// CClientAppView::NewL()
//
// Creates instance of CClientAppView.
// ----------------------------------------------------------------------------
CClientAppView* CClientAppView::NewL(const TRect& aRect)
  {
    CClientAppView* self = CClientAppView::NewLC(aRect);
    CleanupStack::Pop(self);
    return self;
  }

// ----------------------------------------------------------------------------
// CClientAppView::NewLC()
//
// Creates instance of CClientAppView.
// ----------------------------------------------------------------------------
CClientAppView* CClientAppView::NewLC(const TRect& aRect)
  {
    CClientAppView* self = new (ELeave) CClientAppView;
    CleanupStack::PushL(self);
    self->ConstructL(aRect);
    return self;
  }

// ----------------------------------------------------------------------------
// CClientAppView::CClientAppView()
//
// First phase construction.
// ----------------------------------------------------------------------------
CClientAppView::CClientAppView()
  {
  }

// ----------------------------------------------------------------------------
// CClientAppView::~CClientAppView()
//
// Destructor.
// ----------------------------------------------------------------------------
CClientAppView::~CClientAppView()
  {
    delete iOutputWindow;
  delete iStatusWindow;
  }

// ----------------------------------------------------------------------------
// CClientAppView::ConstructL()
//
// Second phase construction.
// ----------------------------------------------------------------------------
void CClientAppView::ConstructL(const TRect& aRect)
  {
  // Create a window for this application view
  CreateWindowL();

  // Create output window
  iOutputWindow = new (ELeave) CEikRichTextEditor();
  iOutputWindow->SetContainerWindowL(*this);
  iOutputWindow->ConstructL(this, KNumberOfLines, KTextLimit,
    CEikEdwin::EReadOnly | CEikEdwin::EAvkonDisableCursor,
    EGulFontControlAll, EGulNoSymbolFonts);
  iOutputWindow->CreateScrollBarFrameL()->SetScrollBarVisibilityL(
      CEikScrollBarFrame::EOff, CEikScrollBarFrame::EOn);

  // Create status window
  iStatusWindow = new (ELeave) CEikRichTextEditor();
  iStatusWindow->SetContainerWindowL(*this);
  iStatusWindow->ConstructL(this, KNumberOfLines, KTextLimit,
    CEikEdwin::EReadOnly | CEikEdwin::EAvkonDisableCursor,
    EGulFontControlAll, EGulNoSymbolFonts);
  iStatusWindow->CreateScrollBarFrameL()->SetScrollBarVisibilityL(
      CEikScrollBarFrame::EOff, CEikScrollBarFrame::EOn);

  iStatusWindow->SetFocus(ETrue);

  // Activate the window, which makes it ready to be drawn
  ActivateL();

  // Set the windows size
  SetRect(aRect);
  }

// ---------------------------------------------------------
// CClientAppView::SizeChanged()
// Called whenever SetExtent(), SetSize(), SetRect(),
// SetCornerAndSize(), or SetExtentToWholeScreen() are
// called on the control. NB: The window server does not
// generate size-changed events.
// ---------------------------------------------------------

void CClientAppView::SizeChanged()
    {
    TRect r = Rect();
    TInt scrollbarWidth = GetScrollbarWidth(iOutputWindow);
    TInt editorWidth = r.Width() - KSizeOffset - scrollbarWidth;
    
    TPoint upperEditPosition = TPoint(KBorderThickness, KBorderThickness);
    TInt upperEditHeight = r.Height()*3/4 - KSizeOffset;
    iOutputWindow->SetExtent(upperEditPosition, TSize(editorWidth,
        upperEditHeight));
    
    upperEditPosition.iY += upperEditHeight + KBorderThickness;
    upperEditHeight = r.Height()/4 - KBorderThickness;
    iStatusWindow->SetExtent(upperEditPosition, TSize(editorWidth,
        upperEditHeight));
    }

// -----------------------------------------------------------------------------
// CClientAppView::GetHelpContext()
// Gets the control's help context. Returns a NULL context by default.
// -----------------------------------------------------------------------------

void CClientAppView::GetHelpContext(TCoeHelpContext& aContext) const
    {
    // Get the help context for the application
    aContext.iMajor = KUidHelpFile;

    aContext.iContext = KContextApplication;
    }

// ---------------------------------------------------------
// CClientAppView::HandleResourceChange()
// Called by the framework when resource is changed.
// ---------------------------------------------------------

void CClientAppView::HandleResourceChange(TInt aType)
    {
    CCoeControl::HandleResourceChange(aType);

    // ADDED FOR SCALABLE UI SUPPORT
    // *****************************
    if ( aType==KEikDynamicLayoutVariantSwitch )
        {
        TRect rect;
        AknLayoutUtils::LayoutMetricsRect(AknLayoutUtils::EMainPane, rect);
        SetRect(rect);
        }
    }

// ----------------------------------------------------------------------------
// CClientAppView::Draw()
//
// Draw this application's view to the screen
// ----------------------------------------------------------------------------
void CClientAppView::Draw(const TRect& aRect) const
  {
    CWindowGc& gc = SystemGc();

    gc.SetPenStyle( CGraphicsContext::ENullPen );
    gc.SetBrushColor( KBackgroundColor );
    gc.SetBrushStyle( CGraphicsContext::ESolidBrush );
    gc.DrawRect( aRect );
    
    // Draw a custom border around the focused rich text editor
    gc.SetBrushColor(KBorderColor);
    if (iOutputWindow->IsFocused())
        {
        TInt scrollbarWidth = GetScrollbarWidth(iOutputWindow);
        TRect editorRect = iOutputWindow->Rect();
        editorRect.Grow(KBorderThickness + scrollbarWidth, KBorderThickness);
        editorRect.Move(scrollbarWidth, 0);
        gc.DrawRect(editorRect);
        }
        
    if (iStatusWindow->IsFocused())
        {
        TInt scrollbarWidth = GetScrollbarWidth(iStatusWindow);
        TRect editorRect = iStatusWindow->Rect();
        editorRect.Grow(KBorderThickness + scrollbarWidth, KBorderThickness);
        editorRect.Move(scrollbarWidth, 0);
        gc.DrawRect(editorRect);
        }
  }

// ----------------------------------------------------------------------------
// Returns the width (breadth) of the scrollbar for a particular editor
// ----------------------------------------------------------------------------
TInt CClientAppView::GetScrollbarWidth(CEikRichTextEditor* anEditor) const
    {
    TInt scrollbarWidth = anEditor->ScrollBarFrame()->
        ScrollBarBreadth(CEikScrollBar::EVertical);
    // TODO: If scrollbars are not drawn yet, the width remains zero. In this
    // case, an intentionally magical number is returned.
    if (scrollbarWidth == 0)
        {
        scrollbarWidth = 8;
        }
    return scrollbarWidth;
    }
    
// ----------------------------------------------------------------------------
// CClientAppView::CountComponentControls()
//
// Returns number of controls in this compound control.
// ----------------------------------------------------------------------------
TInt CClientAppView::CountComponentControls() const
  {
    return KNumberOfControls;
  }

// ----------------------------------------------------------------------------
// CClientAppView::ComponentControl()
//
// Returns pointer to control with index aIndex.
// ----------------------------------------------------------------------------
CCoeControl* CClientAppView::ComponentControl(TInt aIndex) const
  {
    switch(aIndex)
    {
    case 0:
      return iOutputWindow;
    case 1:
      return iStatusWindow;
    default:
      Panic(EClientView);
      return 0;
    }
  }

// ----------------------------------------------------------------------------
// CClientAppView::HandleControlEventL()
//
// Handles control events.
// ----------------------------------------------------------------------------
void CClientAppView::HandleControlEventL(CCoeControl* /* aControl */,
                       TCoeEvent /* aEventType */ )
  {
  }

// ----------------------------------------------------------------------------
// CClientAppView::ResetL()
//
// Resets contents of output and status window.
// ----------------------------------------------------------------------------
void CClientAppView::ResetL()
  {
    iOutputWindow->Text()->Reset();
    iOutputWindow->HandleTextChangedL();
    iOutputWindow->SetCursorPosL(0, EFalse);

    iStatusWindow->Text()->Reset();
    iStatusWindow->HandleTextChangedL();
    iStatusWindow->SetCursorPosL(0, EFalse);
  }

// ----------------------------------------------------------------------------
// CClientAppView::OfferKeyEventL()
//
// Handles key events.
// ----------------------------------------------------------------------------
TKeyResponse CClientAppView::OfferKeyEventL(const TKeyEvent& aKeyEvent,
                        TEventCode aType)
  {
  // Catch EStdKeyNkp5 and EStdKeyDevice3; they are used here to switch
  // the active CEikRichTextEditor.
  if(aType == EEventKey)
    {
    switch(aKeyEvent.iScanCode)
      {
      case EStdKeyNkp5:
      case EStdKeyDevice3:
        SwitchFocus();
        return EKeyWasConsumed;
      }
    }

  // Redirect keyevents to controls
    if (iOutputWindow)
    {
      if (iOutputWindow->IsFocused())
      return iOutputWindow->OfferKeyEventL(aKeyEvent, aType);
    }

    if (iStatusWindow)
    {
      if (iStatusWindow->IsFocused())
      return iStatusWindow->OfferKeyEventL(aKeyEvent, aType);
    }

    return EKeyWasNotConsumed;
  }

// ----------------------------------------------------------------------------
// CClientAppView::SwitchFocus()
//
// Switches focus between the two rich text editors.
// ----------------------------------------------------------------------------
void CClientAppView::SwitchFocus()
  {
  if (iOutputWindow->IsFocused())
    {
    iOutputWindow->SetFocus(EFalse);
    iStatusWindow->SetFocus(ETrue);
    }
  else
    {
    iStatusWindow->SetFocus(EFalse);
    iOutputWindow->SetFocus(ETrue);
    }
  // Update the borders
  DrawNow();
  }
  
// ----------------------------------------------------------------------------
// CClientAppView::AddToStatusWindowL()
//
// Add a single line of text to the status window.
// ----------------------------------------------------------------------------
void CClientAppView::AddToStatusWindowL(const TDesC& aText)
  {
  // Append aText to status window with line break
  CRichText* text = iStatusWindow->RichText();
  text->InsertL(text->DocumentLength(), aText);
  text->InsertL(text->DocumentLength(), CEditableText::ELineBreak);
  iStatusWindow->HandleTextChangedL();
  iStatusWindow->MoveCursorL(TCursorPosition::EFLineDown, EFalse);
  }

// ----------------------------------------------------------------------------
// CClientAppView::AddToOutputWindowL()
//
// Add text to outputwindow. No formatting is provided.
// ----------------------------------------------------------------------------
void CClientAppView::AddToOutputWindowL(const TDesC8& aText)
  {
  // Convert 8-bit aText to 16-bit and append it to end of output window
  HBufC* tempBuf = HBufC::NewL(aText.Length());
  CleanupStack::PushL(tempBuf);
  tempBuf->Des().Copy(aText);
  CRichText* text = iOutputWindow->RichText();
  text->InsertL(text->DocumentLength(), *tempBuf);
  iOutputWindow->HandleTextChangedL();

  CleanupStack::PopAndDestroy(tempBuf);
  }

// ----------------------------------------------------------------------------
// CClientAppView::ClientEvent()
//
// Called by CClientEngine to notify events to user
// ----------------------------------------------------------------------------
void CClientAppView::ClientEvent(const TDesC& aEventDescription)
  {
  TRAPD(err, AddToStatusWindowL(aEventDescription));
  if(err)
    Panic(EClientView);
  }

// ----------------------------------------------------------------------------
// CClientAppView::ClientBodyReceived()
//
// Called by CClientEngine when part of response body received
// ----------------------------------------------------------------------------
void CClientAppView::ClientBodyReceived(const TDesC8& aBodyData)
  {
  TRAPD(err, AddToOutputWindowL(aBodyData));
  if(err)
    Panic(EClientView);
  }

// end of file
