# -*- coding: utf-8 -*-

import time
import threading
import urllib
from cache import Cache
from xbmctorrent.tmdb import API_KEY

# tmdb api has 40 requests per 10 seconds limit
RATE_LIMIT = 40  # request in TIME_WINDOW
TIME_WINDOW = 10  # seconds

remaining = RATE_LIMIT
start_window = 0.0
lock = threading.Lock()


def url_get(url, params={}, headers={}):
    import urllib2
    from contextlib import closing
    from xbmctorrent.common import USER_AGENT
    global remaining, start_window

    if params:
        import urllib
        url = "%s?%s" % (url, urllib.urlencode(params))

    req = urllib2.Request(url)
    req.add_header("User-Agent", USER_AGENT)
    for k, v in headers.items():
        req.add_header(k, v)

    try:
        with lock:
            if start_window == 0.0:
                start_window = time.clock()
            if remaining < 2:  # 1 left just for safety
                wait = TIME_WINDOW - (time.clock() - start_window)
                if wait > 0:
                    time.sleep(wait + 0.2)  # 0.2 for safe reset
                remaining = RATE_LIMIT
                start_window = time.clock()
            remaining -= 1

        with closing(urllib2.urlopen(req)) as response:
            data = response.read()
            if response.headers.get("Content-Encoding", "") == "gzip":
                import zlib
                return zlib.decompressobj(16 + zlib.MAX_WBITS).decompress(data)
            return data
    except urllib2.HTTPError as e:
        if e.code == 429:
            print("[tmdb]: rate limit is reached.")
        return None


def url_get_json(*args, **kwargs):
    import json
    data = url_get(*args, **kwargs)
    return data and json.loads(data) or {}


class TmDb:
    """

    API:
        scraper  - скрапер
        search   - поиск фильмов
        movie    - профайл фильма

    """

    def __init__(self):
        self.api_key = API_KEY  # 'f7f51775877e0bb6703520952b3c7840'
        self.base_url = "http://api.themoviedb.org/3"
        self.cache = Cache('tmdb.db', 1.1)
        self.params = {
            "api_key": self.api_key,
            "language": "ru"
        }
        self.headers = {
            'Referer': self.base_url
        }

    # API
    def scraper(self, search, year):
        try:
            if not isinstance(search, list):
                search = [search]
            tag = 'scraper:' + urllib.quote_plus(":".join(search).encode('utf8'))
        except:
            return None
        else:
            if year:
                tag += ':' + str(year)

            id = self.cache.get(tag, self._scraper, search, year)
            if not id:
                return None

            return self.movie(id)

    def search(self, name):
        return self._search(name)

    def movie(self, id):
        id = str(id)
        return self.cache.get('movie:' + id, self._movie, id)

    def _movie(self, imdb_id):
        from xbmctorrent.utils import first

        meta = url_get_json("%s/movie/%s" % (self.base_url, imdb_id), params=self.params, headers=self.headers) or {}

        if not meta:
            return False, None

        def img(key, size="original", default=""):
            return meta.get(key) and self._image(meta[key], size=size) or default

        def m(key, default=""):
            return meta.get(key) or default

        def m_crew(job):
            return first([crew["name"] for crew in (m("credits", default={}).get("crew") or []) if crew["job"] == job])

        def get_studio():
            return (first(sorted(m("production_companies") or [], key=lambda x: x["id"])) or {}).get("name") or ""

        res = {
            "icon": img("poster_path", size="w500"),
            "thumbnail": img("poster_path", size="w500"),
            "info": {
                "count": m("id"),
                "title": m("title"),
                "originaltitle": m("original_title"),
                "genre": meta.get("genres") and ", ".join([genre["name"] for genre in meta["genres"]]) or "",
                "plot": m("overview"),
                "plot_outline": m("overview"),
                "tagline": m("tagline"),
                "rating": m("vote_average"),
                "duration": m("runtime"),
                "code": m("imdb_id"),
                "cast": [cast["name"] for cast in (m("credits", default={}).get("cast") or [])],
                "director": m_crew("Director"),
                "writer": m_crew("Writer") or m_crew("Novel") or m_crew("Screenplay"),
                "studio": get_studio(),
                "year": meta.get("release_date") and meta["release_date"].split("-")[0] or 0,
            },
            "properties": {
                "fanart_image": img("backdrop_path"),
            },
        }

        timeout = True

        # если фильм свежий, то кладем в кэш НЕ на долго (могут быть обновления на сайте)
        if 'year' not in res['info'] or int(res['info']['year']) >= time.gmtime(time.time()).tm_year:
            timeout = 7 * 24 * 60 * 60  # week

        return timeout, res

    def _movie_clear(self, dirname):
        import os
        for filename in os.listdir(dirname):
            try:
                os.unlink(os.path.join(dirname, filename))
            except:
                raise
        try:
            os.rmdir(dirname)
        except:
            raise

    def _search(self, search, year=None):
        params = self.params
        if not isinstance(search, list):
            search = [search]

        if year:
            params["year"] = year

        results = None
        for name in search:
            params["query"] = name.encode("utf-8")
            result = url_get_json("%s/search/movie" % self.base_url, params=params, headers=self.headers)
            if not result:
                continue
            results = result.get("results")
            if results:
                break

        if not results:
            return None

        return results[0].get("id")

    def _scraper(self, search, year):
        timeout = True

        # если фильм свежий, то кладем в кэш НЕ на долго (могут быть обновления на сайте)
        if year and year >= time.gmtime(time.time()).tm_year:
            timeout = 7 * 24 * 60 * 60  # week

        tbdbid = self._search(search, year)

        if tbdbid is None:
            return False, None

        elif tbdbid == "":
            # сохраняем пустой результат на 3-е суток
            return 259200, None

        else:
            return timeout, tbdbid

    def config(self):
        return self.cache.get('config:tmdb', self._config)

    def _config(self):
        timeout = 24 * 60 * 60  # one day
        return timeout, url_get_json("%s/configuration" % self.base_url, params={"api_key": self.api_key}, headers=self.headers)

    def _image(self, rel_url, size="original"):
        return "%s/%s/%s" % (self.config()["images"]["base_url"], size, rel_url)
