import os
import stat
import subprocess
import time
import xbmc
from xbmctorrent import logpipe, plugin
from xbmctorrent.common import RESOURCES_PATH
from xbmctorrent.osarch import PLATFORM
from xbmctorrent.utils import url_get


def android_get_current_appid():
    with open("/proc/%d/cmdline" % os.getpid()) as fp:
        return fp.read().rstrip("\0")


def ensure_exec_perms(file_):
    st = os.stat(file_)
    os.chmod(file_, st.st_mode | stat.S_IEXEC)
    return file_

def get_t2h_checksum(path):
    with open(path) as fp:
        fp.seek(-40, os.SEEK_END)  # we put a sha1 there
        return fp.read()

def get_torrent2http_binary():
    platform = PLATFORM.copy()

    binary = "torrent2http" + (".exe" if platform["os"] == "windows" else "")
    binary_dir = os.path.join(RESOURCES_PATH, "bin", "%(os)s_%(arch)s" % platform)
    binary_path = os.path.join(binary_dir, binary)

    if not os.path.isfile(binary_path):
        plugin.log.error("Unable to find torrent2http binary with path: %s" % binary_path)
        return None, None

    if platform["os"] == "android":
        binary_dir_legacy = binary_dir.replace("/storage/emulated/0", "/storage/emulated/legacy")
        if os.path.exists(binary_dir_legacy):
            binary_dir = binary_dir_legacy
        xbmc.log("Using binary folder: %s" % binary_dir)
        app_id = android_get_current_appid()
        xbmc_data_path = os.path.join("/data", "data", app_id)

        android_binary_dir = os.path.join(xbmc_data_path, "files", plugin.id, "bin", "%(os)s_%(arch)s" % platform)
        android_binary_path = os.path.join(android_binary_dir, binary)

        if not os.path.exists(android_binary_path) or get_t2h_checksum(binary_path) != get_t2h_checksum(android_binary_path):
            import shutil
            try:
                os.makedirs(android_binary_dir)
            except OSError:
                pass
            try:
                shutil.rmtree(android_binary_dir)
            except OSError as e:
                plugin.log.error("Unable to remove destination path for update: %s" % e)
                pass
            try:
                shutil.copytree(binary_dir, android_binary_dir)
            except OSError as e:
                plugin.log.error("Unable to copy to destination path for update: %s" % e)
                pass

        binary_path = android_binary_path
        binary_dir = android_binary_dir

    return binary_dir, ensure_exec_perms(binary_path)


def logger(message):
    import xbmc
    xbmc.log("[torrent2http] %s" % message, level=xbmc.LOGDEBUG)


def find_free_port(host):
    import socket
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    s.bind((host, 0))
    port = s.getsockname()[1]
    s.close()
    return port


def start(**kwargs):
    torrent2http_dir, torrent2http_bin = get_torrent2http_binary()
    if not torrent2http_dir:
        return

    host = "127.0.0.1"  # don't use localhost because python has an issue with it
    args = [torrent2http_bin]
    bind_port = find_free_port(host)
    bind_address = "%s:%d" % (host, bind_port)
    kwargs["bind"] = bind_address

    for k, v in kwargs.items():
        if v is not None:
            if isinstance(v, bool):
                if v:
                    args.append("--%s" % k)
                else:
                    args.append("--%s=false" % k)
            else:
                args.append("--%s" % k)
                if not isinstance(v, str) and not isinstance(v, unicode):
                    v = str(v)
                args.append(v)

    plugin.log.debug(repr(args))

    log_thread = logpipe.LogPipe(logger)
    kwargs = {
        "cwd": torrent2http_dir,
        "stderr": log_thread,
        "stdout": log_thread
    }
    if PLATFORM["os"] == "windows":
        si = subprocess.STARTUPINFO()
        si.dwFlags |= 1
        si.wShowWindow = 0
        kwargs["startupinfo"] = si
    else:
        env = os.environ.copy()
        env["LD_LIBRARY_PATH"] = "%s:%s" % (
            torrent2http_dir, env.get("LD_LIBRARY_PATH", ""))
        kwargs["env"] = env
        kwargs["close_fds"] = True

    proc = subprocess.Popen(args, **kwargs)
    proc.bind_address = bind_address

    def proc_close():
        if proc.poll() is None:
            try:
                url_get("http://%s/shutdown" %
                        proc.bind_address, with_immunicity=False)
            except:
                pass
            finished = False
            os.close(log_thread.write_fd)
            start = time.time()
            while (time.time() - start) < 10:
                time.sleep(0.2)
                if proc.poll() is not None:
                    finished = True
                    break
            if not finished:
                logger("timeout occured while shutting down")
                proc.kill()
            proc.wait()
    proc.close = proc_close
    return proc
