# -*- coding: utf-8 -*-

import os
import threading
import time
import xbmc
from contextlib import closing, contextmanager, nested
from xbmctorrent import plugin
from xbmctorrent.osarch import PLATFORM
from xbmctorrent.utils import get_current_list_item

ADDON_KEY = "n51LvQoTlJzNGaFxseRK-uvnvX-sD4Vm5Axwmc4UcoD-jruxmKsuJaH0eVgE"

DEFAULT_ACESTREAM_HOST = "127.0.0.1"
DEFAULT_ACESTREAM_PORT = 62062

ACESTREAM_CONNECTION_TIMEOUT = 20   # s
ACESTREAM_CONNECTION_POLL = 1000  # ms
PLAYING_EVENT_INTERVAL = 1000  # ms

XBFONT_LEFT = 0x00000000
XBFONT_RIGHT = 0x00000001
XBFONT_CENTER_X = 0x00000002
XBFONT_CENTER_Y = 0x00000004
XBFONT_TRUNCATED = 0x00000008
XBFONT_JUSTIFY = 0x00000010

STATE_STRS = {
    "idle": "Бездействие",
    "loading": "Загружается транспортный файл",
    "starting": "Инициализация воспроизведения",
    "prebuf": "Пребуферизация",
    "dl": "Загрузка контента",
    "buf": "Буферизация",
    "check": "Проверка",
    "wait": "Ожидание",
    "err": "Ошибка"
}

VIEWPORT_WIDTH = 1920.0
VIEWPORT_HEIGHT = 1080.0
OVERLAY_WIDTH = int(VIEWPORT_WIDTH * 0.6)  # 60% size
OVERLAY_HEIGHT = 150


def ace_supported():
    return PLATFORM["os"] == "windows" or PLATFORM["os"] == "linux"


class AceStreamPlayer(xbmc.Player):

    def init(self, torrent, index=0, name="", is_raw=False):
        self.torrent = torrent
        self.index = index
        self.file_name = name
        self.is_raw = is_raw
        self.on_playback_started = []
        self.on_playback_resumed = []
        self.on_playback_paused = []
        self.on_playback_stopped = []
        self.on_playback_ended = []
        self.on_playback_seek = []
        return self

    def onPlayBackStarted(self):
        for f in self.on_playback_started:
            f()

    def onPlayBackResumed(self):
        for f in self.on_playback_resumed:
            f()

    def onPlayBackPaused(self):
        for f in self.on_playback_paused:
            f()

    def onPlayBackStopped(self):
        for f in self.on_playback_stopped:
            f()

    def onPlayBackSeek(self, time, seekOffset):
        for f in self.on_playback_seek:
            f(time, seekOffset)

    def onPlayBackEnded(self):
        for f in self.on_playback_ended:
            f()

    def _get_status_lines(self, status):
        return [
            self.file_name,
            "%s%% %s" % (status["progress"], STATE_STRS[status["status"]]),
            "%(speed)d kb/s / %(peers)d " % status
        ]

    @contextmanager
    def attach(self, callback, *events):
        for event in events:
            event.append(callback)
        yield
        for event in events:
            event.remove(callback)

    def loop(self):
        from xbmctorrent.utils import SafeDialogProgress
        from xbmctorrent.player import OverlayText

        has_resolved = False
        plugin.log.info("Starting AceStream...")

        def safe_cast(val, to_type, default=None):
            try:
                return to_type(val)
            except ValueError:
                return default

        engine_params = {
            "filename": self.file_name,
            "host": plugin.get_setting("ace_host", str) or DEFAULT_ACESTREAM_HOST,
            "port": safe_cast(plugin.get_setting("ace_port"), int, DEFAULT_ACESTREAM_PORT),
            "keep_enc": plugin.get_setting("ace_keep_encripted", bool)
        }

        if plugin.get_setting("keep_files", bool):
            plugin.log.info("Will keep file after playback.")
            engine_params["dlpath"] = xbmc.validatePath(xbmc.translatePath(plugin.get_setting("dlpath") or "special://temp/" + plugin.id))

            if engine_params["dlpath"] and "://" in engine_params["dlpath"]:
                # Translate smb:// url to UNC path on windows, very hackish
                if PLATFORM["os"] == "windows" and engine_params["dlpath"].lower().startswith("smb://"):
                    engine_params["dlpath"] = engine_params["dlpath"].replace("smb:", "").replace("/", "\\")

        with closing(AceEngine(**engine_params)) as engine:
            plugin.log.info("Opening download dialog...")

            # Get currently selected item
            current_item = get_current_list_item()

            with closing(SafeDialogProgress(delay_create=0)) as dialog:
                dialog.create("AceStream Player")

                connected = engine.try_to_connect()

                n = 1
                start = time.time()
                while (time.time() - start) < ACESTREAM_CONNECTION_TIMEOUT:
                    if xbmc.abortRequested or dialog.iscanceled():
                        return

                    dialog.update(0, self.file_name, "Попытка соединения с AceStream (%s)" % n)
                    if not connected:
                        plugin.log.info("Starting AceStream engine")

                        if engine.start():
                            xbmc.sleep(ACESTREAM_CONNECTION_POLL)

                        connected = engine.try_to_connect()
                        continue
                    else:
                        if engine.is_ready():
                            break
                        plugin.log.info("Engine is not ready")

                    xbmc.sleep(ACESTREAM_CONNECTION_POLL)
                    n = n + 1

                if not engine.is_ready():
                    return

                dialog.update(0, self.file_name, "Соединение установлено. Запуск загрузки данных...")

                if not engine.play(self.torrent, self.index, is_raw=self.is_raw):
                    return

                while not has_resolved:
                    if xbmc.abortRequested or dialog.iscanceled():
                        return

                    status = engine.get_status()

                    if status["error"]:
                        dialog.update(int(status["progress"]), *[self.file_name, STATE_STRS[status["status"]], status["error"]])
                        xbmc.sleep(PLAYING_EVENT_INTERVAL)
                        return

                    if status["state"] > 0 and status["status"]:
                        dialog.update(int(status["progress"]), *self._get_status_lines(status))

                    if status["state"] >= 2 and status["status"] == "dl" and not has_resolved:  # Downloading?
                        if int(status["progress"]) >= 0:
                            plugin.log.info("Resolving to %s" % status["url"])
                            has_resolved = True
                            if not current_item["info"].get("title"):
                                current_item["label"] = self.file_name
                            current_item["path"] = status["url"]
                            plugin.set_resolved_url(current_item)
                            break

                    xbmc.sleep(PLAYING_EVENT_INTERVAL)

            # We are now playing
            plugin.log.info("Now playing torrent...")

            with closing(OverlayText(w=OVERLAY_WIDTH, h=OVERLAY_HEIGHT, alignment=XBFONT_CENTER_X | XBFONT_CENTER_Y)) as overlay:
                with nested(self.attach(overlay.show, self.on_playback_paused),
                            self.attach(overlay.hide, self.on_playback_resumed, self.on_playback_stopped),
                            self.attach(engine.on_start, self.on_playback_started),
                            self.attach(engine.on_pause, self.on_playback_paused),
                            self.attach(engine.on_resume, self.on_playback_resumed),
                            self.attach(engine.on_stop, self.on_playback_stopped),
                            self.attach(engine.on_seek, self.on_playback_seek),
                            self.attach(engine.on_end, self.on_playback_ended),
                            self.attach(self.pause, engine.on_playback_paused),
                            self.attach(self.play, engine.on_playback_resumed)):
                    while not xbmc.abortRequested and self.isPlaying() and not engine.error:
                        overlay.text = "\n".join(self._get_status_lines(engine.get_status()))
                        xbmc.sleep(PLAYING_EVENT_INTERVAL)

        plugin.log.info("Closing AceStream player.")


class AceEngine():

    def __init__(self, **kwargs):
        self.state = None
        self.host = DEFAULT_ACESTREAM_HOST
        self.port = DEFAULT_ACESTREAM_PORT
        self.sink = None
        self.duration = None
        self.link = None
        self.is_ad = False
        self.is_live = False
        self.cansave = []
        self.files = []
        self.key = None
        self.version = None
        self.status = None
        self.progress = 0
        self.speed = 0
        self.peers = 0
        self.on_playback_resumed = []
        self.on_playback_paused = []
        self.dlpath = None
        self.index = 0
        self.filename = None
        self.keep_enc = False
        self.error = False
        self.error_msg = None

        for k, v in kwargs.items():
            try:
                setattr(self, k, v)
            except:
                pass

    def on_start(self):
        try:
            self.duration = int(xbmc.Player().getTotalTime() * 1000)
        except:
            self.duration = 0
        self.sink.send("DUR %s %s" % (self.link, self.duration))
        self.sink.send("PLAYBACK %s 0" % self.link)

    def on_pause(self):
        self.sink.send("EVENT pause")

    def on_resume(self):
        self.sink.send("EVENT play")
        pass

    def on_stop(self):
        self.sink.send("EVENT stop")
        self.sink.send("STOP")

    def on_seek(self, time, seekOffset):
        self.sink.send("EVENT seek position=%s" % int(time / 1000))

    def on_end(self):
        self.sink.send("PLAYBACK %s 100" % self.link)
        self.sink.send("EVENT stop")
        self.sink.send("STOP")

    def close(self):
        if self.sink:
            if self.state > 0:
                self.on_stop()
                xbmc.sleep(100)
            self.sink.send("SHUTDOWN")

    def shutdown(self):
        if self.sink:
            self.sink.end()

    def start(self):
        if not self._is_local():
            return True

        if PLATFORM["os"] == "windows":
            return self._start_windows()

        return self._start_linux()

    def try_to_connect(self):
        if not self.sink:
            self.sink = ASSink(self, self.host, 100)

        if self.sink.connect(self._get_ace_port()):
            self.sink.send("HELLOBG")
            return True

        return False

    def load(self, torrent, is_raw=False):
        import random
        import base64

        if is_raw:
            format = "RAW"
            torrent = base64.b64encode(torrent)
        else:
            format = "TORRENT"

        self.sink.send("LOADASYNC %s %s %s 0 0 0" % (str(random.randint(0, 0x7fffffff)), format, torrent))

    def play(self, torrent, index=0, is_raw=False):
        import base64

        self.index = index
        try:
            if is_raw:
                format = "RAW"
                torrent = base64.b64encode(torrent)
            else:
                format = "TORRENT"

            self.sink.send("START %s %s %s 0 0 0" % (format, torrent, str(self.index)))
            return True
        except:
            return False

    def get_status(self):
        return {
            "state": self.state,
            "status": self.status,
            "progress": self.progress,
            "speed": self.speed,
            "peers": self.peers,
            "url": self.link,
            "error": self.error_msg
        }

    def is_ready(self):
        return self.state >= 0

    def _start_windows(self):
        try:
            import _winreg
            key = _winreg.OpenKey(_winreg.HKEY_CURRENT_USER, "Software\\AceStream")
            path = _winreg.QueryValueEx(key, "EnginePath")[0]
        except:
            # trying previous version
            try:
                import _winreg
                key = _winreg.OpenKey(_winreg.HKEY_CURRENT_USER, "Software\\TorrentStream")
                path = _winreg.QueryValueEx(key, "EnginePath")[0]
            except:
                self.error = 1
                return False
        try:
            plugin.log.info("Starting AceStream engine: %s" % path)
            os.startfile(path)
        except:
            plugin.log.info("Error starting AceStream engine: %s" % path)
            self.error = 1
            return False

        return True

    def _start_linux(self):
        import subprocess

        try:
            subprocess.Popen(["acestreamengine", "--client-console"])
        except:
            plugin.log.warning("AceEngine not installed")
            return False

        return True

    def _get_ace_port(self):
        aceport = self.port
        if self._is_local() and PLATFORM["os"] == "windows":
            try:
                import _winreg
                key = _winreg.OpenKey(_winreg.HKEY_CURRENT_USER, "Software\\AceStream")
                engine_exe = _winreg.QueryValueEx(key, "EnginePath")[0]
                path = engine_exe.replace("ace_engine.exe", "")
                pfile = os.path.join(path, "acestream.port")
                hfile = open(pfile, "r")
                aceport = int(hfile.read())
            except:
                # trying previous version
                try:
                    import _winreg
                    key = _winreg.OpenKey(_winreg.HKEY_CURRENT_USER, "Software\\TorrentStream")
                    engine_exe = _winreg.QueryValueEx(key, "EnginePath")[0]
                    path = engine_exe.replace("tsengine.exe", "")
                    pfile = os.path.join(path, "acestream.port")
                    hfile = open(pfile, "r")
                    aceport = int(hfile.read())
                except:
                    pass
        return aceport

    def _track_sink_event(self, event, params):
        self.last_event = event
        self.last_event_params = params

        if event == "STATUS":
            self._update_status(params)

        elif event == "START" or event == "PLAY":
            parts = params.split(" ")
            self.link = parts[0].replace(DEFAULT_ACESTREAM_HOST, self.host)

            if len(parts) > 1:
                if "ad=1" in parts:
                    self.is_ad = True
                    self.sink.send("PLAYBACK %s 100" % self.link)
                    self.link = None
                else:
                    self.is_ad = False

                if "stream=1" in self.params:
                    self.is_live = True

        elif event == "AUTH":
            self.auth = int(params)
            self.state = 0  # ready

        elif event == "STATE":
            self.state = int(params)

        elif event == "EVENT":
            parts = params.split(" ")
            if len(parts) > 1 and "cansave" in parts:
                self.cansave.append({
                    "index": int(parts[1].split("=")[1]),
                    "infohash": parts[2].split("=")[1],
                    "format": parts[3].split("=")[1]
                })
                if self.dlpath:
                    self._save_file()

        elif event == "RESUME":
            for f in self.on_playback_resumed:
                f()

        elif event == "PAUSE":
            for f in self.on_playback_paused:
                f()

        elif event == "HELLOTS":
            parts = params.split(" ")
            for part in parts:
                k, v = part.split("=")
                if k == "key":
                    self.key = v
                elif k == "version":
                    self.version = v

            ready = "READY"
            if self.key:
                import hashlib
                sha1 = hashlib.sha1()

                sha1.update(self.key + ADDON_KEY)
                ready = "READY key=%s-%s" % (ADDON_KEY.split("-")[0], sha1.hexdigest())

            self.sink.send(ready)

        elif event == "LOADRESP":
            import json

            loadresp_json = json.loads(params[params.find("{"):len(params)])
            self.infohash = loadresp_json["infohash"]
            if loadresp_json["status"] > 0:
                self.files = loadresp_json["files"]

        elif event == "SHUTDOWN":
            self.shutdown()

    def _update_status(self, params):
        import re

        try:
            ss = re.compile("main:[a-z]+", re.S)
            s1 = re.findall(ss, params)[0]
            self.status = s1.split(":")[1]

            if self.status == "starting":
                self.progress = 0
            if self.status == "loading":
                self.progress = 0
            if self.status == "prebuf":
                parts = params.split(";")
                self.progress = int(parts[1])
                self.speed = int(parts[5])
                self.peers = int(parts[8])
            if self.status == "buf":
                parts = params.split(";")
                self.progress = int(parts[1])
                self.speed = int(parts[5])
                self.peers = int(parts[8])
            if self.status == "dl":
                parts = params.split(";")
                self.progress = int(parts[1])
                self.speed = int(parts[3])
                self.peers = int(parts[6])
            if self.status == "check":
                self.progress = int(params.split(";")[1])
                self.speed = 0
            if self.status == "idle":
                self.progress = 0
            if self.status == "wait":
                self.progress = 0
            if self.status == "err":
                parts = params.split(";")
                self.error = True
                self.error_id = parts[1]
                self.error_msg = parts[2]
        except:
            plugin.log.error("Error with text=%s" % params)

    def _save_file(self):
        import urllib

        for cansave in self.cansave:
            if cansave["index"] == int(self.index):
                savepath = self.dlpath + (self.filename or cansave["infohash"])
                if cansave["format"] == "encrypted":
                    if not self.keep_enc:
                        return
                    savepath = "%s.acemedia" % savepath

                import xbmcvfs

                dirname = os.path.dirname(savepath)
                if not xbmcvfs.exists(dirname):
                    xbmcvfs.mkdir(dirname)
                self.sink.send("SAVE infohash=%s index=%s path=%s" % (cansave["infohash"], self.index, urllib.quote(savepath)))

    def _is_local(self):
        return self.host == DEFAULT_ACESTREAM_HOST


class ASSink(threading.Thread):

    def __init__(self, engine, host, interval):
        threading.Thread.__init__(self)

        import socket

        self.engine = engine
        self.host = host
        self.interval = interval
        self.sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.active = True
        self.last_received = None
        self.last_com = None
        self.buffer = 5000000
        self.temp = ""

    def connect(self, port):
        try:
            self.sock.connect((self.host, port))
            self.start()
            return True
        except:
            return False

    def end(self):
        self.active = False

    def send(self, command):
        plugin.log.debug(">> %s" % command)

        try:
            self.sock.send("%s\r\n" % command)
        except:
            plugin.log.error("Error sending command to AceStream")
            return False

        return True

    def run(self):
        plugin.log.info("Sink thread going to running state")

        while self.active:
            try:
                self.last_received = self.sock.recv(self.buffer)
            except:
                self.last_received = ""

            ind = self.last_received.find("\r\n")
            cnt = self.last_received.count("\r\n")

            if ind != -1 and cnt == 1:
                self.last_received = self.temp + self.last_received[:ind]
                self.temp = ""
                self._exec_com()
            elif cnt > 1:
                fcom = self.last_received
                ind = 1
                while ind != -1:
                    ind = fcom.find("\r\n")
                    self.last_received = fcom[:ind]
                    self._exec_com()
                    fcom = fcom[(ind + 2):]
            elif ind == -1:
                self.temp = self.temp + self.last_received
                self.last_received = None

            xbmc.sleep(self.interval)

        plugin.log.info("Sink thread stopped")

    def _exec_com(self):
        plugin.log.debug("<< %s" % self.last_received)

        pos = self.last_received.find(" ")
        if pos >= 0:
            command = self.last_received[:pos]
            params = self.last_received[pos + 1:]
        else:
            command = self.last_received
            params = ""

        self.engine._track_sink_event(command, params)
