<?php
/**
 * ファイル名: functions.php
 * 説明: テーマの基本機能とカスタマイザー設定
 * 
 * @package Etransform
 * @author Etransform
 * @version 1.0.0
 * @license GPL v2 or later
 */

if (!defined('ABSPATH')) {
    exit;
}

/* ==========================================================================
   1. テーマセットアップ
   ========================================================================== */

function corporate_pro_setup() {
    add_theme_support('title-tag');
    add_theme_support('post-thumbnails');
    add_theme_support('html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
    ));
    
    add_theme_support('custom-logo', array(
        'height'      => 60,
        'width'       => 200,
        'flex-height' => true,
        'flex-width'  => true,
    ));
    
    register_nav_menus(array(
        'primary' => 'メインメニュー',
    ));
}
add_action('after_setup_theme', 'corporate_pro_setup');

/* ==========================================================================
   2. スクリプト・スタイルの読み込み
   ========================================================================== */

function corporate_pro_scripts() {
    // スタイル
    wp_enqueue_style(
        'corporate-pro-style',
        get_stylesheet_uri(),
        array(),
        '1.0.1'
    );
    
    // モバイルメニュースクリプト
    wp_enqueue_script(
        'corporate-pro-mobile-menu',
        get_template_directory_uri() . '/js/mobile-menu.js',
        array(),
        '1.0.0',
        true
    );
}
add_action('wp_enqueue_scripts', 'corporate_pro_scripts');

/* ==========================================================================
   3. カスタマイザー設定
   ========================================================================== */

function corporate_pro_customize_register($wp_customize) {
    
    // ヘッダー設定
    $wp_customize->add_section('corporate_pro_header', array(
        'title'    => 'ヘッダー設定',
        'priority' => 30,
    ));
    
    $wp_customize->add_setting('header_background_color', array(
        'default'           => '#ffffff',
        'sanitize_callback' => 'sanitize_hex_color',
    ));
    
    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'header_background_color', array(
        'label'    => 'ヘッダー背景色',
        'section'  => 'corporate_pro_header',
        'settings' => 'header_background_color',
    )));
    
    $wp_customize->add_setting('logo_text_color', array(
        'default'           => '#333333',
        'sanitize_callback' => 'sanitize_hex_color',
    ));
    
    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'logo_text_color', array(
        'label'    => 'ロゴの色',
        'section'  => 'corporate_pro_header',
        'settings' => 'logo_text_color',
    )));
    
    $wp_customize->add_setting('menu_text_color', array(
        'default'           => '#333333',
        'sanitize_callback' => 'sanitize_hex_color',
    ));
    
    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'menu_text_color', array(
        'label'    => 'メニューの色',
        'section'  => 'corporate_pro_header',
        'settings' => 'menu_text_color',
    )));
    
    $wp_customize->add_setting('sns_icon_color', array(
        'default'           => '#666666',
        'sanitize_callback' => 'sanitize_hex_color',
    ));
    
    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'sns_icon_color', array(
        'label'    => 'SNSアイコンの色',
        'section'  => 'corporate_pro_header',
        'settings' => 'sns_icon_color',
    )));
    
    // ヘッダーテキスト設定
    $wp_customize->add_setting('header_title_text', array(
        'default'           => '',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('header_title_text', array(
        'label'       => 'ヘッダータイトル',
        'description' => '空欄の場合はサイト名が表示されます',
        'section'     => 'corporate_pro_header',
        'type'        => 'text',
    ));
    
    $wp_customize->add_setting('header_subtitle_text', array(
        'default'           => '',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('header_subtitle_text', array(
        'label'       => 'ヘッダーサブタイトル',
        'description' => 'タイトルの下に表示する補足テキスト',
        'section'     => 'corporate_pro_header',
        'type'        => 'text',
    ));
    
    $wp_customize->add_setting('header_title_color', array(
        'default'           => '#333333',
        'sanitize_callback' => 'sanitize_hex_color',
    ));
    
    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'header_title_color', array(
        'label'    => 'タイトル文字色',
        'section'  => 'corporate_pro_header',
        'settings' => 'header_title_color',
    )));
    
    $wp_customize->add_setting('header_subtitle_color', array(
        'default'           => '#666666',
        'sanitize_callback' => 'sanitize_hex_color',
    ));
    
    $wp_customize->add_control(new WP_Customize_Color_Control($wp_customize, 'header_subtitle_color', array(
        'label'    => 'サブタイトル文字色',
        'section'  => 'corporate_pro_header',
        'settings' => 'header_subtitle_color',
    )));
}
add_action('customize_register', 'corporate_pro_customize_register');

function corporate_pro_sns_customizer($wp_customize) {
    $wp_customize->add_section('corporate_pro_sns', array(
        'title'    => 'SNS設定',
        'priority' => 32,
    ));
    
    $wp_customize->add_setting('sns_twitter', array(
        'default'           => '',
        'sanitize_callback' => 'esc_url_raw',
    ));
    
    $wp_customize->add_control('sns_twitter', array(
        'label'    => 'Twitter URL',
        'section'  => 'corporate_pro_sns',
        'type'     => 'url',
    ));
    
    $wp_customize->add_setting('sns_instagram', array(
        'default'           => '',
        'sanitize_callback' => 'esc_url_raw',
    ));
    
    $wp_customize->add_control('sns_instagram', array(
        'label'    => 'Instagram URL',
        'section'  => 'corporate_pro_sns',
        'type'     => 'url',
    ));
    
    $wp_customize->add_setting('sns_youtube', array(
        'default'           => '',
        'sanitize_callback' => 'esc_url_raw',
    ));
    
    $wp_customize->add_control('sns_youtube', array(
        'label'    => 'YouTube URL',
        'section'  => 'corporate_pro_sns',
        'type'     => 'url',
    ));
}
add_action('customize_register', 'corporate_pro_sns_customizer');

function corporate_pro_footer_customizer($wp_customize) {
    $wp_customize->add_section('corporate_pro_footer', array(
        'title'    => 'フッター設定',
        'priority' => 33,
    ));
    
    // ========== 1. 会社情報 ==========
    $wp_customize->add_setting('footer_description', array(
        'default'           => '信頼と実績のITソリューション企業',
        'sanitize_callback' => 'sanitize_textarea_field',
    ));
    
    $wp_customize->add_control('footer_description', array(
        'label'    => '会社説明文',
        'section'  => 'corporate_pro_footer',
        'type'     => 'textarea',
    ));
    
    // ========== 2. サイトマップ ==========
    $wp_customize->add_setting('footer_sitemap_title', array(
        'default'           => 'サイトマップ',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_sitemap_title', array(
        'label'    => 'サイトマップタイトル',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    // ========== 3. サービス ==========
    $wp_customize->add_setting('footer_services_title', array(
        'default'           => 'サービス',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_services_title', array(
        'label'    => 'サービスタイトル',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    // サービス1〜4
    for ($i = 1; $i <= 4; $i++) {
        $wp_customize->add_setting('footer_service_' . $i . '_name', array(
            'default'           => '',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        
        $wp_customize->add_control('footer_service_' . $i . '_name', array(
            'label'    => 'サービス' . $i . ' 名前',
            'section'  => 'corporate_pro_footer',
            'type'     => 'text',
        ));
        
        $wp_customize->add_setting('footer_service_' . $i . '_url', array(
            'default'           => '#',
            'sanitize_callback' => 'esc_url_raw',
        ));
        
        $wp_customize->add_control('footer_service_' . $i . '_url', array(
            'label'    => 'サービス' . $i . ' URL',
            'section'  => 'corporate_pro_footer',
            'type'     => 'url',
        ));
    }
    
    // ========== 4. お問い合わせ ==========
    $wp_customize->add_setting('footer_contact_title', array(
        'default'           => 'お問い合わせ',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_contact_title', array(
        'label'    => 'お問い合わせタイトル',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('footer_postal_code', array(
        'default'           => '〒000-0000',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_postal_code', array(
        'label'    => '郵便番号',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('footer_address', array(
        'default'           => '東京都○○区○○ 1-2-3',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_address', array(
        'label'    => '住所',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('footer_phone', array(
        'default'           => '03-1234-5678',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_phone', array(
        'label'    => '電話番号',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('footer_phone_link', array(
        'default'           => '0312345678',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_phone_link', array(
        'label'       => '電話番号(リンク用・ハイフンなし)',
        'description' => '例: 0312345678',
        'section'     => 'corporate_pro_footer',
        'type'        => 'text',
    ));
    
    $wp_customize->add_setting('footer_email', array(
        'default'           => 'info@example.com',
        'sanitize_callback' => 'sanitize_email',
    ));
    
    $wp_customize->add_control('footer_email', array(
        'label'    => 'メールアドレス',
        'section'  => 'corporate_pro_footer',
        'type'     => 'email',
    ));
    
    $wp_customize->add_setting('footer_business_hours', array(
        'default'           => '営業時間: 9:00 - 18:00',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_business_hours', array(
        'label'    => '営業時間',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    $wp_customize->add_setting('footer_holidays', array(
        'default'           => '定休日: 土日祝',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_holidays', array(
        'label'    => '定休日',
        'section'  => 'corporate_pro_footer',
        'type'     => 'text',
    ));
    
    // ========== 5. コピーライト ==========
    $wp_customize->add_setting('footer_copyright_text', array(
        'default'           => '',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('footer_copyright_text', array(
        'label'       => 'コピーライト文言',
        'description' => '空欄の場合は「© 年 会社名. All Rights Reserved.」が表示されます',
        'section'     => 'corporate_pro_footer',
        'type'        => 'text',
    ));
}
add_action('customize_register', 'corporate_pro_footer_customizer');

function corporate_pro_customize_css() {
    $header_bg = get_theme_mod('header_background_color', '#ffffff');
    $logo_color = get_theme_mod('logo_text_color', '#333333');
    $menu_color = get_theme_mod('menu_text_color', '#333333');
    $sns_color = get_theme_mod('sns_icon_color', '#666666');
    $title_color = get_theme_mod('header_title_color', '#333333');
    $subtitle_color = get_theme_mod('header_subtitle_color', '#666666');
    ?>
    <style type="text/css">
        :root {
            --header-bg: <?php echo esc_attr($header_bg); ?>;
            --logo-color: <?php echo esc_attr($logo_color); ?>;
            --menu-color: <?php echo esc_attr($menu_color); ?>;
            --sns-color: <?php echo esc_attr($sns_color); ?>;
            --title-color: <?php echo esc_attr($title_color); ?>;
            --subtitle-color: <?php echo esc_attr($subtitle_color); ?>;
        }
        .site-header {
            background: var(--header-bg);
        }
        .site-logo a,
        .text-logo {
            color: var(--logo-color);
        }
        .main-navigation a {
            color: var(--menu-color);
        }
        .header-sns a {
            color: var(--sns-color);
        }
        .header-title-text {
            color: var(--title-color);
        }
        .header-subtitle-text {
            color: var(--subtitle-color);
        }
    </style>
    <?php
}
add_action('wp_head', 'corporate_pro_customize_css');

/* ==========================================================================
   4. ナビゲーション関連
   ========================================================================== */

class Corporate_Pro_Walker_Nav_Menu extends Walker_Nav_Menu {
    function start_lvl(&$output, $depth = 0, $args = null) {
        $indent = str_repeat("\t", $depth);
        $output .= "\n$indent<ul class=\"sub-menu\">\n";
    }
}

function corporate_pro_default_menu() {
    echo '<ul class="nav-menu">';
    echo '<li><a href="' . esc_url(home_url('/')) . '">ホーム</a></li>';
    
    $about_page = get_page_by_title('会社概要');
    if ($about_page) {
        echo '<li><a href="' . esc_url(get_permalink($about_page->ID)) . '">会社概要</a></li>';
    }
    
    $service_page = get_page_by_title('サービス');
    if ($service_page) {
        echo '<li class="menu-item-has-children">';
        echo '<a href="' . esc_url(get_permalink($service_page->ID)) . '">サービス</a>';
        echo '<ul class="sub-menu">';
        echo '<li><a href="#">サービス1</a></li>';
        echo '<li><a href="#">サービス2</a></li>';
        echo '<li><a href="#">サービス3</a></li>';
        echo '</ul>';
        echo '</li>';
    }
    
    $news_page = get_page_by_title('お知らせ');
    if ($news_page) {
        echo '<li><a href="' . esc_url(get_permalink($news_page->ID)) . '">お知らせ</a></li>';
    }
    
    $contact_page = get_page_by_title('お問い合わせ');
    if ($contact_page) {
        echo '<li><a href="' . esc_url(get_permalink($contact_page->ID)) . '">お問い合わせ</a></li>';
    }
    
    echo '</ul>';
}

/* ==========================================================================
   5. ヘッダー・フッターヘルパー関数
   ========================================================================== */

function get_header_social_links() {
    $social_links = array(
        'twitter' => array(
            'url' => get_theme_mod('sns_twitter'),
            'label' => 'Twitter',
            'icon' => '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true"><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"/></svg>'
        ),
        'instagram' => array(
            'url' => get_theme_mod('sns_instagram'),
            'label' => 'Instagram',
            'icon' => '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" aria-hidden="true"><rect x="2" y="2" width="20" height="20" rx="5" ry="5"/><path d="M16 11.37A4 4 0 1 1 12.63 8 4 4 0 0 1 16 11.37z"/><line x1="17.5" y1="6.5" x2="17.51" y2="6.5"/></svg>'
        ),
        'youtube' => array(
            'url' => get_theme_mod('sns_youtube'),
            'label' => 'YouTube',
            'icon' => '<svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor" aria-hidden="true"><path d="M23.498 6.186a3.016 3.016 0 0 0-2.122-2.136C19.505 3.545 12 3.545 12 3.545s-7.505 0-9.377.505A3.017 3.017 0 0 0 .502 6.186C0 8.07 0 12 0 12s0 3.93.502 5.814a3.016 3.016 0 0 0 2.122 2.136c1.871.505 9.376.505 9.376.505s7.505 0 9.377-.505a3.015 3.015 0 0 0 2.122-2.136C24 15.93 24 12 24 12s0-3.93-.502-5.814zM9.545 15.568V8.432L15.818 12l-6.273 3.568z"/></svg>'
        )
    );
    
    return array_filter($social_links, function($social) {
        return !empty($social['url']);
    });
}

function render_social_links($links) {
    if (empty($links)) return;
    ?>
    <div class="header-sns">
        <?php foreach ($links as $social) : ?>
            <a href="<?php echo esc_url($social['url']); ?>" 
               target="_blank" 
               rel="noopener noreferrer" 
               aria-label="<?php echo esc_attr($social['label']); ?>">
                <?php echo $social['icon']; ?>
            </a>
        <?php endforeach; ?>
    </div>
    <?php
}

function get_footer_social_links() {
    return get_header_social_links();
}

function get_sitemap_pages() {
    $pages = array(
        'home' => array(
            'title' => 'ホーム',
            'url' => home_url('/')
        )
    );
    
    $page_titles = array('会社概要', 'サービス', 'お知らせ', 'お問い合わせ');
    
    foreach ($page_titles as $title) {
        $page = get_page_by_title($title);
        if ($page) {
            $pages[sanitize_title($title)] = array(
                'title' => $title,
                'url' => get_permalink($page->ID)
            );
        }
    }
    
    return $pages;
}

function get_footer_services() {
    $services = array();
    
    for ($i = 1; $i <= 4; $i++) {
        $name = get_theme_mod('footer_service_' . $i . '_name');
        $url = get_theme_mod('footer_service_' . $i . '_url', '#');
        
        if ($name) {
            $services[] = array(
                'name' => $name,
                'url' => $url
            );
        }
    }
    
    if (empty($services)) {
        $services = array(
            array('name' => 'Webサイト制作', 'url' => '#'),
            array('name' => 'システム開発', 'url' => '#'),
            array('name' => 'コンサルティング', 'url' => '#'),
            array('name' => '保守・運用', 'url' => '#')
        );
    }
    
    return $services;
}

function get_contact_info() {
    return array(
        'address' => array(
            'postal_code' => get_theme_mod('footer_postal_code', '〒000-0000'),
            'address' => get_theme_mod('footer_address', '東京都○○区○○ 1-2-3'),
            'icon' => '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" aria-hidden="true"><path d="M21 10c0 7-9 13-9 13s-9-6-9-13a9 9 0 0 1 18 0z"/><circle cx="12" cy="10" r="3"/></svg>'
        ),
        'phone' => array(
            'display' => get_theme_mod('footer_phone', '03-1234-5678'),
            'link' => get_theme_mod('footer_phone_link', '0312345678'),
            'icon' => '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" aria-hidden="true"><path d="M22 16.92v3a2 2 0 0 1-2.18 2 19.79 19.79 0 0 1-8.63-3.07 19.5 19.5 0 0 1-6-6 19.79 19.79 0 0 1-3.07-8.67A2 2 0 0 1 4.11 2h3a2 2 0 0 1 2 1.72 12.84 12.84 0 0 0 .7 2.81 2 2 0 0 1-.45 2.11L8.09 9.91a16 16 0 0 0 6 6l1.27-1.27a2 2 0 0 1 2.11-.45 12.84 12.84 0 0 0 2.81.7A2 2 0 0 1 22 16.92z"/></svg>'
        ),
        'email' => array(
            'email' => get_theme_mod('footer_email', 'info@example.com'),
            'icon' => '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" aria-hidden="true"><path d="M4 4h16c1.1 0 2 .9 2 2v12c0 1.1-.9 2-2 2H4c-1.1 0-2-.9-2-2V6c0-1.1.9-2 2-2z"/><polyline points="22,6 12,13 2,6"/></svg>'
        ),
        'hours' => array(
            'business_hours' => get_theme_mod('footer_business_hours', '営業時間: 9:00 - 18:00'),
            'holidays' => get_theme_mod('footer_holidays', '定休日: 土日祝'),
            'icon' => '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" aria-hidden="true"><circle cx="12" cy="12" r="10"/><polyline points="12 6 12 12 16 14"/></svg>'
        )
    );
}