import os
import sys
import socket
import time

from kodi_six import xbmc, xbmcplugin, xbmcgui
import six
from six.moves import urllib_request
from six.moves import urllib_parse
from six.moves import urllib_response
from six.moves import urllib_error

from elementum.logger import log
from elementum.config import ELEMENTUMD_HOST
from elementum.addon import ADDON, ADDON_ID, ADDON_PATH
from elementum.util import notify, getLocalizedString, getLocalizedLabel

try:
    import simplejson as json
except ImportError:
    import json


HANDLE = int(sys.argv[1])


class RedirectException(Exception):
    pass


class PlayerException(Exception):
    pass


class InfoLabels(dict):
    def __init__(self, *args, **kwargs):
        self.update(*args, **kwargs)

    def __getitem__(self, key):
        return dict.get(self, key.lower(), "")

    def __setitem__(self, key, val):
        dict.__setitem__(self, key.lower(), val)

    def update(self, *args, **kwargs):
        for k, v in dict(*args, **kwargs).items():
            self[k] = v


class closing(object):
    def __init__(self, thing):
        self.thing = thing

    def __enter__(self):
        return self.thing

    def __exit__(self, *exc_info):
        self.thing.close()


class NoRedirectHandler(urllib_request.HTTPRedirectHandler):
    def http_error_302(self, req, fp, code, msg, headers):
        infourl = urllib_response.addinfourl(fp, headers, headers["Location"])
        infourl.status = code
        infourl.code = code
        log.debug("Redirecting to: %s", headers["Location"])
        return infourl
    http_error_300 = http_error_302
    http_error_301 = http_error_302
    http_error_303 = http_error_302
    http_error_307 = http_error_302


def getInfoLabels():
    id_list = [int(s) for s in sys.argv[0].split("/") if s.isdigit()]
    tmdb_id = id_list[0] if id_list else None

    # if xbmc.getInfoLabel("ListItem.DBID"):
    #     url = "%s/infolabels" % (ELEMENTUMD_HOST)
    # elif not tmdb_id:
    if not tmdb_id:
        request_url = sys.argv[0] + sys.argv[2]
        parsed_url = urllib_parse.urlparse(request_url)
        query = urllib_parse.parse_qs(parsed_url.query)
        log.debug("Parsed URL: %s, Query: %s", repr(parsed_url), repr(query))
        if 'tmdb' in query and 'type' in query and query['type'][0] == 'search':
            tmdb_id = query['tmdb'][0]
            url = "%s/search/infolabels/%s" % (ELEMENTUMD_HOST, tmdb_id)
        elif '/search' in parsed_url and 'tmdb' in query:
            tmdb_id = query['tmdb'][0]
            url = "%s/search/infolabels/%s" % (ELEMENTUMD_HOST, tmdb_id)
        elif '/search' in parsed_url and 'q' in query:
            tmdb_id = query['q'][0]
            url = "%s/search/infolabels/%s" % (ELEMENTUMD_HOST, tmdb_id)
        elif 'tmdb' in query and 'type' in query and query['type'][0] == 'movie':
            tmdb_id = query['tmdb'][0]
            url = "%s/movie/%s/infolabels" % (ELEMENTUMD_HOST, tmdb_id)
        elif 'show' in query:
            tmdb_id = query['show'][0]
            if 'season' in query and 'episode' in query:
                url = "%s/show/%s/season/%s/episode/%s/infolabels" % (ELEMENTUMD_HOST, tmdb_id, query['season'][0], query['episode'][0])
            else:
                url = "%s/show/%s/infolabels" % (ELEMENTUMD_HOST, tmdb_id)
        else:
            url = "%s/infolabels" % (ELEMENTUMD_HOST)
    elif 'movie' in sys.argv[0]:
        url = "%s/movie/%s/infolabels" % (ELEMENTUMD_HOST, tmdb_id)
    elif ('episode' in sys.argv[0] or 'show' in sys.argv[0]) and len(id_list) > 2:
        url = "%s/show/%s/season/%s/episode/%s/infolabels" % (ELEMENTUMD_HOST, tmdb_id, id_list[1], id_list[2])
    elif 'show' in sys.argv[0] and len(id_list) == 2:
        url = "%s/show/%s/season/%s/episode/%s/infolabels" % (ELEMENTUMD_HOST, tmdb_id, id_list[1], 1)
    else:
        url = "%s/infolabels" % (ELEMENTUMD_HOST)

    log.debug("Resolving TMDB item by calling %s for %s" % (url, repr(sys.argv)))

    try:
        with closing(urllib_request.urlopen(url)) as response:
            resolved = json.loads(response.read(), parse_int=str)
            if not resolved:
                return {}

            if 'info' in resolved and resolved['info']:
                resolved.update(resolved['info'])

            if 'art' in resolved and resolved['art']:
                resolved['artbanner'] = ''
                for k, v in resolved['art'].items():
                    resolved['art' + k] = v

            if 'info' in resolved:
                del resolved['info']
            if 'art' in resolved:
                del resolved['art']
            if 'stream_info' in resolved:
                del resolved['stream_info']

            if 'DBTYPE' not in resolved:
                resolved['DBTYPE'] = 'video'
            if 'mediatype' not in resolved or resolved['mediatype'] == '':
                resolved['mediatype'] = resolved['DBTYPE']

            return resolved
    except:
        log.debug("Could not resolve TMDB item: %s" % tmdb_id)
        return {}


def _json(url):
    with closing(urllib_request.urlopen(url)) as response:
        if response.code == 300:
            raise PlayerException(response.info().getheader('Location'))
        elif response.code == 301:
            raise RedirectException(response.info().getheader('Location'))
        elif response.code >= 302 and response.code <= 307:
            # Pause currently playing Elementum file to avoid doubling requests
            try:
                if xbmc.Player().isPlaying() and ADDON_ID in xbmc.Player().getPlayingFile():
                    xbmc.Player().pause()
            except:
                pass

            _infoLabels = InfoLabels(getInfoLabels())

            item = xbmcgui.ListItem(
                path=response.geturl(),
                label=_infoLabels["label"],
                label2=_infoLabels["label2"],
                thumbnailImage=_infoLabels["thumbnail"])

            item.setArt({
                "thumb": _infoLabels["artthumb"],
                "poster": _infoLabels["artposter"],
                "tvshowposter": _infoLabels["arttvshowposter"],
                "banner": _infoLabels["artbanner"],
                "fanart": _infoLabels["artfanart"],
                "clearart": _infoLabels["artclearart"],
                "clearlogo": _infoLabels["artclearlogo"],
                "landscape": _infoLabels["artlandscape"],
                "icon": _infoLabels["articon"]
            })

            item.setInfo(type='Video', infoLabels=_infoLabels)
            xbmcplugin.setResolvedUrl(HANDLE, True, item)
            return

            # TODO: If somebody wants to try to fix the issue with having correct naming in Kodi player information.
            # Currently Kodi pushes file name, which is stored on the disk (inside of strm file).
            # And the problem is that we can't change it (or at least, I'm not aware of any ways to do it).
            # So I have tried to remove it and create a playlist, but that does not work with starting anything from a library.

            # try:
            #     log.debug("PLAYING: %s" % (xbmc.Player().getPlayingFile()))
            # except:
            #     log.debug("NOT PLAYING")

            # if "elementum" in response.geturl().lower() and "http" not in response.geturl().lower():
            #     log.debug("RESOLVED: %s" % response.geturl().lower())
            #     xbmcplugin.setResolvedUrl(HANDLE, True, item)
            # else:
            #     playlistId = xbmc.PLAYLIST_VIDEO
            #     if xbmc.PlayList(1).size() > 0:
            #         playlistId = 1
            #     elif xbmc.PlayList(0).size() > 0:
            #         playlistId = 0

            #     playlist = xbmc.PlayList(playlistId)
            #     playlist.clear()
            #     playlist.add(url=response.geturl(), listitem=item, index=0)
            #     log.debug("PLAYLIST: %s --- %d" % (response.geturl(), playlist.size()))
            #     xbmc.Player().play(playlist)

            # return

        payload = response.read()

        try:
            if payload:
                return json.loads(payload)
        except:
            raise Exception(payload)


def run(url_suffix="", retry=0):
    try:
        buffer_timeout = int(ADDON.getSetting("buffer_timeout"))
        if buffer_timeout < 60:
            buffer_timeout = 60
    except:
        buffer_timeout = 60
    buffer_timeout = buffer_timeout * 2

    try:
        preload_timeout = int(ADDON.getSetting("preload_timeout"))
        if preload_timeout < 1:
            preload_timeout = 1
    except:
        preload_timeout = 1

    socket.setdefaulttimeout(buffer_timeout)
    urllib_request.install_opener(urllib_request.build_opener(NoRedirectHandler()))

    # Pause currently playing Elementum file to avoid doubling requests
    try:
        if xbmc.Player().isPlaying() and ADDON_ID in xbmc.Player().getPlayingFile():
            xbmc.Player().pause()
    except:
        pass

    url = sys.argv[0].replace("plugin://%s" % ADDON_ID, ELEMENTUMD_HOST + url_suffix) + sys.argv[2]
    query_add = ""

    if len(sys.argv) > 3:
        app = sys.argv[3].replace(":", "=")

        # Replacing resume=false with resume=true if item is launched from main window
        title = xbmc.getInfoLabel('ListItem.Title')
        label = xbmc.getInfoLabel('ListItem.Label')
        # if "resume" in app and not title and not label:
        #     app = app.replace("resume=false", "resume=true")
        # if "resume" not in app and not title and not label:
        if not title and not label:
            if app:
                app = app.replace("resume=false", "")
                app += "&"
            if app and "resume" not in app:
                app += "resume=true"
        query_add = app
    elif "play" in url or "links" in url:
        query_add = "resume=true"

    if query_add and "resume" not in url:
        query_add = query_add.replace("resume=", "doresume=")
        if "?" in url:
            url += "&" + query_add
        else:
            url += "?" + query_add

    log.debug("Requesting %s from %s" % (url, repr(sys.argv)))

    try:
        data = _json(url)
    except PlayerException as e:
        redirect_url = e.__str__()
        log.debug("Launching player with %s" % (redirect_url))
        xbmcplugin.endOfDirectory(HANDLE, succeeded=True)
        xbmc.sleep(500)
        xbmc.executeJSONRPC('{"jsonrpc":"2.0","method":"Player.Open","params":{"item":{"file":"%s"}},"id":"1"}' % (redirect_url))
        return
    except RedirectException as e:
        redirect_url = e.__str__()
        log.debug("Redirecting Kodi with %s" % (redirect_url))
        xbmcplugin.endOfDirectory(HANDLE, succeeded=True)
        xbmc.sleep(500)
        if "keyboard=1" in sys.argv[0]:
            xbmc.executebuiltin('Container.Update(%s,replace)' % (redirect_url))
        else:
            xbmc.executebuiltin('Container.Update(%s)' % (redirect_url))
        return
    except urllib_error.URLError as e:
        # We can retry the request if connection is refused.
        # For example when plugin has not yet started but is requested by someone.
        if retry <= 2:
            time.sleep(preload_timeout)
            return run(retry=retry + 1)

        if isinstance(e.reason, IOError) or isinstance(e.reason, OSError) or 'Connection refused' in e.reason:
            notify(getLocalizedString(30116), time=7000)
        else:
            import traceback
            map(log.error, traceback.format_exc().split("\n"))
            notify(e.reason, time=7000)
        return
    except Exception as e:
        import traceback
        map(log.error, traceback.format_exc().split("\n"))
        try:
            msg = six.ensure_text(e.__str__(), errors='ignore')
        except:
            try:
                msg = six.ensure_binary(e.__str__(), errors='ignore')
            except:
                msg = repr(e)
        notify(getLocalizedLabel(msg), time=7000)
        return

    if not data:
        return

    if data["content_type"]:
        content_type = data["content_type"]
        if data["content_type"].startswith("menus"):
            content_type = data["content_type"].split("_")[1]

        xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_UNSORTED)
        if content_type != "tvshows":
            xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_LABEL_IGNORE_THE)
        else:
            xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_TITLE_IGNORE_THE)
        xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_DATE)
        xbmcplugin.addSortMethod(HANDLE, xbmcplugin.SORT_METHOD_GENRE)
        xbmcplugin.setContent(HANDLE, content_type)

    listitems = list(range(len(data["items"])))
    for i, item in enumerate(data["items"]):
        # Translate labels
        if item["label"][0:8] == "LOCALIZE":
            item["label"] = getLocalizedLabel(item["label"])
            if isinstance(item["label"], str):
                item["label"] = six.ensure_text(item["label"], 'utf-8')
        if item["label2"][0:8] == "LOCALIZE":
            item["label2"] = getLocalizedLabel(item["label2"])

        listItem = xbmcgui.ListItem(label=item["label"], label2=item["label2"], iconImage=item["icon"], thumbnailImage=item["thumbnail"])
        if item.get("info"):
            listItem.setInfo("video", item["info"])
        if item.get("stream_info"):
            for type_, values in item["stream_info"].items():
                listItem.addStreamInfo(type_, values)
        if item.get("art"):
            listItem.setArt(item["art"])
        elif ADDON.getSetting('default_fanart') == 'true' and item["label"] != six.ensure_text(getLocalizedString(30218), 'utf-8'):
            fanart = os.path.join(ADDON_PATH, "fanart.png")
            listItem.setArt({'fanart': fanart})
        if item.get("context_menu"):
            # Translate context menus
            for m, menu in enumerate(item["context_menu"]):
                if menu[0][0:8] == "LOCALIZE":
                    menu[0] = getLocalizedLabel(menu[0])
            listItem.addContextMenuItems(item["context_menu"])
        listItem.setProperty("isPlayable", item["is_playable"] and "true" or "false")
        if item.get("properties"):
            for k, v in item["properties"].items():
                listItem.setProperty(k, v)
        listitems[i] = (item["path"], listItem, not item["is_playable"])

    xbmcplugin.addDirectoryItems(HANDLE, listitems, totalItems=len(listitems))

    # Set ViewMode
    if data["content_type"]:
        viewMode = ADDON.getSetting("viewmode_%s" % data["content_type"])
        if viewMode:
            try:
                xbmc.executebuiltin('Container.SetViewMode(%s)' % viewMode)
            except Exception as e:
                log.warning("Unable to SetViewMode(%s): %s" % (viewMode, repr(e)))

    xbmcplugin.endOfDirectory(HANDLE, succeeded=True, updateListing=False, cacheToDisc=False)
