#!/usr/bin/python3

import argparse
import random
import sys


N_MIN = 1
N_MAX = 50
COORD_MAX = 2 ** 52
COORD_MIN = -COORD_MAX
CASES_MIN = 1
CASES_MAX = 100
RNG = None


def bounded_int(string, val_min, val_max, name='Value'):
    value = int(string)
    if value < val_min or value > val_max:
        raise argparse.ArgumentTypeError(f'{name} must be in range [{val_min}, {val_max}]')
    return value


def bounded_cases(string):
    return bounded_int(string, CASES_MIN, CASES_MAX, 'Cases')


def bounded_iterations(string):
    return bounded_int(string, N_MIN, N_MAX, 'Iterations')


def bounded_coordinate(string):
    return bounded_int(string, COORD_MIN, COORD_MAX, 'Coordinates')


def _parse_args():
    parser = argparse.ArgumentParser('')
    parser.add_argument(
        'num_cases', type=bounded_cases,
        help='The number of test cases to generate.'
    )
    parser.add_argument(
        'iterations_min', type=bounded_iterations,
        help='The minimum number of iterations in the construction of the Hilbert Curve.'
    )
    parser.add_argument(
        'iterations_max', type=bounded_iterations,
        help='The maximum number of iterations in the construction of the Hilbert Curve.'
    )

    parser.add_argument(
        '--coord-min', type=bounded_coordinate, default=COORD_MIN,
        help='The minimum coordinate allowed.'
    )
    parser.add_argument(
        '--coord-max', type=bounded_coordinate, default=COORD_MAX,
        help='The maximum coordinate allowed.'
    )
    parser.add_argument(
        '--seed', type=int, default=0,
        help='The random number to seed the random number generator with.'
    )
    parser.add_argument(
        '--test-name', type=str,
        help='The name for the test case. E.g., "025-small-cases" will produce files '
             '025-small-cases.in and 025-small-cases.desc. If no name is specified, '
             'output will be printed to stdout.'
    )
    parser.add_argument(
        '--x1', type=bounded_coordinate,
        help='A fixed value to use for x1.'
    )
    parser.add_argument(
        '--y1', type=bounded_coordinate,
        help='A fixed value to use for y1.'
    )
    parser.add_argument(
        '--x2', type=bounded_coordinate,
        help='A fixed value to use for x2.'
    )
    parser.add_argument(
        '--y2', type=bounded_coordinate,
        help='A fixed value to use for y2.'
    )
    return parser.parse_args()


def main():
    args = _parse_args()
    RNG = random.Random(args.seed)
    cases = []
    for _ in range(args.num_cases):
        n = RNG.randint(args.iterations_min, args.iterations_max)
        x1 = args.x1 or RNG.randint(args.coord_min, args.coord_max)
        y1 = args.y1 or RNG.randint(args.coord_min, args.coord_max)
        x2 = args.x2 or RNG.randint(args.coord_min, args.coord_max)
        y2 = args.y2 or RNG.randint(args.coord_min, args.coord_max)
        cases.append((n, x1, y1, x2, y2))
    case_strs = [' '.join([str(value) for value in values]) for values in cases]
    output_lines = [f'{args.num_cases}'] + case_strs
    output = '\n'.join(output_lines) + '\n'
    command = ' '.join(sys.argv)
    if args.test_name is not None:
        test_input_file_name = args.test_name + '.in'
        test_desc_file_name = args.test_name + '.desc'
        with open(test_input_file_name, 'w') as test_input_file:
            test_input_file.write(output)
        with open(test_desc_file_name, 'w') as test_desc_file:
            test_desc_file.write(f'Produced by:\n\t{command}\n')
    else:
        sys.stdout.write(output)


if __name__ == '__main__':
    main()
