﻿namespace MusicStore.Test.Controllers
{
    using MusicStore.Controllers;
    using MusicStore.Models;
    using MyTested.AspNetCore.Mvc;
    using System;
    using System.Linq;
    using System.Threading;
    using Xunit;

    public class ShoppingCartControllerTest
    {
        [Fact]
        public void AddToCartShouldThrowExceptionWithInvalidId()
            => MyController<ShoppingCartController>
                .Instance()
                .Calling(c => c.AddToCart(1, CancellationToken.None))
                .ShouldThrow()
                .AggregateException()
                .ContainingInnerExceptionOfType<InvalidOperationException>()
                .WithMessage()
                .Containing("Sequence contains no elements");

        [Theory]
        [InlineData(1)]
        public void AddToCartShouldPopulateSessionCartIfMissing(int albumId)
            => MyController<ShoppingCartController>
                .Instance()
                .WithData(new Album { AlbumId = albumId })
                .Calling(c => c.AddToCart(albumId, CancellationToken.None))
                .ShouldHave()
                .Session(session => session
                    .ContainingEntryWithKey("Session"));

        [Theory]
        [InlineData(1, "TestCart")]
        public void AddToCartShouldSaveTheAlbumsIntoDatabaseAndSession(
            int albumId, 
            string sessionValue)
            => MyController<ShoppingCartController>
                .Instance()
                .WithSession(session => session
                    .WithEntry("Session", sessionValue))
                .WithData(new Album { AlbumId = albumId })
                .Calling(c => c.AddToCart(albumId, CancellationToken.None))
                .ShouldHave()
                .Data(data => data
                    .WithSet<CartItem>(cartItems => cartItems
                        .Any(a => a.AlbumId == albumId && a.CartId == sessionValue)))
                .AndAlso()
                .ShouldReturn()
                .Redirect(result => result
                    .ToAction(nameof(ShoppingCartController.Index)));
    }
}
