﻿namespace MusicStore.Test.Controllers
{
    using Microsoft.AspNetCore.Mvc;
    using MusicStore.Controllers;
    using MusicStore.Models;
    using MyTested.AspNetCore.Mvc;
    using Xunit;

    public class AccountControllerTest
    {
        [Fact]
        public void LoginShouldHaveCorrectAttributes()
            => MyController<AccountController>
                .Instance()
                .Calling(c => c.Login(
                    With.Default<LoginViewModel>(),
                    With.No<string>()))
                .ShouldHave()
                .ActionAttributes(attributes => attributes
                    .RestrictingForHttpMethod(HttpMethod.Post)
                    .AllowingAnonymousRequests()
                    .ContainingAttributeOfType<ValidateAntiForgeryTokenAttribute>());

        [Fact]
        public void LoginShouldReturnRedirectToLocalWithValidLoginViewModel()
        {
            var model = new LoginViewModel
            {
                Email = "valid@valid.com",
                Password = "valid"
            };

            var redirectUrl = "/Test/Url";

            MyController<AccountController>
                .Instance()
                .Calling(c => c.Login(model, redirectUrl))
                .ShouldHave()
                .ValidModelState()
                .AndAlso()
                .ShouldReturn()
                .Redirect(result => result
                    .ToUrl(redirectUrl));
        }

        [Fact]
        public void LoginShouldReturnViewWithSameModelWithInvalidLoginViewModel()
        {
            var model = new LoginViewModel
            {
                Email = "invalid@invalid.com",
                Password = "invalid"
            };

            var redirectUrl = "/Test/Url";

            MyController<AccountController>
                .Instance()
                .Calling(c => c.Login(model, redirectUrl))
                .ShouldHave()
                .ModelState(modelState => modelState
                    .ContainingError(string.Empty)
                    .ThatEquals("Invalid login attempt."))
                .AndAlso()
                .ShouldReturn()
                .View(model);
        }

        [Theory]
        [InlineData("Test/Return/Url")]
        public void LoginShouldHaveReturnUrlInTheViewBag(string returnUrl)
            => MyController<AccountController>
                .Instance()
                .Calling(c => c.Login(returnUrl))
                .ShouldHave()
                .ViewBag(viewBag => viewBag
                    .ContainingEntry("ReturnUrl", returnUrl))
                .AndAlso()
                .ShouldReturn()
                .View();

        [Fact]
        public void ExternalLoginShouldReturnCorrectResult()
            => MyController<AccountController>
                .Instance()
                .WithRouteData()
                .Calling(c => c.ExternalLogin("TestProvider", "TestReturnUrl"))
                .ShouldReturn()
                .Challenge();
    }
}
