﻿namespace MusicStore.Test.Controllers.Admin
{
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc.Rendering;
    using Microsoft.Extensions.Caching.Memory;
    using MusicStore.Areas.Admin.Controllers;
    using MusicStore.Models;
    using MusicStore.Test.Extensions;
    using MusicStore.Test.Mocks;
    using MyTested.AspNetCore.Mvc;
    using System.Linq;
    using System.Threading;
    using Xunit;

    public class StoreManagerControllerTests
    {
        [Fact]
        public void StoreManagerControllerShouldHaveCorrectAttributes()
            => MyController<StoreManagerController>
                .Instance()
                .ShouldHave()
                .Attributes(attributes => attributes
                    .SpecifyingAdminArea()
                    .PassingFor<AuthorizeAttribute>(authorize => authorize
                        .Policy == "ManageStore"));

        [Fact]
        public void CreateShouldSaveAlbumWithValidModelStateAndRedirect()
        {
            var album = new Album
            {
                AlbumId = 1,
                Title = "TestAlbum",
                Price = 50
            };

            MyController<StoreManagerController>
                .Instance()
                .Calling(c => c.Create(
                    album,
                    From.Services<IMemoryCache>(),
                    With.Default<CancellationToken>()))
                .ShouldHave()
                .ValidModelState()
                .AndAlso()
                .ShouldHave()
                .Data(data => data
                    .WithSet<Album>(albums => albums
                        .Any(dataAlbum => dataAlbum.AlbumId == album.AlbumId)))
                .AndAlso()
                .ShouldReturn()
                .Redirect(result => result
                    .ToAction(nameof(StoreManagerController.Index)));
        }

        [Fact]
        public void DetailsShouldNotSaveCacheEntryWithDisabledGlobalCache()
        {
            var album = new Album
            {
                AlbumId = int.MaxValue
            };

            MyController<StoreManagerController>
                .Instance()
                .WithOptions(options => options
                    .For<AppSettings>(settings => settings.CacheDbResults = false))
                .WithData(album)
                .Calling(c => c.Details(
                    MockProvider.ThrowableMemoryCache,
                    int.MaxValue))
                .ShouldReturn()
                .View(result => result
                    .WithModelOfType<Album>()
                    .Passing(model => model.AlbumId == int.MaxValue));
        }

        [Fact]
        public void RemoveAlbumConfirmedShouldHaveCorrectAttributes()
            => MyController<StoreManagerController>
                .Instance()
                .Calling(c => c.RemoveAlbumConfirmed(
                    With.No<IMemoryCache>(),
                    With.No<int>(),
                    With.No<CancellationToken>()))
                .ShouldHave()
                .ActionAttributes(attributes => attributes
                    .RestrictingForHttpMethod(HttpMethod.Post)
                    .ChangingActionNameTo("RemoveAlbum"));

        [Fact]
        public void CreateShouldHaveValidEntriesInViewBag()
        {
            var genres = new[]
            {
                new Genre { GenreId = 1, Name = "Rock" },
                new Genre { GenreId = 2, Name = "Rap" }
            };

            var artists = new[]
            {
                new Artist { ArtistId = 1, Name = "Tupac" },
                new Artist { ArtistId = 2, Name = "Biggie" }
            };

            MyController<StoreManagerController>
                .Instance()
                .WithData(genres)
                .WithData(artists)
                .Calling(c => c.Create())
                .ShouldHave()
                .ViewBag(viewBag => viewBag
                    .ContainingEntries(new
                    {
                        GenreId = new SelectList(
                            From.Services<MusicStoreContext>().Genres, "GenreId", "Name"),

                        ArtistId = new SelectList(
                            From.Services<MusicStoreContext>().Artists, "ArtistId", "Name")
                    }))
                .AndAlso()
                .ShouldReturn()
                .View();
        }
    }
}
