﻿namespace MusicStore.Test.Controllers
{
    using MusicStore.Controllers;
    using MusicStore.Models;
    using MyTested.AspNetCore.Mvc;
    using System.Threading;
    using Xunit;

    public class CheckoutControllerTest
    {
        [Fact]
        public void CheckoutControllerShouldHaveAuthorizeAttribute()
            => MyController<CheckoutController>
                .Instance()
                .ShouldHave()
                .Attributes(attributes => attributes
                    .RestrictingForAuthorizedRequests());

        [Fact]
        public void AddressAndPaymentShouldReturnDefaultView()
            => MyController<CheckoutController>
                .Instance()
                .Calling(c => c.AddressAndPayment())
                .ShouldReturn()
                .View();

        [Fact]
        public void AddressAndPaymentShouldRerurnViewWithInvalidPostedPromoCode()
            => MyController<CheckoutController>
                .Instance()
                .WithHttpRequest(request => request
                    .WithFormField("PromoCode", "Invalid"))
                .Calling(c => c.AddressAndPayment(
                    From.Services<MusicStoreContext>(),
                    With.Default<Order>(),
                    CancellationToken.None))
                .ShouldHave()
                .InvalidModelState()
                .AndAlso()
                .ShouldReturn()
                .View(result => result
                    .WithModel(With.Default<Order>()));

        [Theory]
        [InlineData(1, 1, "TestCart")]
        public void AddressAndPaymentShouldRerurnRedirectWithValidData(
            int albumId,
            int orderId,
            string cartSession)
            => MyController<CheckoutController>
                .Instance()
                .WithHttpRequest(request => request
                    .WithFormField("PromoCode", "FREE"))
                .WithSession(session => session
                    .WithEntry("Session", cartSession))
                .WithUser()
                .WithRouteData()
                .WithData(data => data
                    .WithEntities(entities =>
                    {
                        var album = new Album { AlbumId = albumId, Price = 10 };
                        var cartItem = new CartItem
                        {
                            Count = 1,
                            CartId = cartSession,
                            AlbumId = albumId,
                            Album = album
                        };
                        entities.Add(album);
                        entities.Add(cartItem);
                    }))
                .WithoutValidation()
                .Calling(c => c.AddressAndPayment(
                    From.Services<MusicStoreContext>(),
                    new Order { OrderId = orderId },
                    With.No<CancellationToken>()))
                .ShouldReturn()
                .Redirect(result => result
                    .To<CheckoutController>(c => c
                        .Complete(With.Any<MusicStoreContext>(), orderId)));

        [Theory]
        [InlineData(1, "MyTestUser")]
        public void CompleteShouldReturnViewWithCorrectIdWithFoundOrderForTheUser(
            int orderId,
            string username)
            => MyController<CheckoutController>
                .Instance()
                .WithUser(user => user
                    .WithUsername(username))
                .WithData(new Order
                {
                    OrderId = orderId,
                    Username = username
                })
                .Calling(c => c.Complete(
                    From.Services<MusicStoreContext>(),
                    orderId))
                .ShouldReturn()
                .View(result => result
                    .WithModel(orderId));

        [Theory]
        [InlineData(1)]
        public void CompleteShouldReturnErrorViewWithInvalidOrderForTheUser(int orderId)
            => MyController<CheckoutController>
                .Instance()
                .WithUser(user => user
                    .WithUsername("InvalidUser"))
                .WithData(new Order
                {
                    OrderId = orderId,
                    Username = "MyTestUser"
                })
                .Calling(c => c.Complete(
                    From.Services<MusicStoreContext>(),
                    orderId))
                .ShouldReturn()
                .View(result => result
                    .WithName("Error"));
    }
}
