﻿namespace MusicStore.Test.Controllers
{
    using Microsoft.Extensions.Caching.Memory;
    using Microsoft.Extensions.DependencyInjection;
    using MusicStore.Controllers;
    using MusicStore.Models;
    using MusicStore.Test.Data;
    using MusicStore.Test.Extensions;
    using MyTested.AspNetCore.Mvc;
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using Xunit;

    public class HomeControllerTest
    {
        [Fact]
        public void HomeControllerShouldHaveNoAttributes()
            => MyController<HomeController>
                .Instance()
                .ShouldHave()
                .NoAttributes();

        [Fact]
        public void AccessDeniedShouldReturnOkStatusCodeAndProperView()
            => MyController<HomeController>
                .Instance()
                .Calling(c => c.AccessDenied())
                .ShouldHave()
                .HttpResponse(response => response
                    .WithStatusCode(HttpStatusCode.OK))
                .AndAlso()
                .ShouldReturn()
                .View(result => result
                    .WithName("~/Views/Shared/AccessDenied.cshtml"));

        [Fact]
        public void IndexShouldNotUseCacheIfOptionsDisableIt()
            => MyController<HomeController>
                .Instance()
                .WithOptions(options => options
                    .For<AppSettings>(settings => settings.CacheDbResults = false))
                .WithData(AlbumData.Many)
                .Calling(c => c.Index(
                    From.Services<MusicStoreContext>(),
                    From.Services<IMemoryCache>()))
                .ShouldPassForThe<IServiceProvider>(services => Assert.Null(services
                    .GetRequiredService<IMemoryCache>().Get("topselling")));

        [Theory]
        [InlineData(6)]
        public void IndexShouldSaveCorrectCacheEntriesIfOptionsEnableIt(int totalAlbums)
            => MyController<HomeController>
                .Instance()
                .WithOptions(options => options
                    .For<AppSettings>(settings => settings.CacheDbResults = true))
                .WithData(AlbumData.Many)
                .Calling(c => c.Index(
                    From.Services<MusicStoreContext>(),
                    From.Services<IMemoryCache>()))
                .ShouldHave()
                .MemoryCache(cache => cache
                    .ContainingEntry(entry => entry
                        .WithKey("topselling")
                        .WithPriority(CacheItemPriority.High)
                        .WithAbsoluteExpirationRelativeToNow(TimeSpan.FromMinutes(10))
                        .WithValueOfType<List<Album>>()
                        .Passing(albums => albums.Count == totalAlbums)))
                .AndAlso()
                .ShouldReturn()
                .View(result => result
                    .WithModelOfType<List<Album>>()
                    .Passing(albums => albums.Count == totalAlbums));

        [Theory]
        [InlineData(6)]
        public void IndexShouldGetAlbumsFromCacheIfEntryExists(int totalAlbums)
            => MyController<HomeController>
                .Instance()
                .WithOptions(options => options
                    .For<AppSettings>(settings => settings.CacheDbResults = true))
                .WithMemoryCache(cache => cache
                    .WithEntry("topselling", AlbumData.Many.Take(totalAlbums).ToList()))
                .Calling(c => c.Index(
                    From.Services<MusicStoreContext>(),
                    From.Services<IMemoryCache>()))
                .ShouldReturn()
                .View(result => result
                    .WithCollectionModelOfType<Album>(albums => albums
                        .Count == 6));
    }
}
