<?php
/**
 * @package    JEM
 * @copyright  (C) 2013-2026 joomlaeventmanager.net
 * @copyright  (C) 2005-2009 Christoph Lukes
 * @license    https://www.gnu.org/licenses/gpl-3.0 GNU/GPL
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;

/**
 * Categories Model
 *
 */
class JemModelCategories extends ListModel
{
    /**
     * Constructor.
     *
     * @param  array  An optional associative array of configuration settings.
     * @see    JController
     */
    public function __construct($config = array())
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = array(
                'id', 'a.id',
                'title', 'a.catname',
                'alias', 'a.alias',
                'published', 'a.published',
                'access', 'a.access', 'access_level',
                'language', 'a.language',
                'checked_out', 'a.checked_out',
                'checked_out_time', 'a.checked_out_time',
                'created_time', 'a.created_time',
                'created_user_id', 'a.created_user_id',
                'lft', 'a.lft',
                'rgt', 'a.rgt',
                'level', 'a.level',
                'path', 'a.path',
            );
        }

        parent::__construct($config);
    }

    /**
     * Method to auto-populate the model state.
     *
     * Note. Calling getState in this method will result in recursion.
     *
     * @param  string  An optional ordering field.
     * @param  string  An optional direction (asc|desc).
     *
     * @return    void
     */
    protected function populateState($ordering = null, $direction = null)
    {
        // Initialise variables.
        $app = Factory::getApplication();
        $context = $this->context;

        $extension = $app->getUserStateFromRequest('com_jem.categories.filter.extension', 'extension', 'com_jem', 'cmd');

        $this->setState('filter.extension', $extension);
        $parts = explode('.', $extension);

        // extract the component name
        $this->setState('filter.component', $parts[0]);

        // extract the optional section name
        $this->setState('filter.section', (count($parts) > 1) ? $parts[1] : null);

        $search = $this->getUserStateFromRequest($context.'.search', 'filter_search');
        $this->setState('filter.search', $search);

        $level = $this->getUserStateFromRequest($context.'.filter.level', 'filter_level', 0, 'int');
        $this->setState('filter.level', $level);

        $access = $this->getUserStateFromRequest($context.'.filter.access', 'filter_access', 0, 'int');
        $this->setState('filter.access', $access);

        $published = $this->getUserStateFromRequest($context.'.filter.published', 'filter_published', '');
        $this->setState('filter.published', $published);

        $language = $this->getUserStateFromRequest($context.'.filter.language', 'filter_language', '');
        $this->setState('filter.language', $language);

        // List state information.
        parent::populateState('a.lft', 'asc');
    }

    /**
     * Method to get a store id based on model configuration state.
     *
     * This is necessary because the model is used by the component and
     * different modules that might need different sets of data or different
     * ordering requirements.
     *
     * @param  string  $id  A prefix for the store id.
     *
     * @return string  A store id.
     */
    protected function getStoreId($id = '')
    {
        // Compile the store id.
        $id .= ':'.$this->getState('filter.search');
        $id .= ':'.$this->getState('filter.extension');
        $id .= ':'.$this->getState('filter.published');
        $id .= ':'.$this->getState('filter.language');

        return parent::getStoreId($id);
    }

    /**
     * @return string
     */
    protected function getListQuery()
    {
        // Create a new query object.
        $db    = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);
        $user  = JemFactory::getUser();

        // Select the required fields from the table.
        $query->select(
            $this->getState(
                'list.select',
                'a.id, a.catname, a.color, a.alias, a.note, a.published, a.access' .
                ', a.checked_out, a.groupid, a.checked_out_time, a.created_user_id' .
                ', a.path, a.parent_id, a.level, a.lft, a.rgt' .
                ', a.language'
            )
        );
        $query->from('#__jem_categories AS a');

        // Join over the language
        $query->select('l.title AS language_title');
        $query->join('LEFT', $db->quoteName('#__languages').' AS l ON l.lang_code = a.language');

        // Join over the users for the checked out user.
        $query->select('uc.name AS editor');
        $query->join('LEFT', '#__users AS uc ON uc.id=a.checked_out');

        // Join over the asset groups.
        $query->select('ag.title AS access_level');
        $query->join('LEFT', '#__viewlevels AS ag ON ag.id = a.access');

        // Join over the users for the author.
        $query->select('ua.name AS author_name');
        $query->join('LEFT', '#__users AS ua ON ua.id = a.created_user_id');

        // Join over the groups
        $query->select('gr.name AS catgroup');
        $query->join('LEFT', '#__jem_groups AS gr ON gr.id = a.groupid');

        // Filter on the level.
        if ($level = $this->getState('filter.level')) {
            $query->where('a.level <= '.(int) $level);
        }

        // Filter by access level.
        if ($access = $this->getState('filter.access')) {
            $query->where('a.access = ' . (int) $access);
        }

        // Implement View Level Access
        if (!$user->authorise('core.admin'))
        {
            $groups = implode(',', $user->getAuthorisedViewLevels());
            $query->where('a.access IN ('.$groups.')');
        }

        // Filter by published state
        $published = $this->getState('filter.published');
        if (is_numeric($published)) {
            $query->where('a.published = ' . (int) $published);
        }
        elseif ($published === '') {
            $query->where('(a.published IN (0, 1))');
        }

        $query->where('(a.alias NOT LIKE "root")');

        // Filter by search in title
        $search = $this->getState('filter.search');
        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $query->where('a.id = '.(int) substr($search, 3));
            }
            elseif (stripos($search, 'author:') === 0) {
                $search = $db->Quote('%'.$db->escape(substr($search, 7), true).'%');
                $query->where('(ua.name LIKE '.$search.' OR ua.username LIKE '.$search.')');
            }
            else {
                $search = $db->Quote('%'.$db->escape($search, true).'%');
                $query->where('(a.title LIKE '.$search.' OR a.alias LIKE '.$search.' OR a.note LIKE '.$search.')');
            }
        }

        // Filter on the language.
        if ($language = $this->getState('filter.language')) {
            $query->where('a.language = '.$db->quote($language));
        }

        // Add the list ordering clause
        $listOrdering = $this->getState('list.ordering', 'a.lft');
        $listDirn = $db->escape($this->getState('list.direction', 'ASC'));
        if ($listOrdering == 'a.access') {
            $query->order('a.access '.$listDirn.', a.lft '.$listDirn);
        } else {
            $query->order($db->escape($listOrdering).' '.$listDirn);
        }

        //echo nl2br(str_replace('#__','jos_',$query));
        return $query;
    }

    /**
     *
     */
    public function getItems()
    {
        $items = parent::getItems();
        $app   = Factory::getApplication();

        foreach ($items as $item) {
            $item->assignedevents = $this->countCatEvents($item->id);
        }

        return $items;
    }

    private function countCatEvents($id)
    {
        $db = Factory::getContainer()->get('DatabaseDriver');
        $query = $db->getQuery(true);

        $query = 'SELECT COUNT(catid) as num'
                .' FROM #__jem_cats_event_relations'
                .' WHERE catid = '.(int)$id
                .' GROUP BY catid'
                ;

        $db->setQuery($query);
        $result = $db->loadResult('catid');

        return $result;
    }

}
