#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/inotify.h>
#include <pthread.h>
#include <errno.h>
#include <dirent.h>
#include <stdio.h>
#include <time.h>

#define PATH_MAX 4096
#define RETRY_DELAY_US 100000 // 100ms 重试间隔
#define MAX_RETRIES 5        // 5 次重试
#define CHECK_INTERVAL 45    // 每 45 秒检查一次新安装的应用

// 跳过路径列表
static const char *skipPaths[] = {
    "/acct", "/odm_dlkm", "/apex", "/oem", "/bin", "/opconfig", "/cache", "/opcust", "/config", "/postinstall",
    "/cust", "/proc", "/d/", "/product", "/data_mirror", "/second_stage_resources", "/debug_ramdisk", "/dev", "/sys",
    "/etc", "/system", "/linkerconfig", "/system_ext", "/lost+found", "/tmp", "/metadata", "/vendor", "/mnt", "/vendor_dlkm", "/odm"
};
static const int skipPathsCount = sizeof(skipPaths) / sizeof(skipPaths[0]);

static const char *skipFullPaths[] = {
    "/data/user/", "/data/user", "/data/media/0/Android/data/", "/data/media/0/Android/data", "/data/data/", "/data/data",
    "/data", "/data/", "/data/media/0", "/data/media/0/", "/data/media/0/Downloa", "/data/media/0/Download/", "/data/media/0/Android",
    "/data/media/0/Android/", "/sdcard", "/sdcard/", "/sdcard/Download", "/sdcard/Download/", "/sdcard/Android", "/sdcard/Android/",
    "/storage", "/storage/", "/storage/emulated/0", "/storage/emulated/0/", "/storage/emulated/0/Download", "/storage/emulated/0/Download/",
    "/storage/emulated/0/Android", "/storage/emulated/0/Android/", "/"
};
static const int skipFullPathsCount = sizeof(skipFullPaths) / sizeof(skipFullPaths[0]);

// 动态数组结构体
typedef struct {
    char **data;
    size_t size;
    size_t capacity;
} StringArray;

// 日志宏
#define LOG_INFO(fmt, ...) do { printf(fmt "\n", ##__VA_ARGS__); fflush(stdout); } while (0)
#define LOG_ERROR(fmt, ...) do { fprintf(stderr, fmt " (%s)\n", ##__VA_ARGS__, strerror(errno)); fflush(stderr); } while (0)

// 函数声明
static void initStringArray(StringArray *arr);
static void appendStringArray(StringArray *arr, const char *str);
static void freeStringArray(StringArray *arr);
static int shouldSkipPath(const char *path);
static StringArray readConfigFile(const char *configFile);
static char *getUserDataDir(void);
static char *getMediaDataDir(void);
static char *getParentDir(const char *path);
static char *getAppDir(const char *path);
static char *extractPackageName(const char *path);
static int isPackageInstalled(const char *packageName);
static int getPackageUidGid(const char *packageName, uid_t *uid, gid_t *gid);
static int setPathOwnership(const char *path, uid_t uid, gid_t gid);
static int createDirRecursively(const char *path, uid_t uid, gid_t gid);
static int createFile(const char *filename);
static int removePathRecursive(const char *path);
static void *monitorFile(void *arg);
static void *monitorManager(void *arg);

// 动态数组实现
static void initStringArray(StringArray *arr) {
    arr->size = 0;
    arr->capacity = 4;
    arr->data = malloc(arr->capacity * sizeof(char *));
}

static void appendStringArray(StringArray *arr, const char *str) {
    if (arr->size >= arr->capacity) {
        arr->capacity *= 2;
        arr->data = realloc(arr->data, arr->capacity * sizeof(char *));
    }
    arr->data[arr->size++] = strdup(str);
}

static void freeStringArray(StringArray *arr) {
    for (size_t i = 0; i < arr->size; i++) free(arr->data[i]);
    free(arr->data);
    arr->size = arr->capacity = 0;
}

// 是否跳过路径
static int shouldSkipPath(const char *path) {
    for (int i = 0; i < skipPathsCount; i++) {
        if (strncmp(path, skipPaths[i], strlen(skipPaths[i])) == 0) return 1;
    }
    for (int i = 0; i < skipFullPathsCount; i++) {
        if (strcmp(path, skipFullPaths[i]) == 0) return 1;
    }
    return 0;
}

// 读取配置文件
static StringArray readConfigFile(const char *configFile) {
    StringArray filePaths;
    initStringArray(&filePaths);

    FILE *file = fopen(configFile, "r");
    if (!file) {
        LOG_ERROR("无法打开配置文件: %s", configFile);
        return filePaths;
    }

    char line[PATH_MAX];
    while (fgets(line, sizeof(line), file)) {
        line[strcspn(line, "\n")] = 0;
        if (strlen(line) == 0 || line[0] == '#') continue;
        appendStringArray(&filePaths, line);
    }
    fclose(file);
    LOG_INFO("配置文件读取完成，路径数: %zu", filePaths.size);
    return filePaths;
}

// 获取用户数据目录
static char *getUserDataDir(void) {
    char *androidData = getenv("ANDROID_DATA");
    if (androidData) {
        char *userDir = malloc(strlen(androidData) + 8);
        snprintf(userDir, PATH_MAX, "%s/user/0", androidData);
        if (access(userDir, F_OK) == 0) return userDir;
        free(userDir);
    }
    char *defaultDir = strdup("/data/user/0");
    if (access(defaultDir, F_OK) == 0) return defaultDir;
    free(defaultDir);
    return strdup("/data/user");
}

// 获取媒体数据目录
static char *getMediaDataDir(void) {
    char *androidData = getenv("ANDROID_DATA");
    if (androidData) {
        char *mediaDir = malloc(strlen(androidData) + 9);
        snprintf(mediaDir, PATH_MAX, "%s/media/0", androidData);
        if (access(mediaDir, F_OK) == 0) return mediaDir;
        free(mediaDir);
    }
    char *defaultDir = strdup("/data/media/0");
    if (access(defaultDir, F_OK) == 0) return defaultDir;
    free(defaultDir);
    return strdup("/data/media");
}

// 获取父目录
static char *getParentDir(const char *path) {
    char *copy = strdup(path);
    char *lastSlash = strrchr(copy, '/');
    if (lastSlash && lastSlash != copy) {
        *lastSlash = '\0';
    } else {
        free(copy);
        return strdup("/");
    }
    return copy;
}

// 获取应用目录
static char *getAppDir(const char *path) {
    char *copy = strdup(path);
    char *userDataDir = getUserDataDir();
    char *mediaDataDir = getMediaDataDir();
    char *pos;

    if ((pos = strstr(copy, userDataDir)) == copy) {
        pos += strlen(userDataDir) + 1;
        char *nextSlash = strchr(pos, '/');
        if (nextSlash) {
            *nextSlash = '\0';
            char *appDir = malloc(strlen(userDataDir) + strlen(pos) + 2);
            snprintf(appDir, PATH_MAX, "%s/%s", userDataDir, pos);
            free(copy);
            free(userDataDir);
            free(mediaDataDir);
            return appDir;
        }
    } else if ((pos = strstr(copy, mediaDataDir)) == copy) {
        pos += strlen(mediaDataDir) + strlen("/Android/data/");
        char *nextSlash = strchr(pos, '/');
        if (nextSlash) {
            *nextSlash = '\0';
            char *appDir = malloc(strlen(mediaDataDir) + strlen("/Android/data/") + strlen(pos) + 2);
            snprintf(appDir, PATH_MAX, "%s/Android/data/%s", mediaDataDir, pos);
            free(copy);
            free(userDataDir);
            free(mediaDataDir);
            return appDir;
        }
    }

    free(copy);
    free(mediaDataDir);
    char *fallback = malloc(strlen(userDataDir) + 2);
    snprintf(fallback, PATH_MAX, "%s", userDataDir);
    free(userDataDir);
    return fallback;
}

// 从路径提取包名
static char *extractPackageName(const char *path) {
    char *copy = strdup(path);
    char *userDataDir = getUserDataDir();
    char *mediaDataDir = getMediaDataDir();
    char *pos;

    if ((pos = strstr(copy, userDataDir)) == copy) {
        pos += strlen(userDataDir) + 1;
        char *nextSlash = strchr(pos, '/');
        if (nextSlash) {
            *nextSlash = '\0';
            char *packageName = strdup(pos);
            free(copy);
            free(userDataDir);
            free(mediaDataDir);
            return packageName;
        }
    } else if ((pos = strstr(copy, mediaDataDir)) == copy) {
        pos += strlen(mediaDataDir) + strlen("/Android/data/");
        char *nextSlash = strchr(pos, '/');
        if (nextSlash) {
            *nextSlash = '\0';
            char *packageName = strdup(pos);
            free(copy);
            free(userDataDir);
            free(mediaDataDir);
            return packageName;
        }
    }

    free(copy);
    free(userDataDir);
    free(mediaDataDir);
    return NULL;
}

// 检查包是否已安装
static int isPackageInstalled(const char *packageName) {
    FILE *file = fopen("/data/system/packages.list", "r");
    if (!file) {
        LOG_ERROR("无法打开 packages.list");
        return 1; // 默认假设已安装
    }

    char line[PATH_MAX];
    int installed = 0;
    while (fgets(line, sizeof(line), file)) {
        char pkg[PATH_MAX];
        if (sscanf(line, "%s", pkg) == 1 && strcmp(pkg, packageName) == 0) {
            installed = 1;
            break;
        }
    }
    fclose(file);
    return installed;
}

// 从 packages.list 获取应用的 UID 和 GID
static int getPackageUidGid(const char *packageName, uid_t *uid, gid_t *gid) {
    FILE *file = fopen("/data/system/packages.list", "r");
    if (!file) {
        LOG_ERROR("无法打开 packages.list");
        return -1;
    }

    char line[PATH_MAX];
    int found = 0;
    while (fgets(line, sizeof(line), file)) {
        char pkg[PATH_MAX];
        uid_t pkg_uid;
        if (sscanf(line, "%s %u", pkg, &pkg_uid) == 2 && strcmp(pkg, packageName) == 0) {
            *uid = pkg_uid;
            *gid = pkg_uid; // GID 通常与 UID 相同
            found = 1;
//            LOG_INFO("从packages.list提取: %s", line);
            break;
        }
    }
    fclose(file);

    if (!found) {
        LOG_ERROR("未在packages.list中找到包: %s", packageName);
        return -1;
    }
    return 0;
}

// 递归设置路径的 UID 和 GID
static int setPathOwnership(const char *path, uid_t uid, gid_t gid) {
    struct stat st;
    if (lstat(path, &st) == -1) {
        if (errno == ENOENT) return 0; // 路径不存在，跳过
        LOG_ERROR("无法访问路径: %s", path);
        return -1;
    }

    if (lchown(path, uid, gid) != 0) {
        LOG_ERROR("无法设置路径所有者: %s", path);
        return -1;
    }
//    LOG_INFO("路径所有者设置为: UID=%d, GID=%d: %s", uid, gid, path);

    if (S_ISDIR(st.st_mode)) {
        DIR *dir = opendir(path);
        if (!dir) {
            LOG_ERROR("无法打开目录: %s", path);
            return -1;
        }

        struct dirent *entry;
        while ((entry = readdir(dir)) != NULL) {
            if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0) continue;
            char subPath[PATH_MAX];
            snprintf(subPath, sizeof(subPath), "%s/%s", path, entry->d_name);
            if (setPathOwnership(subPath, uid, gid) != 0) {
                closedir(dir);
                return -1;
            }
        }
        closedir(dir);
    }
    return 0;
}

// 递归创建目录
static int createDirRecursively(const char *path, uid_t uid, gid_t gid) {
    char temp[PATH_MAX];
    snprintf(temp, sizeof(temp), "%s", path);

    char *pos;
    for (pos = temp + 1; *pos; pos++) {
        if (*pos == '/') {
            *pos = '\0';
            struct stat st;
            if (stat(temp, &st) != 0) {
                if (mkdir(temp, 0755) && errno != EEXIST) {
                    LOG_ERROR("无法创建文件夹: %s", temp);
                    return -1;
                }
                if (lchown(temp, uid, gid) != 0) {
                    LOG_ERROR("无法设置目录所有者: %s", temp);
                }
            }
            *pos = '/';
        }
    }
    struct stat st;
    if (stat(temp, &st) != 0) {
        if (mkdir(temp, 0755) && errno != EEXIST) {
            LOG_ERROR("无法创建文件夹: %s", temp);
            return -1;
        }
        if (lchown(temp, uid, gid) != 0) {
            LOG_ERROR("无法设置目录所有者: %s", temp);
        }
    }
    return 0;
}

// 创建文件
static int createFile(const char *filename) {
    LOG_INFO("尝试创建文件: %s", filename);

    char *parentDir = getParentDir(filename);
    char *packageName = extractPackageName(filename);

    if (!packageName) {
        LOG_ERROR("无法提取包名: %s，跳过创建", filename);
        free(parentDir);
        return -1;
    }

    uid_t uid, gid;
    if (getPackageUidGid(packageName, &uid, &gid) != 0) {
        LOG_ERROR("应用未安装，跳过创建: %s", packageName);
        free(parentDir);
        free(packageName);
        return -1;
    }
    LOG_INFO("从packages.list获取UID/GID: UID=%d, GID=%d for %s", uid, gid, packageName);

    // 设置应用目录权限
    char *appDir = getAppDir(filename);
    if (setPathOwnership(appDir, uid, gid) != 0) {
        LOG_ERROR("无法设置应用目录权限: %s", appDir);
    }
    free(appDir);

    if (createDirRecursively(parentDir, uid, gid) != 0) {
        LOG_ERROR("无法创建父目录: %s", parentDir);
        free(parentDir);
        free(packageName);
        return -1;
    }

    struct stat st;
    if (stat(filename, &st) == 0) {
        if (S_ISDIR(st.st_mode)) {
            LOG_INFO("目标路径是文件夹，删除: %s", filename);
            if (removePathRecursive(filename) != 0) {
                LOG_ERROR("无法删除目标文件夹: %s", filename);
                free(parentDir);
                free(packageName);
                return -1;
            }
        } else {
            LOG_INFO("目标路径是文件，删除: %s", filename);
            if (unlink(filename) != 0) {
                LOG_ERROR("无法删除目标文件: %s", filename);
                free(parentDir);
                free(packageName);
                return -1;
            }
        }
    }

    FILE *file = NULL;
    for (int attempts = 0; attempts < MAX_RETRIES && !file; attempts++) {
        file = fopen(filename, "w");
        if (!file) {
            if (errno == EISDIR) {
                LOG_INFO("目标路径变为文件夹，删除: %s", filename);
                if (removePathRecursive(filename) != 0) {
                    LOG_ERROR("无法删除目标文件夹: %s", filename);
                    break;
                }
            }
            if (attempts < MAX_RETRIES - 1) usleep(RETRY_DELAY_US);
        }
    }
    if (!file) {
        LOG_ERROR("多次尝试后仍无法创建文件: %s", filename);
        free(parentDir);
        free(packageName);
        return -1;
    }

    fclose(file);
    LOG_INFO("文件已创建: %s", filename);

    if (lchown(filename, uid, gid) != 0) {
        LOG_ERROR("无法设置文件所有者: %s", filename);
    } else {
        LOG_INFO("文件所有者设置为: UID=%d, GID=%d: %s", uid, gid, filename);
    }

    if (chmod(filename, 0600) != 0) {
        LOG_ERROR("无法设置文件权限为0600: %s", filename);
    } else {
        LOG_INFO("文件权限已设置为0600: %s", filename);
    }

    free(parentDir);
    free(packageName);
    return 0;
}

// 递归删除路径
static int removePathRecursive(const char *path) {
    struct stat st;
    if (lstat(path, &st) == -1) {
        if (errno == ENOENT) return 0;
        LOG_ERROR("无法访问路径: %s", path);
        return -1;
    }

    if (S_ISDIR(st.st_mode)) {
        DIR *dir = opendir(path);
        if (!dir) {
            LOG_ERROR("无法打开目录: %s", path);
            return -1;
        }

        struct dirent *entry;
        while ((entry = readdir(dir)) != NULL) {
            if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0) continue;
            char subPath[PATH_MAX];
            snprintf(subPath, sizeof(subPath), "%s/%s", path, entry->d_name);
            if (removePathRecursive(subPath) != 0) {
                closedir(dir);
                return -1;
            }
        }
        closedir(dir);
        if (rmdir(path) != 0) {
            LOG_ERROR("无法删除目录: %s", path);
            return -1;
        }
    } else {
        if (unlink(path) != 0) {
            LOG_ERROR("无法删除文件: %s", path);
            return -1;
        }
    }

    LOG_INFO("路径已删除: %s", path);
    return 0;
}

// 文件监控线程
static void *monitorFile(void *arg) {
    char *filename = (char *)arg;
    LOG_INFO("开始监控文件: %s", filename);

    char *packageName = extractPackageName(filename);
    if (!packageName) {
        LOG_ERROR("无法从路径提取包名: %s", filename);
        free(filename);
        return NULL;
    }

    int inotifyFd = inotify_init();
    if (inotifyFd == -1) {
        LOG_ERROR("无法初始化 inotify: %s", filename);
        free(packageName);
        free(filename);
        return NULL;
    }

    int wd = inotify_add_watch(inotifyFd, filename, IN_DELETE_SELF | IN_MODIFY | IN_MOVE_SELF | IN_DELETE | IN_CREATE);
    if (wd == -1) {
        LOG_ERROR("无法添加 inotify 监控: %s", filename);
        close(inotifyFd);
        free(packageName);
        free(filename);
        return NULL;
    }

    char buf[4096] __attribute__((aligned(__alignof__(struct inotify_event))));
    ssize_t numRead;

    while (1) {
        numRead = read(inotifyFd, buf, sizeof(buf));
        if (numRead <= 0) {
            LOG_ERROR("inotify 读取失败或关闭: %s", filename);
            if (!isPackageInstalled(packageName)) {
                LOG_INFO("应用已卸载，清理并退出监控: %s", packageName);
                char *appDir = getAppDir(filename);
                removePathRecursive(appDir); // 清理残留目录
                free(appDir);
                break;
            }
            continue;
        }

        for (char *ptr = buf; ptr < buf + numRead;) {
            struct inotify_event *event = (struct inotify_event *)ptr;
            LOG_INFO("检测到文件事件: %s (mask: %u)", filename, event->mask);

            if (event->mask & (IN_DELETE_SELF | IN_MODIFY | IN_MOVE_SELF | IN_DELETE | IN_CREATE)) {
                char *parentDir = getParentDir(filename);
                struct stat st;
                if (stat(filename, &st) != 0 || S_ISDIR(st.st_mode)) {
                    LOG_INFO("文件不存在或变为文件夹，开始重建: %s", filename);
                    removePathRecursive(filename);
                    if (createDirRecursively(parentDir, 1000, 1000) == 0) {
                        if (createFile(filename) != 0) {
                            LOG_ERROR("文件重建失败: %s", filename);
                        }
                    }
                }
                free(parentDir);

                inotify_rm_watch(inotifyFd, wd);
                wd = inotify_add_watch(inotifyFd, filename, IN_DELETE_SELF | IN_MODIFY | IN_MOVE_SELF | IN_DELETE | IN_CREATE);
                if (wd == -1) {
                    LOG_ERROR("无法重新添加 inotify 监控: %s", filename);
                    usleep(RETRY_DELAY_US);
                    if (createFile(filename) == 0) {
                        wd = inotify_add_watch(inotifyFd, filename, IN_DELETE_SELF | IN_MODIFY | IN_MOVE_SELF | IN_DELETE | IN_CREATE);
                        if (wd != -1) {
                            LOG_INFO("成功重新添加监控: %s", filename);
                        }
                    }
                } else {
                    LOG_INFO("成功重新添加监控: %s", filename);
                }
            }
            ptr += sizeof(struct inotify_event) + event->len;
        }

        if (!isPackageInstalled(packageName)) {
            LOG_INFO("应用已卸载，清理并退出监控: %s", packageName);
            char *appDir = getAppDir(filename);
            removePathRecursive(appDir); // 清理残留目录
            free(appDir);
            break;
        }
    }

    close(inotifyFd);
    free(packageName);
    free(filename);
    LOG_INFO("监控线程结束: %s", filename);
    return NULL;
}

// 监控管理线程，周期性检查新安装的应用
static void *monitorManager(void *arg) {
    StringArray *filePaths = (StringArray *)arg;
    pthread_t *threads = malloc(filePaths->size * sizeof(pthread_t));
    int *active = calloc(filePaths->size, sizeof(int)); // 标记每个路径是否已有线程

    // 初始启动所有线程（仅限已安装的应用）
    for (size_t i = 0; i < filePaths->size; i++) {
        if (shouldSkipPath(filePaths->data[i])) continue;
        char *packageName = extractPackageName(filePaths->data[i]);
        if (packageName && isPackageInstalled(packageName)) {
            char *filePathCopy = strdup(filePaths->data[i]);
            if (pthread_create(&threads[i], NULL, monitorFile, filePathCopy) == 0) {
                active[i] = 1;
                LOG_INFO("初始线程创建成功，路径: %s", filePaths->data[i]);
            } else {
                LOG_ERROR("初始线程创建失败: %s", filePaths->data[i]);
                free(filePathCopy);
            }
        } else {
            LOG_INFO("应用未安装，跳过初始线程创建: %s", filePaths->data[i]);
        }
        if (packageName) free(packageName);
    }

    // 周期性检查
    while (1) {
        sleep(CHECK_INTERVAL);
        for (size_t i = 0; i < filePaths->size; i++) {
            if (shouldSkipPath(filePaths->data[i])) continue;
            char *packageName = extractPackageName(filePaths->data[i]);
            if (!active[i] && packageName && isPackageInstalled(packageName)) {
                char *filePathCopy = strdup(filePaths->data[i]);
                if (pthread_create(&threads[i], NULL, monitorFile, filePathCopy) == 0) {
                    active[i] = 1;
                    LOG_INFO("检测到新安装应用，启动监控线程: %s", filePaths->data[i]);
                } else {
                    LOG_ERROR("线程创建失败: %s", filePaths->data[i]);
                    free(filePathCopy);
                }
            }
            if (packageName) free(packageName);
        }
    }

    free(threads);
    free(active);
    return NULL;
}

// 主函数
int main(int argc, char *argv[]) {
    char *configFile = NULL;
    int opt;

    while ((opt = getopt(argc, argv, "f:")) != -1) {
        switch (opt) {
            case 'f': configFile = optarg; break;
            default:
                LOG_ERROR("用法: %s -f <配置文件路径>", argv[0]);
                return 1;
        }
    }

    if (!configFile) {
        LOG_ERROR("配置文件路径不能为空");
        return 1;
    }

    StringArray filePaths = readConfigFile(configFile);
    if (filePaths.size == 0) {
        LOG_ERROR("无法读取配置文件或文件为空: %s", configFile);
        return 1;
    }

    // 初始化路径（仅限已安装的应用）
    for (size_t i = 0; i < filePaths.size; i++) {
        const char *filePath = filePaths.data[i];
        LOG_INFO("初始化路径: %s", filePath);

        if (shouldSkipPath(filePath)) {
            LOG_INFO("跳过路径: %s", filePath);
            continue;
        }

        char *packageName = extractPackageName(filePath);
        uid_t uid, gid;
        if (!packageName || getPackageUidGid(packageName, &uid, &gid) != 0) {
            LOG_INFO("应用未安装，跳过初始化: %s", filePath);
            if (packageName) free(packageName);
            continue;
        }
    //    LOG_INFO("初始化时从packages.list获取UID/GID: UID=%d, GID=%d for %s", uid, gid, packageName);

        char *appDir = getAppDir(filePath);
        setPathOwnership(appDir, uid, gid); // 设置应用目录权限
        free(appDir);

        char *parentDir = getParentDir(filePath);
        removePathRecursive(filePath);
        createDirRecursively(parentDir, uid, gid);
        createFile(filePath);
        free(parentDir);
        free(packageName);
    }

    // 启动监控管理线程
    pthread_t managerThread;
    if (pthread_create(&managerThread, NULL, monitorManager, &filePaths) != 0) {
        LOG_ERROR("无法创建监控管理线程");
        freeStringArray(&filePaths);
        return 1;
    }

    pthread_join(managerThread, NULL);
    freeStringArray(&filePaths);
    LOG_INFO("程序执行完成");
    return 0;
}