#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

// 定义动态字符串数组结构体
typedef struct {
    char** data;
    size_t size;
    size_t capacity;
} StringArray;

// 初始化动态字符串数组
void initStringArray(StringArray* arr) {
    arr->size = 0;
    arr->capacity = 4;
    arr->data = (char**)malloc(arr->capacity * sizeof(char*));
}

// 向动态字符串数组添加字符串
void appendStringArray(StringArray* arr, const char* str) {
    if (arr->size >= arr->capacity) {
        arr->capacity *= 2;
        arr->data = (char**)realloc(arr->data, arr->capacity * sizeof(char*));
    }
    arr->data[arr->size] = strdup(str);
    arr->size++;
}

// 释放动态字符串数组内存
void freeStringArray(StringArray* arr) {
    for (size_t i = 0; i < arr->size; i++) {
        free(arr->data[i]);
    }
    free(arr->data);
    arr->size = 0;
    arr->capacity = 0;
}

// 执行系统命令并同步获取输出
void execCmdSync(char* output, size_t output_size, const char* cmd, const char* arg1, const char* arg2, const char* arg3) {
    char command[1024] = {0};
    snprintf(command, sizeof(command), "%s", cmd);
    if (arg1) {
        strcat(command, " ");
        strcat(command, arg1);
    }
    if (arg2) {
        strcat(command, " ");
        strcat(command, arg2);
    }
    if (arg3) {
        strcat(command, " ");
        strcat(command, arg3);
    }

    FILE* pipe = popen(command, "r");
    if (!pipe) {
        if (output) output[0] = '\0';
        return;
    }

    char buffer[128];
    if (output) {
        output[0] = '\0';
        while (fgets(buffer, sizeof(buffer), pipe) != NULL) {
            strncat(output, buffer, output_size - strlen(output) - 1);
        }
    }
    pclose(pipe);
}

// 直接从管道读取 dumpsys 输出
char* execCmdToPipe(const char* cmd) {
    FILE* pipe = popen(cmd, "r");
    if (!pipe) return NULL;

    char* buffer = NULL;
    size_t buffer_size = 0;
    size_t total_size = 0;
    char temp[4096];

    while (fgets(temp, sizeof(temp), pipe) != NULL) {
        size_t len = strlen(temp);
        if (total_size + len + 1 > buffer_size) {
            buffer_size += 4096; // 每次扩展 4KB
            char* new_buffer = realloc(buffer, buffer_size);
            if (!new_buffer) {
                free(buffer);
                pclose(pipe);
                return NULL;
            }
            buffer = new_buffer;
        }
        memcpy(buffer + total_size, temp, len);
        total_size += len;
    }

    if (buffer) buffer[total_size] = '\0';
    pclose(pipe);
    return buffer;
}

// 方法1：使用 realActivity 获取包名（Android 9 及以下）
char* getTopAppNameRealActivity(char* buf, size_t buf_size) {
    execCmdSync(buf, buf_size, "/system/bin/dumpsys", "activity", "activities", NULL);
    if (strlen(buf) == 0) {
        return NULL;
    }

    const char* KEYWORD = "realActivity=";
    char* topAppName = strstr(buf, KEYWORD);
    if (!topAppName) {
        return NULL;
    }
    topAppName += strlen(KEYWORD); // 移动到包名开始位置

    char* end = strchr(topAppName, '/');
    if (!end) {
        return NULL;
    }
    *end = '\0'; // 截断到包名部分
    return topAppName;
}

// 方法2：使用 mActivityComponent 获取包名（Android 10+）
char* getTopAppNameActivityComponent(char* buf, size_t buf_size) {
    execCmdSync(buf, buf_size, "/system/bin/dumpsys", "activity", "activities", NULL);
    if (strlen(buf) == 0) {
        return NULL;
    }

    const char* KEYWORD = "mActivityComponent=";
    char* topAppName = strstr(buf, KEYWORD);
    if (!topAppName) {
        return NULL;
    }
    topAppName += strlen(KEYWORD); // 移动到包名开始位置

    char* end = strchr(topAppName, '/');
    if (!end) {
        return NULL;
    }
    *end = '\0'; // 截断到包名部分
    return topAppName;
}

// 方法3：使用 dumpsys activity lru 获取 TOP（Android 10+）
char* getTopAppNameLru(char* buf, size_t buf_size) {
    execCmdSync(buf, buf_size, "/system/bin/dumpsys", "activity", "lru", NULL);
    if (strlen(buf) == 0) {
        return NULL;
    }

    const char* TA_KEYWORD = " TOP";
    char* topAppName = strstr(buf, TA_KEYWORD);
    if (!topAppName) {
        return NULL;
    }

    while (topAppName < buf + strlen(buf) && *topAppName != ':') {
        topAppName++; // 跳到包名开始
    }
    topAppName++;

    char* end = strpbrk(topAppName, "/:");
    if (!end) {
        return NULL;
    }
    *end = '\0';
    return topAppName;
}

// 方法4：使用 dumpsys window 获取 mCurrentFocus 或 mResumeActivity
char* getTopAppNameWindow(char* buf, size_t buf_size) {
    // 优先尝试 dumpsys window 中的 mCurrentFocus
    char* tempBuf = execCmdToPipe("/system/bin/dumpsys window");
    if (tempBuf) {
        const char* FOCUS_KEYWORD = "mCurrentFocus=";
        char* topAppName = strstr(tempBuf, FOCUS_KEYWORD);
        if (topAppName) {
            topAppName += strlen(FOCUS_KEYWORD);
            while (*topAppName && (*topAppName == ' ' || *topAppName == '{')) topAppName++;
            // 查找 "u" 后跟数字的模式，例如 u0, u10, u999
            char* start = topAppName;
            while (start && *start) {
                if (*start == 'u' && isdigit(*(start + 1))) {
                    start++; // 跳过 'u'
                    while (isdigit(*start)) start++; // 跳过数字
                    if (*start == ' ') {
                        start++; // 跳过空格
                        break;
                    }
                }
                start++;
            }
            if (start && *start) {
                char* end = strchr(start, '/');
                if (!end) end = strchr(start, ' ');
                if (end) {
                    size_t len = end - start;
                    if (len < buf_size) {
                        strncpy(buf, start, len);
                        buf[len] = '\0';
                        free(tempBuf);
                        return buf;
                    }
                }
            }
        }
        free(tempBuf);
    }

    // 如果失败，回退到 dumpsys window windows 中的 mResumeActivity
    tempBuf = execCmdToPipe("/system/bin/dumpsys window windows");
    if (tempBuf) {
        const char* RESUME_KEYWORD = "mResumeActivity:";
        char* resumeApp = strstr(tempBuf, RESUME_KEYWORD);
        if (resumeApp) {
            resumeApp += strlen(RESUME_KEYWORD);
            while (*resumeApp && (*resumeApp == ' ' || *resumeApp == '{')) resumeApp++;
            // 同样支持多用户 ID
            char* start = resumeApp;
            while (start && *start) {
                if (*start == 'u' && isdigit(*(start + 1))) {
                    start++; // 跳过 'u'
                    while (isdigit(*start)) start++; // 跳过数字
                    if (*start == ' ') {
                        start++; // 跳过空格
                        break;
                    }
                }
                start++;
            }
            if (start && *start) {
                char* end = strchr(start, '/');
                if (!end) end = strchr(start, ' ');
                if (end) {
                    size_t len = end - start;
                    if (len < buf_size) {
                        strncpy(buf, start, len);
                        buf[len] = '\0';
                        free(tempBuf);
                        return buf;
                    }
                }
            }
        }
        free(tempBuf);
    }

    return NULL;
}

// 获取前台应用包名，依次尝试多种方法
char* getTopAppName(char* buf, size_t buf_size) {
    char* topAppName = getTopAppNameRealActivity(buf, buf_size);
    if (topAppName && strlen(topAppName) > 0) {
        return topAppName;
    }
    topAppName = getTopAppNameActivityComponent(buf, buf_size);
    if (topAppName && strlen(topAppName) > 0) {
        return topAppName;
    }
    topAppName = getTopAppNameLru(buf, buf_size);
    if (topAppName && strlen(topAppName) > 0) {
        return topAppName;
    }
    return getTopAppNameWindow(buf, buf_size);
}

// 检查字符串是否在数组中
int containsString(StringArray* arr, const char* str) {
    for (size_t i = 0; i < arr->size; i++) {
        if (strcmp(arr->data[i], str) == 0) {
            return 1;
        }
    }
    return 0;
}

// 根据前台应用包名管理 hosts 文件挂载
void manageHostsFile(const char* topAppName, StringArray* appNames, const char* targetHostsFile, const char* systemHostsFile, const char* targetHostsRewardFile) {
    static int isRewardFileMounted = 0;
    static int isTargetFileMounted = 0;

    if (containsString(appNames, topAppName)) {
        if (!isRewardFileMounted) {
            printf("挂载奖励hosts文件: %s 到 %s\n", targetHostsRewardFile, systemHostsFile);
            fflush(stdout);
            execCmdSync(NULL, 0, "/system/bin/umount", systemHostsFile, NULL, NULL);
            execCmdSync(NULL, 0, "/system/bin/mount", "--bind", targetHostsRewardFile, systemHostsFile);
            isRewardFileMounted = 1;
            isTargetFileMounted = 0;
        }
    } else {
        if (!isTargetFileMounted) {
            printf("挂载目标hosts文件: %s 到 %s\n", targetHostsFile, systemHostsFile);
            fflush(stdout);
            execCmdSync(NULL, 0, "/system/bin/umount", systemHostsFile, NULL, NULL);
            execCmdSync(NULL, 0, "/system/bin/mount", "--bind", targetHostsFile, systemHostsFile);
            isTargetFileMounted = 1;
            isRewardFileMounted = 0;
        }
    }
}

// 读取配置文件
StringArray readConfigFile(const char* configFilePath, char* targetHostsFile, char* systemHostsFile, char* targetHostsRewardFile) {
    StringArray appNames;
    initStringArray(&appNames);

    FILE* configFile = fopen(configFilePath, "r");
    if (!configFile) {
        return appNames;
    }

    char line[256];
    while (fgets(line, sizeof(line), configFile)) {
        line[strcspn(line, "\n")] = 0;
        if (strlen(line) == 0 || line[0] == '#') {
            continue;
        }
        if (strncmp(line, "target_hosts_file=", 18) == 0) {
            strcpy(targetHostsFile, line + 18);
            continue;
        }
        if (strncmp(line, "system_hosts_file=", 18) == 0) {
            strcpy(systemHostsFile, line + 18);
            continue;
        }
        if (strncmp(line, "target_hosts_reward_file=", 25) == 0) {
            strcpy(targetHostsRewardFile, line + 25);
            continue;
        }
        appendStringArray(&appNames, line);
    }
    fclose(configFile);
    return appNames;
}

int main(int argc, char* argv[]) {
    char* configFilePath = NULL;
    for (int i = 1; i < argc; ++i) {
        if (strcmp(argv[i], "-f") == 0 && i + 1 < argc) {
            configFilePath = argv[++i];
        }
    }

    if (!configFilePath) {
        fprintf(stderr, "用法: %s -f <配置文件路径>\n", argv[0]);
        fflush(stderr);
        return 1;
    }

    char targetHostsFile[256] = "/data/adb/modules/GGAT_10007/system/etc/hosts";
    char systemHostsFile[256] = "/system/etc/hosts";
    char targetHostsRewardFile[256] = {0};

    StringArray appNames = readConfigFile(configFilePath, targetHostsFile, systemHostsFile, targetHostsRewardFile);

    printf("预先挂载目标hosts文件: %s 到 %s\n", targetHostsFile, systemHostsFile);
    fflush(stdout);
    execCmdSync(NULL, 0, "/system/bin/mount", "--bind", targetHostsFile, systemHostsFile);

    char buf[4096];
    while (1) {
        freeStringArray(&appNames);
        appNames = readConfigFile(configFilePath, targetHostsFile, systemHostsFile, targetHostsRewardFile);

        char* topAppName = getTopAppName(buf, sizeof(buf));
        if (topAppName && strlen(topAppName) > 0) {
            printf("当前前台应用包名: %s\n", topAppName);
            fflush(stdout);
            manageHostsFile(topAppName, &appNames, targetHostsFile, systemHostsFile, targetHostsRewardFile);
        } else {
            fprintf(stderr, "无法获取前台应用包名\n");
            fflush(stderr);
        }

        sleep(5);
    }

    freeStringArray(&appNames);
    return 0;
}