#include <fstream>
#include <string>
#include <vector>
#include <algorithm>
#include <iostream>
#include <cstring>
#include <thread>
#include <chrono>

// 执行系统命令并同步获取输出
void ExecCmdSync(std::string* output, const char* cmd, const char* arg1 = nullptr, const char* arg2 = nullptr, const char* arg3 = nullptr) {
    std::string command = cmd;
    if (arg1) command += " " + std::string(arg1);
    if (arg2) command += " " + std::string(arg2);
    if (arg3) command += " " + std::string(arg3);

    FILE* pipe = popen(command.c_str(), "r");
    if (!pipe) {
        if (output) *output = "";
        return;
    }

    char buffer[128];
    if (output) {
        while (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
            *output += buffer;
        }
    }
    pclose(pipe);
}

// 读取配置文件，返回应用包名列表，并获取hosts文件路径
std::vector<std::string> ReadConfigFile(const std::string& configFilePath, std::string& targetHostsFile, std::string& systemHostsFile, std::string& targetHostsRewardFile) {
    std::vector<std::string> appNames;
    std::ifstream configFile(configFilePath);
    std::string line;
    while (std::getline(configFile, line)) {
        if (line.empty() || line[0] == '#') {
            continue; // 跳过注释行和空行
        }
        if (line.find("target_hosts_file=") == 0) {
            targetHostsFile = line.substr(strlen("target_hosts_file="));
            continue;
        }
        if (line.find("system_hosts_file=") == 0) {
            systemHostsFile = line.substr(strlen("system_hosts_file="));
            continue;
        }
        if (line.find("target_hosts_reward_file=") == 0) {
            targetHostsRewardFile = line.substr(strlen("target_hosts_reward_file="));
            continue;
        }
        appNames.push_back(line); // 添加应用包名到列表
    }
    return appNames;
}

// 获取当前前台应用的包名
std::string GetTopAppName() {
    std::string buf;
    ExecCmdSync(&buf, "/system/bin/dumpsys", "activity", "o");
    if (buf.empty()) {
        return {};
    }

    constexpr char TA_KEYWORD[] = " (top-activity)";
    char *topappName = strstr(buf.data(), TA_KEYWORD);
    if (topappName == nullptr) {
        return {};
    }
    topappName--;
    while (topappName > buf.data() && *topappName != ' ') {
        topappName--;
    }
    while (topappName < buf.data() + buf.size() && *topappName != ':') {
        topappName++;
    }
    topappName++;

    char *end = std::min(strchr(topappName, '/'), strchr(topappName, ':'));
    if (end == nullptr) {
        return {};
    }
    *end = '\0';
    return topappName;
}

// 根据前台应用包名管理hosts文件的挂载
void ManageHostsFile(const std::string& topAppName, const std::vector<std::string>& appNames, const std::string& targetHostsFile, const std::string& systemHostsFile, const std::string& targetHostsRewardFile) {
    static bool isRewardFileMounted = false;
    static bool isTargetFileMounted = false;

    if (std::find(appNames.begin(), appNames.end(), topAppName) != appNames.end()) {
        if (!isRewardFileMounted) {
            // 挂载奖励hosts文件
            std::cout << "挂载奖励hosts文件: " << targetHostsRewardFile << " 到 " << systemHostsFile << std::endl;
            ExecCmdSync(nullptr, "/system/bin/umount", systemHostsFile.c_str());
            ExecCmdSync(nullptr, "/system/bin/mount", "--bind", targetHostsRewardFile.c_str(), systemHostsFile.c_str());
            isRewardFileMounted = true;
            isTargetFileMounted = false;
        }
    } else {
        if (!isTargetFileMounted) {
            // 挂载目标hosts文件
            std::cout << "挂载目标hosts文件: " << targetHostsFile << " 到 " << systemHostsFile << std::endl;
            ExecCmdSync(nullptr, "/system/bin/umount", systemHostsFile.c_str());
            ExecCmdSync(nullptr, "/system/bin/mount", "--bind", targetHostsFile.c_str(), systemHostsFile.c_str());
            isTargetFileMounted = true;
            isRewardFileMounted = false;
        }
    }
}

int main(int argc, char* argv[]) {
    std::string configFilePath;
    for (int i = 1; i < argc; ++i) {
        if (std::strcmp(argv[i], "-f") == 0 && i + 1 < argc) {
            configFilePath = argv[++i];
        }
    }

    if (configFilePath.empty()) {
        std::cerr << "用法: " << argv[0] << " -f <配置文件路径>" << std::endl;
        return 1;
    }

    std::string targetHostsFile = "/data/adb/modules/GGAT_10007/system/etc/hosts";
    std::string systemHostsFile = "/system/etc/hosts";
    std::string targetHostsRewardFile;

    std::vector<std::string> appNames = ReadConfigFile(configFilePath, targetHostsFile, systemHostsFile, targetHostsRewardFile);

    // 预先挂载目标hosts文件
    std::cout << "预先挂载目标hosts文件: " << targetHostsFile << " 到 " << systemHostsFile << std::endl;
    ExecCmdSync(nullptr, "/system/bin/mount", "--bind", targetHostsFile.c_str(), systemHostsFile.c_str());

    while (true) {
        std::string topAppName = GetTopAppName();

        if (!topAppName.empty()) {
            std::cout << "当前前台应用包名: " << topAppName << std::endl;
            ManageHostsFile(topAppName, appNames, targetHostsFile, systemHostsFile, targetHostsRewardFile);
        } else {
            std::cerr << "无法获取前台应用包名" << std::endl;
        }

        std::this_thread::sleep_for(std::chrono::seconds(5)); // 每5秒获取一次前台应用包名
    }

    return 0;
}
