#include <iostream>
#include <fstream>
#include <filesystem>
#include <thread>
#include <vector>
#include <string>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/inotify.h>
#include <limits.h>

// 需要跳过的路径列表
const std::vector<std::string> skipPaths = {
    "/acct", "/odm_dlkm", "/apex", "/oem", "/bin", "/opconfig", "/cache", "/opcust", "/config", "/postinstall",
    "/cust", "/proc", "/d/", "/product", "/data_mirror", "/second_stage_resources", "/debug_ramdisk", "/dev", "/sys",
    "/etc", "/system", "/linkerconfig", "/system_ext", "/lost+found", "/tmp", "/metadata", "/vendor", "/mnt", "/vendor_dlkm", "/odm"
};

const std::vector<std::string> skipFullPaths = {
    "/data/user/", "/data/user", "/data/media/0/Android/data/", "/data/media/0/Android/data", "/data/data/", "/data/data",
    "/data", "/data/", "/data/media/0", "/data/media/0/", "/data/media/0/Downloa", "/data/media/0/Download/", "/data/media/0/Android",
    "/data/media/0/Android/", "/sdcard", "/sdcard/", "/sdcard/Download", "/sdcard/Download/", "/sdcard/Android", "/sdcard/Android/",
    "/storage", "/storage/", "/storage/emulated/0", "/storage/emulated/0/", "/storage/emulated/0/Download", "/storage/emulated/0/Download/",
    "/storage/emulated/0/Android", "/storage/emulated/0/Android/", "/"
};

// 检查路径是否需要跳过
bool shouldSkipPath(const std::string& path) {
    // 检查是否在 skipPaths 中
    for (const auto& skipPath : skipPaths) {
        if (path.find(skipPath) == 0) {
            return true;
        }
    }

    // 检查是否在 skipFullPaths 中
    for (const auto& skipFullPath : skipFullPaths) {
        if (path == skipFullPath) {
            return true;
        }
    }

    return false;
}

// 读取配置文件，返回文件路径列表
std::vector<std::string> readConfigFile(const std::string& configFile) {
    std::vector<std::string> filePaths;
    std::ifstream file(configFile);
    std::string line;
    while (std::getline(file, line)) {
        // 忽略注释行
        if (line.empty() || line[0] == '#') continue;
        filePaths.push_back(line);
    }
    return filePaths;
}

// 创建文件并设置权限
void createFile(const std::string& filename) {
    std::ofstream file(filename);
    if (file.is_open()) {
        std::cout << "空文件已创建: " << filename << std::endl;
        file.close();
        if (chmod(filename.c_str(), 0000) == 0) {
            std::cout << "文件权限已设置为000: " << filename << std::endl;
        } else {
            std::cerr << "无法设置文件权限: " << filename << std::endl;
        }
    } else {
        std::cerr << "无法创建空文件: " << filename << std::endl;
    }
}

// 监控单个文件或文件夹
void monitorFile(const std::string& filename) {
    int inotifyFd = inotify_init();
    if (inotifyFd == -1) {
        std::cerr << "无法初始化 inotify" << std::endl;
        return;
    }

    int wd = inotify_add_watch(inotifyFd, filename.c_str(), IN_DELETE_SELF | IN_MOVE_SELF | IN_DELETE | IN_MOVE);
    if (wd == -1) {
        std::cerr << "无法添加 inotify 监控: " << filename << std::endl;
        close(inotifyFd);
        return;
    }

    char buf[4096] __attribute__ ((aligned(__alignof__(struct inotify_event))));
    const struct inotify_event *event;
    ssize_t numRead;

    while (true) {
        numRead = read(inotifyFd, buf, sizeof(buf));
        if (numRead == 0) {
            std::cerr << "inotify 读取到 0 字节" << std::endl;
            break;
        }

        if (numRead == -1) {
            std::cerr << "读取 inotify 事件失败" << std::endl;
            break;
        }

        for (char *ptr = buf; ptr < buf + numRead; ) {
            event = (const struct inotify_event *) ptr;
            if (event->mask & (IN_DELETE_SELF | IN_MOVE_SELF | IN_DELETE | IN_MOVE)) {
                std::filesystem::path filePath(filename);
                std::filesystem::path dirPath = filePath.parent_path();

                // 删除目录或文件
                if (std::filesystem::exists(filePath)) {
                    if (std::filesystem::is_directory(filePath)) {
                        std::filesystem::remove_all(filePath);
                        std::cout << "目录已删除: " << filePath << std::endl;
                    } else {
                        std::filesystem::remove(filePath);
                        std::cout << "文件已删除: " << filePath << std::endl;
                    }
                }

                // 使用 mkdir -p 命令创建文件夹
                std::string command = "mkdir -p " + dirPath.string();
                if (system(command.c_str()) == 0) {
                    std::cout << "文件夹已重新创建: " << dirPath << std::endl;

                    // 设置文件夹权限为 777
                    if (chmod(dirPath.c_str(), 0777) == 0) {
                        std::cout << "文件夹权限已设置为777: " << dirPath << std::endl;
                    } else {
                        std::cerr << "无法设置文件夹权限: " << dirPath << std::endl;
                    }

                    // 创建一个空文件
                    createFile(filename);
                } else {
                    std::cerr << "无法重新创建文件夹: " << dirPath << std::endl;
                }

                // 重新添加 inotify 监控
                inotify_rm_watch(inotifyFd, wd);
                wd = inotify_add_watch(inotifyFd, filename.c_str(), IN_DELETE_SELF | IN_MOVE_SELF | IN_DELETE | IN_MOVE);
                if (wd == -1) {
                    std::cerr << "无法重新添加 inotify 监控: " << filename << std::endl;
                    close(inotifyFd);
                    return;
                }
            }
            ptr += sizeof(struct inotify_event) + event->len;
        }
    }

    close(inotifyFd);
}

int main(int argc, char *argv[]) {
    std::string configFile;
    int opt;

    // 解析命令行参数
    while ((opt = getopt(argc, argv, "f:")) != -1) {
        switch (opt) {
            case 'f':
                configFile = optarg;
                break;
            default:
                std::cerr << "Usage: " << argv[0] << " -f <config_file>" << std::endl;
                return EXIT_FAILURE;
        }
    }

    if (configFile.empty()) {
        std::cerr << "配置文件路径不能为空。" << std::endl;
        return EXIT_FAILURE;
    }

    std::vector<std::string> filePaths = readConfigFile(configFile);

    // 为每个文件路径启动一个监控线程
    std::vector<std::thread> threads;
    for (const auto& filePath : filePaths) {
        // 检查是否需要跳过该路径
        if (shouldSkipPath(filePath)) {
            std::cout << "跳过路径: " << filePath << std::endl;
            continue;
        }

        std::filesystem::path dirPath = std::filesystem::path(filePath).parent_path();
        
        // 删除目录或文件
        if (std::filesystem::exists(filePath)) {
            if (std::filesystem::is_directory(filePath)) {
                std::filesystem::remove_all(filePath);
                std::cout << "目录已删除: " << filePath << std::endl;
            } else {
                std::filesystem::remove(filePath);
                std::cout << "文件已删除: " << filePath << std::endl;
            }
        }

        // 使用 mkdir -p 命令创建文件夹
        std::string command = "mkdir -p " + dirPath.string();
        if (system(command.c_str()) == 0) {
            std::cout << "文件夹已创建: " << dirPath << std::endl;

            // 设置文件夹权限为 777
            if (chmod(dirPath.c_str(), 0777) == 0) {
                std::cout << "文件夹权限已设置为777: " << dirPath << std::endl;
            } else {
                std::cerr << "无法设置文件夹权限: " << dirPath << std::endl;
            }

            // 创建一个空文件
            createFile(filePath);
            // 启动监控线程
            threads.emplace_back(monitorFile, filePath);
        } else {
            std::cerr << "无法创建文件夹: " << dirPath << std::endl;
        }
    }

    // 等待所有监控线程结束
    for (auto& thread : threads) {
        thread.join();
    }

    return 0;
}
