#!/usr/bin/env node

/**
 * Session Timer - Track Claude/Codex session durations
 * No external dependencies, works with MD-ACTIVE/ and MD-ARCHIVE/ structure
 */

const fs = require('fs');
const path = require('path');

// Paths relative to repo root
const ACTIVE_DIR = 'MD-ACTIVE';
const ARCHIVE_DIR = 'MD-ARCHIVE/reference';
const SESSION_META_FILE = path.join(ACTIVE_DIR, '.session-meta.json');
const TIMES_LOG_FILE = path.join(ARCHIVE_DIR, 'SESSION-TIMES.json');
const TIMES_SUMMARY_FILE = path.join(ARCHIVE_DIR, 'SESSION-TIMES.md');

// Ensure directories exist
function ensureDirs() {
  if (!fs.existsSync(ACTIVE_DIR)) fs.mkdirSync(ACTIVE_DIR, { recursive: true });
  if (!fs.existsSync(ARCHIVE_DIR)) fs.mkdirSync(ARCHIVE_DIR, { recursive: true });
}

// Load or create session meta
function loadSessionMeta() {
  if (!fs.existsSync(SESSION_META_FILE)) return null;
  try {
    return JSON.parse(fs.readFileSync(SESSION_META_FILE, 'utf8'));
  } catch (e) {
    return null;
  }
}

// Load or create times log
function loadTimesLog() {
  if (!fs.existsSync(TIMES_LOG_FILE)) return [];
  try {
    return JSON.parse(fs.readFileSync(TIMES_LOG_FILE, 'utf8'));
  } catch (e) {
    return [];
  }
}

// Save session meta
function saveSessionMeta(meta) {
  fs.writeFileSync(SESSION_META_FILE, JSON.stringify(meta, null, 2));
}

// Save times log
function saveTimesLog(log) {
  fs.writeFileSync(TIMES_LOG_FILE, JSON.stringify(log, null, 2));
}

// Format duration
function formatDuration(ms) {
  const hours = Math.floor(ms / (1000 * 60 * 60));
  const minutes = Math.floor((ms % (1000 * 60 * 60)) / (1000 * 60));
  const seconds = Math.floor((ms % (1000 * 60)) / 1000);
  
  if (hours > 0) return `${hours}h ${minutes}m ${seconds}s`;
  if (minutes > 0) return `${minutes}m ${seconds}s`;
  return `${seconds}s`;
}

// Start session
function startSession() {
  ensureDirs();
  
  const existing = loadSessionMeta();
  if (existing && existing.startTime && !existing.endTime) {
    console.log('⚠️  Session already running since', new Date(existing.startTime).toLocaleString());
    console.log('   Run "status" to check or "end" to finish current session');
    return;
  }
  
  const meta = {
    startTime: Date.now(),
    startDate: new Date().toISOString()
  };
  
  saveSessionMeta(meta);
  console.log('🚀 Session started at', new Date().toLocaleString());
}

// End session
function endSession() {
  ensureDirs();
  
  const meta = loadSessionMeta();
  if (!meta || !meta.startTime || meta.endTime) {
    console.log('❌ No active session found. Run "start" to begin a session.');
    return;
  }
  
  const endTime = Date.now();
  const duration = endTime - meta.startTime;
  
  // Update meta with end time
  meta.endTime = endTime;
  meta.endDate = new Date().toISOString();
  meta.duration = duration;
  
  // Add to cumulative log
  const log = loadTimesLog();
  log.push({
    startTime: meta.startTime,
    endTime: meta.endTime,
    duration: meta.duration,
    startDate: meta.startDate,
    endDate: meta.endDate
  });
  
  saveTimesLog(log);
  
  // Clear active session
  if (fs.existsSync(SESSION_META_FILE)) {
    fs.unlinkSync(SESSION_META_FILE);
  }
  
  console.log('✅ Session completed!');
  console.log('   Duration:', formatDuration(duration));
  console.log('   Total sessions:', log.length);
  
  generateSummary();
}

// Show current status
function showStatus() {
  const meta = loadSessionMeta();
  if (!meta || !meta.startTime || meta.endTime) {
    console.log('💤 No active session');
    return;
  }
  
  const elapsed = Date.now() - meta.startTime;
  console.log('⏱️  Session active since', new Date(meta.startTime).toLocaleString());
  console.log('   Elapsed time:', formatDuration(elapsed));
}

// Generate summary
function generateSummary() {
  ensureDirs();
  
  const log = loadTimesLog();
  if (log.length === 0) {
    console.log('📊 No completed sessions yet');
    return;
  }
  
  // Calculate totals
  const totalDuration = log.reduce((sum, session) => sum + session.duration, 0);
  const totalSessions = log.length;
  const avgDuration = totalDuration / totalSessions;
  
  // Recent sessions (last 10)
  const recentSessions = log.slice(-10);
  
  const summary = `# Session Times Summary

**Total Sessions**: ${totalSessions}  
**Total Time**: ${formatDuration(totalDuration)}  
**Average Session**: ${formatDuration(avgDuration)}

## Recent Sessions

${recentSessions.reverse().map((session, i) => {
  const date = new Date(session.startTime).toLocaleDateString();
  const duration = formatDuration(session.duration);
  return `${i + 1}. ${date} - ${duration}`;
}).join('\n')}

## Cumulative Development Time

This data tracks Claude/Codex-assisted development sessions to help with:
- Project time estimation and billing
- Understanding development velocity patterns
- Session duration optimization

*Generated by session-timer.js*
`;
  
  fs.writeFileSync(TIMES_SUMMARY_FILE, summary);
  
  console.log('📊 Session Summary:');
  console.log('   Total sessions:', totalSessions);
  console.log('   Total time:', formatDuration(totalDuration));
  console.log('   Average session:', formatDuration(avgDuration));
  console.log('   Summary written to:', TIMES_SUMMARY_FILE);
}

// Main command handling
const command = process.argv[2];

switch (command) {
  case 'start':
    startSession();
    break;
  case 'end':
    endSession();
    break;
  case 'status':
    showStatus();
    break;
  case 'summary':
    generateSummary();
    break;
  default:
    console.log('Usage: node session-timer.js [start|end|status|summary]');
    console.log('');
    console.log('Commands:');
    console.log('  start   - Begin tracking a new session');
    console.log('  end     - End current session and log duration');
    console.log('  status  - Show current session status');
    console.log('  summary - Generate/update session summary');
    process.exit(1);
}