#!/bin/bash

# Setup Git Hooks for Claude/Codex Development Workflow
# Installs a pre-push hook that runs `npm run ci` automatically

HOOK_DIR=".git/hooks"
HOOK_FILE="$HOOK_DIR/pre-push"

# Check if we're in a git repository
if [ ! -d ".git" ]; then
    echo "❌ Not a git repository. Run this from your repo root."
    exit 1
fi

# Check if npm run ci exists
if ! npm run | grep -q "ci"; then
    echo "⚠️  Warning: 'npm run ci' script not found in package.json"
    echo "   The hook will be installed but may fail when triggered."
    echo "   Add a 'ci' script to your package.json or remove the hook later."
fi

# Create hooks directory if it doesn't exist
mkdir -p "$HOOK_DIR"

# Create the pre-push hook
cat > "$HOOK_FILE" << 'EOF'
#!/bin/bash

echo "🔍 Running pre-push checks..."

# Run npm ci checks
if npm run ci; then
    echo "✅ Pre-push checks passed"
    exit 0
else
    echo "❌ Pre-push checks failed"
    echo "   Fix issues above or use 'git push --no-verify' to skip"
    exit 1
fi
EOF

# Make the hook executable
chmod +x "$HOOK_FILE"

echo "✅ Pre-push hook installed successfully!"
echo ""
echo "What this does:"
echo "  - Runs 'npm run ci' automatically before every push"
echo "  - Prevents pushes if CI checks fail"
echo "  - Can be bypassed with 'git push --no-verify' if needed"
echo ""
echo "To remove later: rm .git/hooks/pre-push"