

// 감지된 클립 URL을 저장하는 Map (baseUrl -> segment pattern)
const detectedClips = new Map();

// Note: Service Worker에서는 WASM을 직접 로드할 수 없으므로
// 썸네일 생성은 UI 레벨에서 처리합니다.

// 독립 윈도우 관리
let standaloneWindowId = null;

// 아이콘 클릭 시 독립 윈도우 열기
chrome.action.onClicked.addListener(() => {
	openStandaloneWindow();
});

// 독립 윈도우 열기
async function openStandaloneWindow() {
	// 이미 열려있는 창이 있으면 포커스
	if (standaloneWindowId !== null) {
		try {
			const existingWindow = await chrome.windows.get(standaloneWindowId);
			await chrome.windows.update(standaloneWindowId, { focused: true });
			return;
		} catch (e) {
			// 창이 닫혔으면 새로 생성
			standaloneWindowId = null;
		}
	}

	// 새 독립 윈도우 생성
	const window = await chrome.windows.create({
		url: chrome.runtime.getURL('index.html'),
		type: 'popup',
		width: 600,
		height: 650,
		focused: true
	});

	standaloneWindowId = window.id;

	// 창이 닫히면 ID 초기화
	chrome.windows.onRemoved.addListener((windowId) => {
		if (windowId === standaloneWindowId) {
			standaloneWindowId = null;
		}
	});
}

// webRequest API 사용 가능 여부 확인
if (chrome.webRequest) {
	
} else {
	console.error('webRequest API not available!');
}

// HLS 세그먼트 패턴 감지 (webRequest)

chrome.webRequest.onBeforeRequest.addListener(
	(details) => {
		const url = details.url;

		// 모든 요청 로깅 (테스트용)
		

		// .ts 파일만 로깅
		if (!url.includes('.ts')) return;

		

		// 필터 조건 체크
		const isPstaticDomain = url.includes('.pstatic.net');
		const hasGliveClip = url.includes('glive-clip');
		const hasHls = url.includes('/hls/');
		const hasTsExtension = url.includes('.ts');

		

		// 모든 조건을 만족하는 경우에만 처리
		if (isPstaticDomain && hasGliveClip && hasHls && hasTsExtension) {
			

			// {segment}.ts 패턴 추출 (쿼리 파라미터 포함)
			const match = url.match(/(.*\/)([^/]+)\.ts(\?.*)?$/);
			if (match) {
				const baseUrl = match[1];
				const segmentName = match[2];
				const queryParams = match[3] || '';

				
				

				// 패턴 1: 순수 숫자 (예: 0.ts, 1.ts, 123.ts)
				// 패턴 2: UUID-숫자 (예: 7d2ee7e5-eee6-11f0-9788-48df37e2c054-0.ts)
				const isNumericSegment = /^\d+$/.test(segmentName);
				const isUUIDSegment = /^[a-f0-9\-]+-\d+$/.test(segmentName);

				if (isNumericSegment || isUUIDSegment) {
					// 전체 URL (쿼리 파라미터 포함)
					const fullPattern = url;
					

					// 기존에 감지되지 않은 경우에만 저장
					if (!detectedClips.has(baseUrl)) {
						detectedClips.set(baseUrl, {
							firstSegmentUrl: fullPattern, // hdnts 포함된 전체 URL 저장
							baseUrl: baseUrl,
							timestamp: Date.now(),
							thumbnailData: null // 썸네일 데이터 저장용
						});
						

						// UI에 새 클립 감지 알림
						chrome.runtime
							.sendMessage({
								type: 'NEW_CLIP_DETECTED',
								clip: {
									firstSegmentUrl: fullPattern,
									baseUrl: baseUrl,
									timestamp: Date.now(),
									thumbnailData: null
								}
							})
							.catch(() => {
								// UI가 열려있지 않으면 무시
							});
					}
				} else {
					
				}
			}
		} else {
			
		}
	},
	{ urls: ['https://*.pstatic.net/*'] }
);



// ArrayBuffer를 Base64로 변환하는 헬퍼 함수 (CSP-safe)
async function arrayBufferToBase64(buffer) {
	// Blob을 사용한 CSP-safe 방식
	const blob = new Blob([buffer], { type: 'application/octet-binary' });
	return new Promise((resolve, reject) => {
		const reader = new FileReader();
		reader.onloadend = () => {
			if (reader.result) {
				// data:application/octet-binary;base64,xxxxx 형식에서 base64 부분만 추출
				const base64 = reader.result.split(',')[1];
				resolve(base64);
			} else {
				reject(new Error('Failed to convert to base64'));
			}
		};
		reader.onerror = reject;
		reader.readAsDataURL(blob);
	});
}

// Popup에서 감지된 클립 목록 요청
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
	

	if (message.type === 'HLS_SEGMENT_DETECTED') {
		const url = message.url;
		

		// {segment}.ts 패턴 추출 (쿼리 파라미터 포함)
		const match = url.match(/(.*\/)([^/]+)\.ts(\?.*)?$/);
		if (match) {
			const baseUrl = match[1];
			const segmentName = match[2];

			

			// 패턴 1: 순수 숫자 (예: 0.ts, 1.ts, 123.ts)
			// 패턴 2: UUID-숫자 (예: 7d2ee7e5-eee6-11f0-9788-48df37e2c054-0.ts)
			const isNumericSegment = /^\d+$/.test(segmentName);
			const isUUIDSegment = /^[a-f0-9\-]+-\d+$/.test(segmentName);

			if (isNumericSegment || isUUIDSegment) {
				

				// 기존에 감지되지 않은 경우에만 저장
				if (!detectedClips.has(baseUrl)) {
					const newClip = {
						firstSegmentUrl: url, // hdnts 포함된 전체 URL 저장
						baseUrl: baseUrl,
						timestamp: Date.now(),
						downloaded: false,
						thumbnailData: null
					};
					detectedClips.set(baseUrl, newClip);
					

					// UI에 새 클립 감지 알림
					chrome.runtime
						.sendMessage({
							type: 'NEW_CLIP_DETECTED',
							clip: newClip
						})
						.catch(() => {
							// UI가 열려있지 않으면 무시
						});
				}
			}
		}
		sendResponse({ success: true });
		return true;
	}

	if (message.type === 'GET_DETECTED_CLIPS') {
		const clips = Array.from(detectedClips.values());
		
		
		
		sendResponse({ clips });
		return true;
	}

	if (message.type === 'DOWNLOAD_CLIP') {
		(async () => {
			try {
				const result = await downloadAndMergeClip(message.url, message.returnData, (progress) => {
					// 진행상황을 popup으로 전송
					chrome.runtime
						.sendMessage({
							type: 'DOWNLOAD_PROGRESS',
							progress: progress
						})
						.catch(() => {
							// Popup이 닫혀있으면 무시
						});
				});

				// 다운로드 완료 시 downloaded 상태 업데이트
				const match = message.url.match(/(.*\/)([^/]+)\.ts(\?.*)?$/);
				if (match) {
					const baseUrl = match[1];
					const clipData = detectedClips.get(baseUrl);
					if (clipData) {
						clipData.downloaded = true;
						clipData.failed = false; // 성공 시 failed 플래그 제거
						detectedClips.set(baseUrl, clipData);
						
					}
				}

				sendResponse({ success: true, ...result });
			} catch (error) {
				// 다운로드 실패 시 failed 상태 업데이트
				const match = message.url.match(/(.*\/)([^/]+)\.ts(\?.*)?$/);
				if (match) {
					const baseUrl = match[1];
					const clipData = detectedClips.get(baseUrl);
					if (clipData) {
						clipData.failed = true;
						clipData.downloaded = false;
						detectedClips.set(baseUrl, clipData);
						
					}
				}
				sendResponse({ success: false, error: error.message });
			}
		})();
		return true; // 비동기 응답을 위해 true 반환
	}

	if (message.type === 'CONVERSION_PROGRESS') {
		// Offscreen에서 온 진행상황을 popup으로 전달
		chrome.runtime
			.sendMessage({
				type: 'DOWNLOAD_PROGRESS',
				progress: message.progress
			})
			.catch(() => {
				// Popup이 닫혀있으면 무시
			});
		return true;
	}

	if (message.type === 'CLEAR_CLIPS') {
		detectedClips.clear();
		sendResponse({ success: true });
		return true;
	}

	if (message.type === 'REMOVE_CLIP') {
		const baseUrl = message.baseUrl;
		if (detectedClips.has(baseUrl)) {
			detectedClips.delete(baseUrl);
			
			sendResponse({ success: true });
		} else {
			sendResponse({ success: false, error: 'Clip not found' });
		}
		return true;
	}

	if (message.type === 'SAVE_THUMBNAIL') {
		const { baseUrl, thumbnailData } = message;
		const clipData = detectedClips.get(baseUrl);
		if (clipData) {
			clipData.thumbnailData = thumbnailData;
			detectedClips.set(baseUrl, clipData);
			
			sendResponse({ success: true });
		} else {
			sendResponse({ success: false, error: 'Clip not found' });
		}
		return true;
	}

	if (message.type === 'UPDATE_CLIP_STATUS') {
		const { baseUrl, downloaded, failed } = message;
		const clipData = detectedClips.get(baseUrl);
		if (clipData) {
			if (downloaded !== undefined) clipData.downloaded = downloaded;
			if (failed !== undefined) clipData.failed = failed;
			detectedClips.set(baseUrl, clipData);
			
			sendResponse({ success: true });
		} else {
			sendResponse({ success: false, error: 'Clip not found' });
		}
		return true;
	}
});

async function downloadAndMergeClip(firstSegmentUrl, returnData, onProgress) {
	try {
		
		

		// URL 패턴 분석
		const match = firstSegmentUrl.match(/(.*\/)([^/]+)\.ts(\?.*)?$/);
		if (!match) {
			throw new Error('Invalid segment URL pattern');
		}

		const baseUrl = match[1];
		const segmentName = match[2];
		const queryParams = match[3] || '';

		// 세그먼트 번호 추출 (UUID-000005 형식)
		const numberMatch = segmentName.match(/-(\d+)$/);
		const isUUIDPattern = /^[a-f0-9\-]+-\d+$/.test(segmentName);

		let segmentPattern;
		let startIndex = 0;

		if (isUUIDPattern && numberMatch) {
			// UUID-숫자 패턴
			const uuidPart = segmentName.substring(0, segmentName.lastIndexOf('-'));
			const paddingLength = numberMatch[1].length;
			segmentPattern = (index) => {
				const paddedIndex = String(index).padStart(paddingLength, '0');
				return `${baseUrl}${uuidPart}-${paddedIndex}.ts${queryParams}`;
			};
		} else {
			// 순수 숫자 패턴
			segmentPattern = (index) => `${baseUrl}${index}.ts${queryParams}`;
		}

		// 세그먼트 다운로드
		const segments = [];
		let index = startIndex;
		let consecutiveErrors = 0;
		const maxConsecutiveErrors = 3;

		

		while (consecutiveErrors < maxConsecutiveErrors) {
			const segmentUrl = segmentPattern(index);

			try {
				
				const response = await fetch(segmentUrl);

				if (!response.ok) {
					
					consecutiveErrors++;

					if (consecutiveErrors >= maxConsecutiveErrors) {
						
						break;
					}

					index++;
					continue;
				}

				const blob = await response.blob();
				segments.push(blob);
				consecutiveErrors = 0; // 성공하면 에러 카운터 리셋

				

				if (onProgress) {
					onProgress({
						stage: 'downloading',
						current: segments.length,
						total: '?',
						index: index
					});
				}

				index++;
			} catch (error) {
				console.error(`Error downloading segment ${index}:`, error);
				consecutiveErrors++;

				if (consecutiveErrors >= maxConsecutiveErrors) {
					
					break;
				}

				index++;
			}
		}

		if (segments.length === 0) {
			throw new Error('No segments downloaded');
		}

		

		// 모든 세그먼트를 하나의 Blob으로 합치기
		
		const mergedBlob = new Blob(segments, { type: 'video/mp2t' });

		if (onProgress) {
			onProgress({
				stage: 'completed',
				current: segments.length,
				total: segments.length
			});
		}

		// returnData가 true이면 Blob URL을 반환
		if (returnData) {
			// Blob URL 생성 (background script에서 생성하면 popup에서 접근 가능)
			const blobUrl = URL.createObjectURL(mergedBlob);
			
			return {
				segmentCount: segments.length,
				totalSize: mergedBlob.size,
				blobUrl: blobUrl // Blob URL로 전달
			};
		}

		// 다운로드
		const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
		const filename = `chzzk-clip-${timestamp}.ts`;

		// Blob을 ArrayBuffer로 변환 후 Base64 Data URL 생성
		const arrayBuffer = await mergedBlob.arrayBuffer();
		const base64 = await arrayBufferToBase64(arrayBuffer);
		const dataUrl = `data:video/mp2t;base64,${base64}`;

		chrome.downloads.download(
			{
				url: dataUrl,
				filename: `chzzk-clips/${filename}`,
				saveAs: true
			},
			(downloadId) => {
				if (chrome.runtime.lastError) {
					console.error('Download failed:', chrome.runtime.lastError);
				} else {
					
				}
			}
		);

		return {
			segmentCount: segments.length,
			totalSize: mergedBlob.size
		};
	} catch (error) {
		console.error('Error in downloadAndMergeClip:', error);
		throw error;
	}
}
