"""
Version 3.0 - needs BlueCat Micetro version >= 11.0.0
    Copyright (c) 2024 BlueCat Networks
A python >= 3.10 script that implements an interface to PowerDNS 4.8.x
Needs the following python modules:
python-dotenv
requests
and module ingest_env_vars.py (see also notes in the header of this file)

Needs PowerDNS Authoritative Server and the PowerDNS API enabled in the pdns.conf.
"""

import json
import logging.handlers
import re
import sys
from copy import copy
from enum import Enum
from typing import Union, Any

import requests
import ingest_env_vars

# Defaults to "NATIVE", but can be configured in the env variable PDNS_MODE, e.g. to "PRIMARY" instead.
mmPDNSNewZoneType = None

# Further global variables
# will be set through the environment variables (see ingest_evn_vars.py)
base_url = ""            # PowerDNS base url
api_url = "api/v1/"      # PowerDNS API endpoint
pdns_default_ttl = 3600  # 1h
server = ""              # PowerDNS server instance name
headers = {}
the_logger = logging.getLogger('micetro_genericdns_powerdns_logger')


# Error constants
mmErr_zoneUnableToAddRecord = 0x1212
mmErr_zoneUnableToDeleteRecord = 0x1213
mmErr_zoneUnableToModifyRecord = 0x1214

# if you implement the support for a DNS record type append the type in upper case to the list
# unsupported records will cause an NotSupportedRecordType exception that results in an error message
# in Micetro that the record type is not supported.
supported_record_types: tuple[str, ...] = ("A", "AAAA", "CNAME", "NS", "SOA", "PTR", "HINFO", "MX", "DS", "TXT",
                                           "SSHFP", "SRV", "CAA")


def scaled_to_seconds(s_ttl: str) -> int:
    """
    Convert BIND scaled values to seconds, e.g. 1h => 3600
    :param s_ttl: scaled value as string
    :return: seconds as int
    """
    scaled_vals = {"s": 1, "m": 60, "h": 3600, "d": 86400, "w": 604800}
    regex = re.compile('^(?P<value>[0-9]+)(?P<unit>[smhdw])')
    seconds: int = 0
    try:
        if s_ttl is None:
            return pdns_default_ttl
        return int(s_ttl)
    except ValueError:
        ls_ttl = s_ttl.lower()
        while ls_ttl:
            match = regex.match(ls_ttl)
            if match:
                value, unit = int(match.group("value")), match.group("unit")
                if int(value) and unit in scaled_vals:
                    seconds += value * scaled_vals[unit]
                    ls_ttl = ls_ttl[match.end():]
                else:
                    raise Exception(f"Can't convert TTL '{s_ttl}' from scaled value to seconds! ")
        return seconds


def de_qualify(fq_zone_name: str, fq_domain_name: str) -> str:
    """
    Removes the trailing "." if the name ends with the fully qualified zone name.
    :param fq_zone_name: Fully qualified name of the zone.
    :param fq_domain_name: Fully qualified domain name.
    :return:
    """
    zone: str = fq_zone_name.lower()
    zone_name: str = zone[:-1]
    name: str = fq_domain_name.lower()
    if not name.endswith("."):
        if not name.endswith(zone_name):
            if name != "":
                return f"{fq_domain_name}.{zone_name}"  # append zone name as BIND does it without trailing dot.
            return zone_name  # just return the zone name
        return fq_domain_name  # already non FQ
    return fq_domain_name[:-1]


def qualify(zone_name: str, domain_name: str, rec_type: str, is_data: bool = False) -> str:
    """
    Creates FQDN from zone name and domain name if needed. is_data signals that the domain_name
    is the RDATA content and not the owner name of record.
    :param zone_name: Name of the zone, e.g. "example.com."
    :param domain_name: domain name, e.g. "host" or "host.example.com".
    :param rec_type: Upper case DNS record type, e.g. "AAAA"
    :param is_data: If true, it would return just the domain_name as is for A/AAAA,TXT records.
    :return: The domain_name as FQDN or just the RDATA
    """
    the_logger.debug(f"qualify: ZN:{zone_name} DN:{domain_name} type:{rec_type} is_data:{is_data}")
    if is_data and (rec_type == "A" or rec_type == "AAAA" or rec_type == "TXT" or rec_type == "HINFO"):
        return domain_name
    zone = zone_name.lower()
    name = domain_name.lower()
    if name.endswith("."):
        return domain_name  # already FQDN
    elif name.endswith(zone):
        return f"{domain_name}."  # make FQDN

    if (rec_type == "CNAME" or rec_type == "NS" or rec_type == "MX" or rec_type == "SRV" or rec_type == "PTR"
            or rec_type == "NAPTR"):
        # do we have a single label then we append the FQDN of the zone to the single label entry
        if "." not in name:
            return f"{domain_name}.{zone_name}"
        domain_name += "."  # not a single label, so we make it FQDN
    return domain_name  # some other type like TXT, i.e. we do not handle the trailing dot/ FQDN


def equals_case_insensitive(a: str, b: str) -> bool:
    """
    Compares and returns true if two strings are matching, case insensitively
    :param a: string a
    :param b: string b
    :return: True or False
    """
    return a.upper() == b.upper()


def tab_separated_string(text: str) -> str:
    """
    split up a text string larger than 255 in a 255 chars tab separated string.
    :param text: string with more than 255 chars.
    :return: string that split up the >255 chars tab separated, i.e. <fist 255 chars>\t<next chunk>
    """
    chunk_size = 255
    chunks = []
    for i in range(0, len(text), chunk_size):
        chunk = text[i:i + chunk_size]
        chunks.append(chunk)
    return '\t'.join(chunks)


def remove_quotes(text: str) -> str:
    """
    Strips off double quotes from string.
    :param text: input string.
    :return: string without double quotes.
    """
    return text.strip('"')


def remove_quotes_from_fields_and_return_joined(text: str, field_indices: [int, ...], join_with: str = " ") -> str:
    """
    Removed double quotes from field. The fields are determined by spitting up the text value by " "
    It removes then double-quotes only from the listed field indices.
    It joins finally the str chunks with the join_str, which defaults to space.
    :param text: space separated string
    :param field_indices: field indices that should get possible double quotes stripped off
    :param join_with: The char/string that will be used to re-join the chunked up text
    :return: The re-joined processed text
    """
    split = text.split(" ")
    for field_idx in field_indices:
        split[field_idx] = remove_quotes(split[field_idx])
    return join_with.join(split)


def no_tab_string(text: str) -> str:
    return text.replace("\t", "")


def escape_special_tab_characters(text: str) -> str:
    return text.replace("\t", "\\t")


def wrap_in_double_quotes(unquoted_string: str) -> str:
    return f'"{unquoted_string}"'


def rec_to_pdns(zone_fq: str, record: dict, pdns_ttl: int) -> dict:
    """
    Converts a Micetro record into a PDNS compatible record.
    :param zone_fq: Zone Name FQDN
    :param record: Micetro record struct.
    :param pdns_ttl: TTL that should be used for the record.
    :return: record converted to PDNS compatible name, ttl, rdata as dictionary
    """
    ttl = str(record.get('ttl')) if record.get('ttl') else pdns_ttl
    ttl = scaled_to_seconds(ttl)
    record['ttl'] = ttl
    if record['type'] == "CAA":
        split = record['data'].split("\t")
        if '"' in split[2]:
            split[2] = split[2].replace('"', '')
        split[2] = '"%s"' % (split[2])
        record['data'] = " ".join(split)
    elif record['type'] in ("NAPTR", "HINFO"):
        split = record['data'].split("\t")

        if record['type'] == "NAPTR":
            # now wrap the fields Flags = 2, Service = 3 and Regular Expression = 4 in double quotes
            for idx in range(2, 5):
                split[idx] = wrap_in_double_quotes(split[idx])
            record['data'] = " ".join(split)
            record['data'] = de_qualify(zone_fq, record['data'])
        else:
            for idx in range(2):
                split[idx] = wrap_in_double_quotes(split[idx])
            record['data'] = " ".join(split)
    elif record['type'] == "MX" or record['type'] == "SRV":
        split = record['data'].split("\t")
        split[len(split) - 1] = qualify(zone_fq, split[len(split) - 1], record['type'])
        record['data'] = " ".join(split)  # and join space separated (if there is something to join)
    elif record['type'] != "TXT" and record['type'] != "SPF":
        if record['type'] == "CNAME" or record['type'] == "PTR":
            record['data'] = "{0}.".format(de_qualify(zone_fq, record['data']))
        record['data'] = record['data'].replace("\t", " ")

    if record["type"] in ["TXT", "SPF"]:
        record['data'] = wrap_in_double_quotes(record['data'])
        record['data'] = no_tab_string(record['data'])
    if record['name'] == "":
        record['name'] = zone_fq
    record['name'] = de_qualify(zone_fq, record['name'])

    if record['type'] == "SOA":
        split = record['data'].split(" ")
        the_logger.debug("SOA split it up to %s" % split)
        the_logger.debug("split count %s" % (split[1].count('.')))
        if split[0] == "":
            split[0] = zone_fq
        if split[1].count('.') == 0:
            split[1] = "{0}.{1}".format(split[1], zone_fq)
        record['data'] = " ".join(split)
        the_logger.debug("New SOA rdata %s" % (record['data']))

    return record


def modify_record(zone_name_fq: str, old_rrset, record_before, pdns_ttl, record_after) -> None:
    the_logger.info("modifyRecord")
    add_or_rm_record(zone_name_fq, old_rrset, record_before,
                     RdataAction.REMOVE)
    records_with_same_name_and_type = get_records_with_same_name_and_type(
        record_after, zone_name_fq[:-1], zone_name_fq, pdns_ttl)
    add_or_rm_record(zone_name_fq, records_with_same_name_and_type, record_after,
                     RdataAction.ADD)


class NotSupportedRecordType(BaseException):
    pass


class RdataAction(Enum):
    ADD = 1
    REMOVE = 2
    UPDATE = 3


def add_or_rm_record(zone_name_fq: str, old_records: list, record: dict, action: RdataAction):
    the_logger.debug("Convert record in add_or_rm_record in %s with action %s" % (record, action))
    record_to_convert_to_pdns = copy(record)
    old_records, result = update_rdata(zone_name_fq, old_records, record_to_convert_to_pdns, action)
    #  result, prio, weight, port = recToPDNS(zone_nameFQ, record_to_convert_to_pdns, pdns_default_ttl)
    the_logger.debug("result of modify %s" % result)
    the_logger.debug("old_records contain %s" % old_records)
    the_logger.debug("have to %s data %s" % (action, result['data']))
    record_name = result.get("name")
    record_type = result.get("type")
    record_ttl = result.get("ttl")
    # only tested record types are supported others will cause an error
    if record_type not in supported_record_types:
        raise NotSupportedRecordType(f"Record type '{record_type}' is not supported!")
    
    rr_sets = [{"name": f"{record_name}.", "type": record_type, "changetype": "REPLACE", "ttl": record_ttl,
               "records": old_records}]
    the_logger.debug(f"patch rrsets {rr_sets}")

    response = requests.patch(base_url + api_url + f"servers/{server}/zones/{zone_name_fq}",
                              headers=headers, json={"rrsets": rr_sets})
    if response.status_code != 204:
        raise Exception(f"Record {record_name} not {action}!")


def get_records_with_same_name_and_type(dns_record: dict, zone_name: str, zone_fq: str = None, default_ttl: int = None):
    match_set = None
    record_to_convert_to_pdns = copy(dns_record)

    pdns_record: dict = rec_to_pdns(zone_fq, record_to_convert_to_pdns, default_ttl)

    response = requests.get(base_url + api_url + f"servers/{server}/zones/{zone_name}", headers=headers)

    rr_sets = response.json().get("rrsets")
    the_logger.debug("RRSETs from PDNS: %s" % json.dumps(rr_sets))
    if pdns_record:
        for rr_set in rr_sets:
            the_logger.debug(rr_set)
            if (equals_case_insensitive(rr_set.get('name'), "{0}.".format(pdns_record.get('name')))
                    and equals_case_insensitive(rr_set.get('type'), pdns_record.get('type'))):
                match_set = rr_set
                break

    if not match_set:
        the_logger.debug("get_records_with_same_name_type returns empty array")
        return []

    return match_set.get("records")


def update_rdata(zone_name_fq: str, old_records: list, record: dict,
                 rdata_action: RdataAction, new_record: dict = None) -> list:
    """
    Updates the rdata in an RRSET of old_records depending on the RdataAction ADD, REMOVE or UPDATE
    :param zone_name_fq: FQDN of the zone in which the RRSET resides
    :param old_records: List of records (existing records) that will be modified
    :param record: record to be removed or updated
    :param rdata_action: ADD, REMOVE or UPDATE
    :param new_record: in case of UPDATE this record is the new once that will replace "record" in the RRSET old_records
    :return: list that contains the modified RRSET and the PDNS converted record, prio, weight and port of it.
    """
    the_logger.debug("Convert record in update_rdata in %s" % record)
    record_to_convert_to_pdns = copy(record)
    result: dict = rec_to_pdns(zone_name_fq, record_to_convert_to_pdns, pdns_default_ttl)
    the_logger.debug("result of modify %s" % result)
    the_logger.debug("old_records contain %s" % old_records)
    the_logger.debug("have to %s the data %s" % (rdata_action, result['data']))
    if rdata_action == RdataAction.ADD:
        old_records.append({'content': result['data'], 'disabled': False})
    elif rdata_action == RdataAction.REMOVE:  # Remove
        remaining_records = []
        for rdata in old_records:
            if not equals_case_insensitive(rdata['content'], result['data']):
                remaining_records.append(rdata)
            else:
                the_logger.debug("found %s" % (rdata['content']))
                #old_records.remove(rdata)
        return [remaining_records, result]  # , prio, weight, port]
    elif rdata_action == RdataAction.UPDATE:  # Update
        updated_records = []
        for rdata in old_records:
            if not equals_case_insensitive(rdata['content'], result['data']):
                updated_records.append(rdata)  # add only the records we are not going to replace
            else:
                the_logger.debug("found old record content to be replaced: %s" % (rdata['content']))
                # add new one
                record_to_convert_to_pdns = copy(new_record)
                result: dict = rec_to_pdns(zone_name_fq, record_to_convert_to_pdns, pdns_default_ttl)
                updated_records.append({'content': result['data'], 'disabled': False})
                the_logger.debug("returning updated RRSET %s" % updated_records)
                return [copy(updated_records), result]
    return [old_records, result]


def do_get_server_info() -> dict[str, str]:
    """
    Tries to list up the server and returns the PowerDNS Auth type.
    If it can not retrieve the data it throws an exception.
    :return: {'type': 'PowerDNS Auth'} or Exception
    """
    response = requests.get(base_url + api_url + f"servers/{server}", headers=headers)
    if response.status_code == 200:
        return {'type': 'PowerDNS Auth'}
    raise Exception("Failed to get server info via the PDNS API!")


def do_get_service_status() -> dict[str, str]:
    """
    Return information about the status of the DNS service itself
    possible return values are:
    :return:
        "undefined" - we have no idea about the service
        "running" - the service is up and running
        "stopped" - the service is stopped
        "exited" - the service has exited
        "fatal" - the service has entered a fatal state

    """
    try:
        resp = requests.get(base_url + api_url + f"servers/{server}", headers=headers)
        the_logger.debug(f"Response from servers/{server} call to PDNS [{resp}]")
        if resp.status_code == 200:
            return {'serviceStatus': 'running'}
        else:
            return {'serviceStatus': 'stopped'}
    except ConnectionError:
        return {'serviceStatus': 'stopped'}


def do_get_views() -> dict[str, list[str]]:
    """
    Return all views available on the DNS server (no views in PowerDNS, i.e. empty array)
    :return: empty array of view names
    """
    return {'views': ['']}


def do_get_zones() -> dict[str, list[dict[str, Union[Union[str, bool], Any]]]]:
    """
    Returns all zones from PowerDNS Auth to be returned to Micetro
    :return: "zones" list of all auth zones that are present on the managed PDNS instance.
    """
    response = requests.get(base_url + api_url + f"servers/{server}/zones", headers=headers)
    zones_response = response.json()
    zones = []
    for zone_response in zones_response:
        zone_type = 'Slave' if equals_case_insensitive(zone_response.get("kind"), "slave") else 'Master'
        zones.append({'view': '', 'name': zone_response.get("name"), 'type': zone_type,
                      'dynamic': False, 'serial': zone_response.get("serial")})
    return {'zones': zones}


def do_get_zone() -> dict[str, str]:
    """
    Returns information for a specific zone on the PDNS instance - it's type and current serial
    :return: the Micetro zone struct.
    """
    text = sys.stdin.read()
    micetro_input = json.loads(text)
    zone_name = micetro_input['params']['name']
    zone_name = zone_name[:-1]  # remove trailing dot

    response = requests.get(base_url + api_url + f"servers/{server}/zones/{zone_name}", headers=headers)
    zone_response = response.json()

    if zone_response.get("kind").lower() in ["native", "master"]:
        zone = {'view': '', 'name': zone_response.get("name"), 'type': 'Master', 'dynamic': False,
                'serial': zone_response.get("serial")}
        return zone

    raise Exception("Zone '%s' not found!" % zone_name)


def do_get_records() -> dict[str, list[dict[str, Union[str, Any]]]]:
    """
    Retrieves records from a specific PDNS zone and returns the "records" json
    :return: "records" json struct
    """
    text = sys.stdin.read()
    micetro_input = json.loads(text)
    zone_name_fq = micetro_input['params']['name']
    zone_name = zone_name_fq[:-1]
    weight = None
    prio = None
    port = None

    response = requests.get(base_url + api_url + f"servers/{server}/zones/{zone_name}", headers=headers)
    if response.status_code != 200:
        raise Exception("Zone '%s' not found!" % zone_name)

    records = []
    zone_response = response.json()

    for record_set in zone_response.get("rrsets"):
        rec_type = record_set.get("type")
        name = qualify(zone_name, record_set.get("name"), rec_type)
        ttl = str(record_set.get("ttl", None))
        if ttl is None or ttl == "":
            ttl = ""
        for record in record_set.get("records"):
            extracted_content = record.get("content")
            if rec_type == "MX":
                content_split = record.get("content").split()
                prio = str(content_split[0])
                extracted_content = content_split[1]
            elif rec_type == "SRV":
                content_split = record.get("content").split()
                prio = str(content_split[0])
                weight = str(content_split[1])
                port = str(content_split[2])
                extracted_content = content_split[3]
            content = qualify(zone_name, extracted_content, rec_type, True)

            # MX and SRV store the priority in the separate prio column (index 4) see select statement
            if rec_type == "MX":
                content = f"{prio}\t{content}"
                the_logger.debug(f"content qualify {content}")
            elif rec_type == "SRV":
                content = f"{prio}\t{weight}\t{port}\t{content}"
            elif rec_type == "NAPTR":
                content = remove_quotes_from_fields_and_return_joined(content, [2, 3, 4], "\t")

            # all other parameters are space separated, but we exclude the listed types like TXT and so on
            if " " in content and rec_type not in ["TXT", "SPF", "CAA", "HINFO", "NAPTR"]:
                content = content.replace(" ", "\t")

            if rec_type in ["TXT", "SPF"]:
                content = remove_quotes(content)
                content = tab_separated_string(content)
            elif rec_type == "CAA":
                content = remove_quotes_from_fields_and_return_joined(content, [2], "\t")
            elif rec_type == "HINFO":
                the_logger.debug(f"HINFO - original content {content}")
                content = remove_quotes_from_fields_and_return_joined(content, [0, 1], "\t")
                the_logger.debug(f"HINFO - removed double-quotes content {content}")

            records.append({'name': name, 'ttl': ttl, 'type': rec_type, 'data': content})
            the_logger.debug("name:%s type:%s data: %s" % (name, rec_type, content))
    the_logger.info("Zone: '%s' number of records retrieved: %s" % (zone_name, len(records)))
    return {'dnsRecords': records}


def do_create_zone() -> int:
    """
    Creates a zone on PDNS as type Master or Slave.
    In case of Master it also updates or adds the records provided by Micetro (e.g. SOA, NS ...).
    :return: HTTP response code
    """
    text = sys.stdin.read()
    text = escape_special_tab_characters(text)
    the_logger.debug("Micetro req: %s" % text)
    input_json = json.loads(text)
    zone_name_fq = input_json['params']['name']
    zone_name = zone_name_fq[:-1]

    zone_type = input_json['params']['kind'] = input_json['params']['type']
    records = input_json['params'].get('dnsRecords')

    if not (equals_case_insensitive(zone_type, "Master") or equals_case_insensitive(zone_type, "Slave")):
        raise Exception(f"Can't create zone {zone_name}: Only zone type Master or Secondary supported!")

    if equals_case_insensitive(zone_type, "Slave"):
        masters = input_json['params']['masters']  # leaving multiple masters
        zone = {"name": zone_name_fq, "kind": zone_type, "masters": masters}

        response = requests.post(base_url + api_url + f"servers/{server}/zones", headers=headers, json=zone)
        if response.status_code != 201:
            raise Exception(f"Zone {zone_name} not created!")
        else:
            return response.json()
    else:  # master or native zone type
        zone = {"name": zone_name_fq, "kind": mmPDNSNewZoneType}

    response = requests.post(base_url + api_url + f"servers/{server}/zones", headers=headers, json=zone)
    if response.status_code != 201:
        raise Exception(f"Zone {zone_name} not created!")

    # now update the records (SOA, NS)
    if records:
        for record in records:
            the_logger.debug("Record create zone %s" % record)
            if record.get("type") == "SOA":
                the_logger.debug("Special case SOA record in createZone")
                record_to_convert_to_pdns = copy(record)
                old_records, result = update_rdata(zone_name_fq, [], record_to_convert_to_pdns,
                                                   RdataAction.ADD)
                rr_sets = [{"name": zone_name_fq, "type": "SOA", "changetype": "REPLACE", "ttl": pdns_default_ttl,
                            "records": old_records}]
                response = requests.patch(base_url + api_url + f"servers/{server}/zones/{zone_name_fq}",
                                          headers=headers,
                                          json={"rrsets": rr_sets})
                the_logger.debug(response.text)
                the_logger.debug(response)
                if response.status_code != 204:
                    raise Exception(f"Zone creation, SOA record update failed {zone_name_fq}")
            else:
                records_with_same_name_and_type = get_records_with_same_name_and_type(record, zone_name, zone_name_fq)
                add_or_rm_record(zone_name_fq, records_with_same_name_and_type, record, RdataAction.ADD)

    return response.status_code


def do_delete_zone() -> int:
    """
    Deletes zone from PDNS
    :return: HTTP status code returned by the PDNS REST API call
    """
    text = sys.stdin.read()
    micetro_input = json.loads(text)
    zone_name = micetro_input['params']['name']
    response = requests.delete(base_url + api_url + f"servers/{server}/zones/{zone_name}", headers=headers)
    if response.status_code != 204:
        raise Exception(f"Zone {zone_name} not deleted!")
    return response.status_code


def do_update_zone() -> int | Exception:
    """
    Updates the zone content. STDIN contains the json struct returned by the Generic DNS Controller,
    which contains the type of the update "AddDNSRecord", "ModifyDNSRecord" or "RemoveDNSRecord"
    :return: code 200 if everything was OK otherwise it raises an exception including the failed updates.
    """
    text = sys.stdin.read()
    text = escape_special_tab_characters(text)
    the_logger.debug("Micetro req: %s" % text)
    micetro_input = json.loads(text)
    zone_name_fq = micetro_input['params']['name']
    zone_name = zone_name_fq[:-1]
    failed_updates = []

    k_type_to_error_map = {'AddDNSRecord': mmErr_zoneUnableToAddRecord,
                           'ModifyDNSRecord': mmErr_zoneUnableToModifyRecord,
                           'RemoveDNSRecord': mmErr_zoneUnableToDeleteRecord}

    for dnsRecordChange in micetro_input['params']['dnsRecordChanges']:
        try:
            # Fetch RRSET from PDNS - in case of action Add we have to fetch the records with same name and type that
            # will be added, i.e. record in dnsRecordAfter.
            # Otherwise, it is a record remove or modify and therefore we use dnsRecordBefore
            if dnsRecordChange.get("type") == "AddDNSRecord":
                records_with_same_name_and_type = get_records_with_same_name_and_type(
                    dnsRecordChange.get('dnsRecordAfter'), zone_name, zone_name_fq, pdns_default_ttl)
            else:
                records_with_same_name_and_type = get_records_with_same_name_and_type(
                    dnsRecordChange.get('dnsRecordBefore'), zone_name, zone_name_fq, pdns_default_ttl)
                the_logger.debug("same name and type= %s" % records_with_same_name_and_type)
            if dnsRecordChange['type'] == 'AddDNSRecord':
                the_logger.info("AddDNSRecord")
                add_or_rm_record(zone_name_fq, records_with_same_name_and_type, dnsRecordChange['dnsRecordAfter'],
                                 RdataAction.ADD)
            if dnsRecordChange['type'] == 'ModifyDNSRecord':
                the_logger.info("ModifyDNSRecord")

                modify_record(zone_name_fq, records_with_same_name_and_type, dnsRecordChange['dnsRecordBefore'],
                              pdns_default_ttl, dnsRecordChange["dnsRecordAfter"])

            elif dnsRecordChange['type'] == 'RemoveDNSRecord':
                the_logger.info("RemoveDNSRecord")
                add_or_rm_record(zone_name_fq, records_with_same_name_and_type, dnsRecordChange['dnsRecordBefore'],
                                 RdataAction.REMOVE)
        except Exception as e:
            failed_updates.append(
                {'changeIndex': dnsRecordChange['changeIndex'],
                 'errorValue': k_type_to_error_map[dnsRecordChange['type']],
                 'errorMessage': str(e)})
    if failed_updates:
        raise Exception(failed_updates)
    return 200


def main():
    env_vars = ingest_env_vars.load_vars()
    global api_url, headers, base_url, server, pdns_default_ttl, mmPDNSNewZoneType, the_logger
    headers = {'Content-type': 'application/json', 'X-API-Key': env_vars.api_key}

    base_url = env_vars.base_url
    server = env_vars.server
    pdns_default_ttl = int(env_vars.pdns_default_ttl)
    mmPDNSNewZoneType = env_vars.pdns_mode

    # log_level INFO or DEBUG is supported
    the_logger.setLevel(env_vars.log_level)

    # Configure logging
    log_file_name: str = env_vars.log_file_name

    # Add the log message handler to the logger 10 MB log file 5 times
    handler = logging.handlers.RotatingFileHandler(log_file_name, maxBytes=1024 * 1024 * 10, backupCount=5)

    # create formatter
    formatter = logging.Formatter("%(asctime)s - %(lineno)3i - %(funcName)20s() - %(message)s")
    handler.setFormatter(formatter)

    the_logger.addHandler(handler)
    result = dict()
    try:
        if len(sys.argv) <= 1:
            raise Exception('missing argument')
        the_logger.info(sys.argv[1])
        if sys.argv[1] == 'GetViews':
            result['result'] = do_get_views()
        elif sys.argv[1] == 'GetServerInfo':
            result['result'] = do_get_server_info()
        elif sys.argv[1] == 'GetServiceStatus':
            result['result'] = do_get_service_status()
        elif sys.argv[1] == 'GetZones':
            result['result'] = do_get_zones()
        elif sys.argv[1] == 'GetZone':
            result['result'] = do_get_zone()
        elif sys.argv[1] == 'GetRecords':
            result['result'] = do_get_records()
        elif sys.argv[1] == 'UpdateZone':
            result['result'] = do_update_zone()
        elif sys.argv[1] == 'CreateZone':
            result['result'] = do_create_zone()
        elif sys.argv[1] == 'DeleteZone':
            result['result'] = do_delete_zone()
        else:
            # Unknown argument
            raise Exception('unknown argument: "' + sys.argv[1] + '"')

    except NotSupportedRecordType as not_supported:
        result['error'] = {'code': 65535, 'message': 'error: ' + str(not_supported)}
    except Exception as e:
        result['error'] = {'code': 42, 'message': 'error: ' + str(e)}

    the_logger.debug(json.dumps(result))
    the_logger.info("Convert result to json")
    result_str = json.dumps(result, indent=4, sort_keys=True)
    the_logger.info("Writing result to stdout")
    print(result_str)
    the_logger.info("Done")


if __name__ == '__main__':
    main()
