// Tool for emulating a digital supply meter.
// 
//                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
//                  October 2012

#define NEGATIVE 0
#define POSITIVE 1
#define ON 0
#define OFF 1

byte outputPinForLed = 13;
byte voltageSensorPin = 2;
byte currentSensorPin = 1;
byte ledDetectorPin = 2;  // digital 
byte ledRepeaterPin = 10;  // digital 

long cycleCount = 0; // used to time LED events, rather than calling millis()
int samplesDuringThisMainsCycle = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths

long noOfSamplePairs = 0;
byte polarityNow; 

boolean beyondStartUpPhase = false;

float energyInBucket = 10; // mimics the operation of a meter at the grid connection point.                                                
int capacityOfEnergyBucket = 3600; // 0.001 kWh = 3600 Joules
int sampleV,sampleI;   // voltage & current samples are integers in the ADC's input range 0 - 1023 
int lastSampleV;     // stored value from the previous loop (HP filter is for voltage samples only)         
float lastFilteredV,filteredV;  //  voltage values after HP-filtering to remove the DC offset

float prevDCoffset;          // <<--- for LPF 
float DCoffset;              // <<--- for LPF 
float cumVdeltasThisCycle;   // <<--- for LPF 
float sampleVminusDC;         // <<--- for LPF
float sampleIminusDC;         // <<--- used with LPF
float lastSampleVminusDC;     // <<--- used with LPF
float sumP;   //  cumulative sum of power calculations within each mains cycle
float PHASECAL; 
float POWERCAL;  // To convert the product of raw V & I samples into Joules.  

// items for charging, including LED display
byte myLEDstate = ON;
boolean myLED_pulseInProgress = false;
unsigned long myLED_onAt;
int consumedPowerRegister = 0;
float energyLevelForDiags = 0;
float lastEnergyLevelForDiags;


void setup()
{  
  Serial.begin(115200);
  pinMode(outputPinForLed, OUTPUT);  
    
   POWERCAL = 0.0555; // Units are Joules per ADC-level squared.  Used for converting the product of 
                    // voltage and current samples into Joules.
                    //    To determine this value, note the rate that the energy bucket's
                    // level increases when a known load is being measured at a convenient
                    // test location (e.g  using a mains extention with the outer cover removed so that 
                    // the current-clamp can fit around just one core.  Adjust POWERCAL so that
                    // 'measured value' = 'expected value' for various loads.  The value of
                    // POWERCAL is not critical as any absolute error will cancel out when 
                    // import and export flows are balanced.                      
                        
  PHASECAL = 1;                         
}


void loop() // each loop is for one pair of V & I measurements
{
  noOfSamplePairs++;              // for stats only
  samplesDuringThisMainsCycle++;  // for power calculation at the start of each mains cycle

  // store values from previous loop
  lastSampleV=sampleV;            // for digital high-pass filter
  lastFilteredV = filteredV;      // for HPF, used to identify the start of each mains cycle
  lastSampleVminusDC = sampleVminusDC;  // for phasecal calculation 
 
// Get the next pair of raw samples.  Because the CT generally adds more phase-advance 
// than the voltage sensor, it makes sense to sample current before voltage
  sampleI = analogRead(currentSensorPin);   
  sampleV = analogRead(voltageSensorPin);   

  // remove the DC offset from these samples as determined by a low-pass filter
  sampleVminusDC = sampleV - DCoffset; 
  sampleIminusDC = sampleI - DCoffset;

  // a high-pass filter is used just for determining the start of each mains cycle  
  filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV);   

  // Establish the polarities of the latest and previous filtered voltage samples
  byte polarityOfLastReading = polarityNow;
  if(filteredV >= 0) 
    polarityNow = POSITIVE; 
  else 
    polarityNow = NEGATIVE;


  if (polarityNow == POSITIVE)
  {
    if (polarityOfLastReading != POSITIVE)
    {
      // This is the start of a new mains cycle (just after the +ve going z-c point)
      cycleCount++; // for stats only
//      checkLedStatus(); // a really useful function, but can be commented out if not required

      // update the Low Pass Filter for DC-offset removal
      prevDCoffset = DCoffset;
      DCoffset = prevDCoffset + (0.01 * cumVdeltasThisCycle); 

      //  Calculate the real power of all instantaneous measurements taken during the 
      //  previous mains cycle, and determine the gain (or loss) in energy.
      float realPower = POWERCAL * sumP / (float)samplesDuringThisMainsCycle;
      float realEnergy = realPower / cyclesPerSecond;

      if (beyondStartUpPhase == true)
      {  
        // Providing that the DC-blocking filters have had sufficient time to settle,    
        // add this power contribution to the energy bucket
        energyInBucket += realEnergy;   
        energyLevelForDiags += realEnergy;   
         
        if (energyInBucket > capacityOfEnergyBucket)
        {
          energyInBucket -= capacityOfEnergyBucket;
          registerConsumedPower();   
        }
        
        if (energyInBucket < 0)
        {
          digitalWrite(outputPinForLed, 1);
          energyInBucket = 0; 
        }  
      }
      else
      {  
        // wait until the DC-blocking filters have had time to settle
        if(cycleCount > 100) // 100 mains cycles is 2 seconds
          beyondStartUpPhase = true;
      }
      checkMyLED_status();
      
      // clear the per-cycle accumulators for use in this new mains cycle.  
      sumP = 0;
      samplesDuringThisMainsCycle = 0;
      cumVdeltasThisCycle = 0;
    } // end of processing that is specific to the first +ve Vsample in each new mains cycle
  }  // end of processing that is specific to positive Vsamples
  
  
  // Processing for ALL Vsamples, both positive and negative
  //------------------------------------------------------------
   
  // Apply phase-shift to the voltage waveform to ensure that the system measures a
  // resistive load with a power factor of unity.
  float  phaseShiftedVminusDC = 
                lastSampleVminusDC + PHASECAL * (sampleVminusDC - lastSampleVminusDC);  
  float instP = phaseShiftedVminusDC * sampleIminusDC; //  power contribution for this pair of V&I samples 
  sumP +=instP;     // cumulative power contributions for this mains cycle 

  cumVdeltasThisCycle += (sampleV - DCoffset); // for use with LP filter
} // end of loop()


void registerConsumedPower()
{
  consumedPowerRegister++;
//  myLED_onAt = millis();
  myLED_onAt = cycleCount;
  digitalWrite(outputPinForLed, 1);  
  myLED_pulseInProgress = true;
  Serial.println(energyLevelForDiags - lastEnergyLevelForDiags);
  lastEnergyLevelForDiags = energyLevelForDiags;
}

void checkMyLED_status()
{
  if (myLED_pulseInProgress == true)
  {
//    timeNow = millis(); 
//    if (timeNow > (myLED_onAt + 38)) // normal pulse duration
    if (cycleCount > (myLED_onAt + 2)) // normal pulse duration
    {
      digitalWrite(outputPinForLed, 0); 
      myLED_pulseInProgress = false; 
//      Serial.print("LED OFF at ");
//      Serial.println(cycleCount);
//      Serial.println(millis());
    }
  }
}  

