/*
 * Emonitor_Core.cpp
 *
 * Created: 06/03/2012 10:30:58
 *  Author: pcunha
 *
 * version with fixed settings for one voltage sensor and three CTs
 * some minor improvements/fixes
 *  Author: JB 23.07.2012
 */

#include "Arduino.h"
#define F_CPU 16000000L
#include <JeeLib.h>
#include <stdio.h>
#include <avr/eeprom.h>


//------------------------------------------------------------------------------------
//DEFINES
//------------------------------------------------------------------------------------
#define VERSION                      4
#define NODE_TYPE                 0x01

#define CONFIG_EEPROM_ADDR ((byte*) 0x10)

#define SERIAL                       1 // enable serial output
//#undef SERIAL

#define DEBUG                        1 // enable simpel debugging mechanisms (not implemented yet)

#define CALC_INTERVAL                1 // in seconds - time wich arduino will do power calculations.

#define FREQUENCY                   50 // IN HZ
#define VOLTAGE_SENSORS              1 // number of voltage sensors (wall transformers)
#define PHASE_SENSORS                3 // number of CTs
#define PHASE_SHIFT                120 // phase shift between phases (120Â° for 3 phase circuits)

//samples per second are F_CPU / ADC prescaler(128) / 13 clock cycles per adc / channel count
#define SAMPLES_PER_SECOND (F_CPU / 128 / 13 / (PHASE_SENSORS + VOLTAGE_SENSORS))

//SAMPLES_PER_SECOND are F_CPU / ADC prescaler(128) / 13 clock cycles per adc / channel count
//the buffer size is samples per second / degrees to go back
//for 3-phase circuits, the degree is 240
//for 2-phase is 90 - i cannot test this...
//for 2-phases of a 3-phase system in theory is 120.

//MAX_SHIFT_DEGREE = 240, ADC_BUFFER_SIZE=PHASE2_OFFSET would be big enough!
#define PHASE1_OFFSET               15
#define PHASE2_OFFSET               31
//#define ADC_BUFFER_SIZE  (PHASE2_OFFSET+1) // minimum memory
#define ADC_BUFFER_SIZE             64 // has to be a power of two to speed up modulo calculations

// 1 voltage sensor, 3 CTs
#define V0                           2 // ADC2
#define I0                           3 // ADC3
#define I1                           0 // ADC0
#define I2                           1 // ADC1

/*
Pooling Order:
1Phase Voltage - 3 Phase CT:
Voltage1 - Current1, Current2, Current3, Voltage1...
*/
unsigned char ADCpins[VOLTAGE_SENSORS+PHASE_SENSORS] = {2, 3, 0, 1};

struct Config
{
    byte  band;
    byte  group;
    byte  nodeID;
    byte  sendTo;
    byte  transmitinterval;
    int   Phase_Calibration[3];
    int   Current_Calibration[3];
    int   Voltage_Calibration[3];
    float Voltage;
    byte  valid; // keep this as last byte
} config;

static void ConfigDefaults()
{
        config.valid                  = 253;
        config.band                   = 8;
        config.group                  = 210;
        config.nodeID                 = 10;
        config.sendTo                 = 1;
        config.transmitinterval       = 3;
        config.Phase_Calibration[0]   = (int)(0.57     *65535);
        config.Phase_Calibration[1]   = (int)(0.57     *65535);
        config.Phase_Calibration[2]   = (int)(0.57     *65535);
        config.Current_Calibration[0] = (int)(0.097    *4096);
        config.Current_Calibration[1] = (int)(0.100    *4096);
        config.Current_Calibration[2] = (int)(0.100    *4096);
        config.Voltage_Calibration[0] = (int)(0.738    *4096);
        config.Voltage_Calibration[1] = (int)(0.740    *4096);
        config.Voltage_Calibration[2] = (int)(0.740    *4096);
        config.Voltage = 230.0;
}


//End of User defineable variables
//---------------------------------------------------
// Do not touch the code below this point, unless you know what you are doing
//---------------------------------------------------
static void ConfigDefaults();
static void loadConfig();
static void saveConfig();
void initRF();
void process_rf();

//tx data
typedef struct
{
  byte nodeId;                         // 1 Byte
  byte command;                        // 1 Byte
  unsigned int txCount;                // 2 bytes
  float totalP;                        // 4 bytes
  float cummKw;                        // 4 bytes
  unsigned char numV;                  // 1 Byte
  unsigned char numI;                  // 1 Byte
  float V1;                            // 4 bytes
  float Irms1;                         // 4 bytes
  float RP1;                           // 4 bytes
  float PF1;                           // 4 bytes
  float Irms2;                         // 4 bytes
  float RP2;                           // 4 bytes
  float PF2;                           // 4 bytes
  float Irms3;                         // 4 bytes
  float RP3;                           // 4 bytes
  float PF3;                           // 4 bytes
} PayloadTX;                           // create structure - a neat way of packaging data for RF comms

//Temp variables
PayloadTX emontx;
volatile unsigned char AdcIndex = 1;   // keeps track of sampled channel
volatile unsigned int  crossCounter;
volatile unsigned int  SampleCounter;
volatile unsigned int  LastSampleCounter;
volatile boolean Cycle_Full;           // stores wherever the cycle has been computed
unsigned char tempidx;

struct CurrentSampleS
{
   signed int  New;
   signed int  Previous;
   signed long Filtered;
   signed long PreviousFiltered;
   signed int  Calibrated;
   float       PhaseShifted;
   long        Sum;
   float       InstPower;
}  CurrentSample[3];                   // for now i will maintain 3 here hardcoded because i dont want to put 300 ifs on the code yet.

struct VoltageSampleS
{
   signed int  New;
   signed int  Previous;
   signed long Filtered;
   signed long PreviousFiltered;
   signed int  Calibrated;
   float       PhaseShifted;
   float Sum;
} VoltageSample[3];                    // Yes phases here, because we are going to create the voltage data based on what we have //for now i will maintain 3 here hardcoded because i dont want to put 300 ifs on the code yet.

float VS0Filtered;
float VS0PreviousFiltered;

struct LastCurrentS
{
   long Sum;
   float InstPower;
} LastCurrent[3];                      // for now i will maintain 3 here hardcoded because i dont want to put 300 ifs on the code yet.

struct LastVoltageS
{
   float Sum;
} LastVoltage[3];

long AccVrms[3];
float AccIrms[3];
float AccRealPower[3] ;
float AccAparentPower[3];
float AccPowerFactor[3];

float cummP;                           // cummulative power

//declare voltage buffer
int ADC_V_BUFFER[ADC_BUFFER_SIZE+1];
unsigned char adc_buffer_index = 0;


unsigned char tcount;
unsigned int  counts;

const int TESTpin = 7;                 // for debugging puposes
const int LEDpin  = 9;                 // On-board emonTx LED


static void loadConfig()
{
   for (byte i = 0; i < sizeof config; ++i)
      ((byte*) &config)[i] = eeprom_read_byte(CONFIG_EEPROM_ADDR + i);
   if (config.valid != 253)
      ConfigDefaults();
}


static void saveConfig()
{
   for (byte i = 0; i < sizeof config; ++i)
      eeprom_write_byte(CONFIG_EEPROM_ADDR + i, ((byte*) &config)[i]);
}


void initRF()
{
   byte freq = config.band == 4 ? RF12_433MHZ :
               config.band == 8 ? RF12_868MHZ :
                                  RF12_915MHZ;
  #ifdef SERIAL
    Serial.print("Initialiazing RF: Node:");
    Serial.print(config.nodeID);
    Serial.print(" Freq:");
    Serial.print(freq);
    Serial.print(" group:");
    Serial.println(config.group);
  #endif
   rf12_initialize(config.nodeID, freq, config.group);
}


void setup()
{
  #ifdef SERIAL
    Serial.begin(57600);                  // begin Serial comm
    Serial.println("");
    delay(100);
    Serial.println("emonTX UP and Running");
    delay(500);
  #endif

  pinMode(TESTpin, OUTPUT);               // Setup TEST pin
  pinMode(LEDpin,  OUTPUT);               // Setup indicator LED
  digitalWrite(TESTpin, HIGH);
  digitalWrite(LEDpin, HIGH);

  loadConfig();
  initRF();

  emontx.numV = VOLTAGE_SENSORS;
  emontx.numI = PHASE_SENSORS;

  // ADC initialization:
  // Enable ADC interrupt and use external voltage reference.
  // Set multiplexer input to first channel and do not setanalog_pins
  // ADC to High Speed Mode. Set Auto Trigger Source to Free Running Mode.
  //
  //   fCLK | ADPS | Prescaler | ADCCLK | Sampl.Rate
  //  ------+------+-----------+--------+-----------
  // 16.000 |  111 |       128 | 125000 |       9615

   //ADC INIT
   ADMUX   = (ADCpins[0] & 0x07)|0x40;   // Set ADC reference to external VFREF and first defined port
   ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2)+(1<<ADATE)+(1<<ADIE);
   ADCSRA |= (1 << ADEN);                 // Enable ADC

   sei();                                 // Enable Global Interrupts

   //put code here to start adc after zero crossing. *TODO*  but not a problem, because we do not pick up first sampled data...

   //Start ADC
  #ifdef SERIAL
   Serial.println("Open Energy Monitor Hack by Pcunha");
   Serial.println("Starting ADC in next cycle");
  #endif
   ADCSRA |= (1<<ADSC);                   // start ADC
}


//------------------------------------------------------------------------------------
//ADC INTERRUPT SERVICE ROUTINE
//------------------------------------------------------------------------------------
ISR(ADC_vect)
{                                         // ADC interrupt
   unsigned char AdcSampled;
   signed long TempL;
   signed int TempS;

   AdcIndex   = (AdcIndex+1)&0x03;        // increment to next channel
   ADMUX = 0x40|(ADCpins[AdcIndex]);      // select next ADC input to be samples
   AdcSampled = ADCpins[(AdcIndex+2)&0x03]; // last ADC input samples, result is in ADC now!

   switch( AdcSampled )
   {
      case V0:                            // first voltage sensor (<32us@16MHz)
         // handle V0 raw values from ADC
         VoltageSample[0].Previous         = VoltageSample[0].New;           // save last sample
         VoltageSample[0].New              = ADC;                            // Read in raw voltage signal from index sample

         // filter V0
         VoltageSample[0].PreviousFiltered = VoltageSample[0].Filtered;      // save last filtered value
         TempL = (((long)VoltageSample[0].Filtered<<8)-VoltageSample[0].Filtered)>>8; // TempL=0.996*VoltageSample[0].Filtered
         TempS = VoltageSample[0].New - VoltageSample[0].Previous;
         VoltageSample[0].Filtered         = TempL+(((long)TempS)<<8)-TempS; // (256-1)*TempS = 255*TempS

         // calibrate V0, V0 is expressed in 1/16V (=62.5mV) steps now !
         TempL = VoltageSample[0].Filtered;
         VoltageSample[0].Calibrated       = (TempL*config.Voltage_Calibration[0])>>16;

         // sum up V0 squared for TRMS calculation
         VoltageSample[0].Sum             += ((long)VoltageSample[0].Calibrated*VoltageSample[0].Calibrated);

         // store in ring buffer
         if( adc_buffer_index>0 )
            adc_buffer_index--;
         else
            adc_buffer_index = (ADC_BUFFER_SIZE-1);
         ADC_V_BUFFER[adc_buffer_index]    = VoltageSample[0].Calibrated;
         break;

      case I0:                            // first current sensor (<XXus@16MHz)
   digitalWrite(TESTpin, HIGH);
         // handle I0 raw values from ADC
         CurrentSample[0].Previous         = CurrentSample[0].New;           // Put last sample on its place
         CurrentSample[0].New              = ADC;                            // Read in raw voltage signal from index sample

         // filter I0
         CurrentSample[0].PreviousFiltered = CurrentSample[0].Filtered;      // Put last sample on its place
         TempL = (((long)CurrentSample[0].Filtered<<8)-CurrentSample[0].Filtered)>>8; // TempL=0.996*CurrentSample[0].Filtered
         TempS = CurrentSample[0].New - CurrentSample[0].Previous;
         CurrentSample[0].Filtered         = TempL+(((long)TempS)<<8)-TempS; // (256-1)*TempS = 255*TempS

         // calibrate I0, I0 is expressed in 1/32A (=31.25mA) steps now !
         TempL = CurrentSample[0].Filtered;
         CurrentSample[0].Calibrated       = (TempL*config.Current_Calibration[0])>>16;

         // sum up I0 squared for TRMS calculation
         CurrentSample[0].Sum             += ((long)CurrentSample[0].Calibrated*CurrentSample[0].Calibrated);
   digitalWrite(TESTpin, LOW);
         break;
   }

   if( AdcIndex==3 )
      SampleCounter++;

   //4807 for 2 channels
   //1602 for 6 channels
   //1  sec
   if (SampleCounter > (SAMPLES_PER_SECOND * CALC_INTERVAL))
   {
      LastSampleCounter    = SampleCounter;
      SampleCounter        = 0;

      LastVoltage[0].Sum   = VoltageSample[0].Sum;
      VoltageSample[0].Sum = 0;

      LastCurrent[0].Sum   = CurrentSample[0].Sum;
      CurrentSample[0].Sum = 0;

      Cycle_Full    = true;
   }
}


void process_rf()
{
   int  node_id     = (rf12_hdr & 0x1F);
   byte remoteid    = rf12_data[0];
   byte command     = rf12_data[1];
   char txbuffer[6] = {0,0,0,0,0,0};

   txbuffer[0] = config.nodeID;

   // put your code here ...

}


void loop()
{
   char i;
   boolean transmit = false;
   float Vrms[3];
   float Irms[3];

   float P[3];

   // skip first cycle to stabilize readings
   if( (tcount==0) && (Cycle_Full==true) )
   {
      tcount     = 1;
      Cycle_Full = false;
   }

   //-------------------------------------------
   if( Cycle_Full==true )
   {
      Cycle_Full = false;

      // Calculation of the root of the mean of the voltage and current squared (rms)
      // Calibration coeficients applied.

      Vrms[0] = sqrt(LastVoltage[0].Sum/LastSampleCounter)/16.0;
      Irms[0] = sqrt(LastCurrent[0].Sum/LastSampleCounter)/32.0;
      P[0]    = Vrms[0]*Irms[0];
      tcount++;

      // transmit
      if( tcount>config.transmitinterval )
      {
         digitalWrite(LEDpin, HIGH);
         tcount = 1;
         counts++;

         #ifdef SERIAL
         Serial.print("Cnt: ");
         Serial.print(LastSampleCounter);
         Serial.print("   Vrms1: ");
         Serial.print(Vrms[0], 2);
         Serial.print("   Current0Sum: ");
         Serial.print(LastCurrent[0].Sum);
         Serial.print("   Irms1: ");
         Serial.print(Irms[0], 2);
         Serial.print("   P1: ");
         Serial.println(P[0], 2);

         Serial.print("Cal V0: ");
         Serial.println(config.Voltage_Calibration[0]);
         #endif
      } // transmit
   } // cycle full
}









