// This tool simulates both the power diversion circuit and the 
// supply meter.  By this means, the interaction between the two
// can be investigated.
//
// Flicker is the effect whereby repeated switching a high-power load on 
// and off can cause the illumination of nearby light bulbs to vary.  
//
// For this tool, all values of current and energy are simulated rather 
// than being measured.  The voltage sensor is only required so that the
// triac can be fired at the right time.  This allows a bulb to be used
// as a convenient output device. 
//
// Although there is no requirement for a physical output device to be 
// used, digital pin (9) is still available for controlling a load via 
// the  triac.  To avoid the need for a mains supply, a DEBUG mode has 
// been included.  This allows the simulalation to be run without any 
// additional hardware.
//
// NORMAL mode is now just a special case of ANTI_FLICKER mode (when the
// A/F crieria are set to their "do nothing" values). 
//
//
//                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
//                  January 2012

#define DEBUG

const byte noOfTriacs = 1; // <-- at least 1, otherwise nothing will happen!

enum polarities {NEGATIVE, POSITIVE};
enum polarities polarityNow;
enum triacStates {TRIAC_ON, TRIAC_OFF}; // the external trigger device is active low
//enum triacStates nextStateOfTriac[noOfTriacs];
enum triacStates nextStateOfTriac;
//enum triacStates triacState[noOfTriacs];
enum triacStates triacState;
enum LEDstates {LED_OFF, LED_ON}; // the LED on pin 13 is active high

const byte outputPinForTrigger = 9;
const byte voltageSensorPin = 2;
const byte currentSensorPin = 1;
const int oneWattHourInJoules = 3600; // 0.001 kWh = 3600 Joules

float surplusPowerSetting = 0; // < ------- now set from user interface
float surplusPowerNow = 0; // < ------- calculated
float maxRateOfChangeOfPower = 500; // in Watts per second
float maxPowerChangePerCycle; //  in Watts per mains cycle

int powerRatingOfDumpLoad; // values are assigned in setup()

long cycleCount = 0;
long cycleCountAtLastOffToOnEvent = 0;
int samplesDuringThisMainsCycle = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths
long noOfSamplePairs = 0;

boolean triggerNeedsToBeArmed = false;
boolean beyondStartUpPhase = false;
boolean overflowDetected = false;
boolean underflowDetected = false;

float energyInBucket_meter = 0;
float energyInBucket_router = 0;     
float energyOffsetOfMeter = 0;
int chargeRegister = 0;

int sampleV;   // as output from the the ADC 0 - 1023 
int lastSampleV;     // stored value from the previous loop (for HP filter)         
float lastFilteredV,filteredV;  //  voltage values after HP-filtering to remove the DC offset

// anti-flicker parameters (NORMAL is now just a special case of ANTI-FLICKER mode)
//
int minCycleCountsBetweenTransitions = 75; // 0 for NORMAL 
                                           // greater than 0 for A/F (in 20mS units)
float offsetOfEnergyThresholds = 0.4;      // 0 for NORMAL
                                           // less than 0.5 for A/F for hysteresis
int energyThreshold_upper = oneWattHourInJoules * (0.5 + offsetOfEnergyThresholds);
int energyThreshold_lower = oneWattHourInJoules * (0.5 - offsetOfEnergyThresholds);
long cycleCountAtLastTransition = 0;
float calErrorOfRouter = 1.0; // 0.9 under-reads by -10%; 1.1 over-reads by 10%.

 // to control the rate of display 
int maxCycleCountForDisplay = 50; // in 20mS units
int cycleCountForDisplay = 0;

#ifdef DEBUG
// components for simulating voltages samples for DEBUG mode
byte noOfVoltageSamplesPerCycle_4debug = 6;
int voltageSamples_4debug[6];
byte vsIndex_4debug = 0;
int perCycleDelayForDebugMode = 20; // <-- nominally 20mS, but can be 
                                    // increased to slow the simulation rate
#endif



void setup()
{  
//  delay(7000); // <--- to avoid losing display with the emonTx 
  
  Serial.begin(9600);
  Serial.setTimeout(250); // the default delay of 1 second feels rather slow
  
  pinMode(outputPinForTrigger, OUTPUT);  
//  pinMode(outputPinForLED, OUTPUT); 
 
  nextStateOfTriac = TRIAC_OFF;
  triacState = TRIAC_OFF;
  powerRatingOfDumpLoad = 3000; // <-- in Watts (or loads can be set individually)
  digitalWrite(outputPinForTrigger, triacState); // keep lamp off until the action starts  

  maxPowerChangePerCycle = (maxRateOfChangeOfPower / cyclesPerSecond); 
  
  
 #ifdef DEBUG    
 /*  populate the voltage sample array for DEBUG use
  */
  float amplitude = 240 * sqrt(2); // in ADC units
  float angleIncrement = (2 * PI) / noOfVoltageSamplesPerCycle_4debug;
  float angleOffset = 0.01; // to avoid sample right at zero crossing point
  float voltage, angle;
  byte index;
  
  for (index = 0; index < noOfVoltageSamplesPerCycle_4debug; index++)
  {
    angle = (index * angleIncrement) + angleOffset;
    voltage = amplitude * sin(angle);
    voltageSamples_4debug[index] = voltage;
  } 
 #endif  
                      
   
  Serial.println(); 
  Serial.println();
  Serial.println ("-----------------------------------------------");
  Serial.print ("Welcome to 'Flicker Demo With Meter'. [freeRam = ");
  Serial.print (freeRam());
  Serial.println (']');
  Serial.println ("Parameters set at compile time:");
  Serial.print ("- simulation mode is                      ");
#ifdef DEBUG 
  Serial.println ("DEBUG ");
#else  
  Serial.println ("normal ");
#endif  
  Serial.print ("- offsetOfEnergyThresholds              = ");
  Serial.println (offsetOfEnergyThresholds);
  Serial.print ("- minCycleCountsBetweenTransitions      = ");
  Serial.println (minCycleCountsBetweenTransitions);
  Serial.print ("- calErrorOfRouter                      = ");
  Serial.println (calErrorOfRouter);
  Serial.print ("- maxRateOfChangeOfPower (Watts/sec)    = ");
  Serial.println (maxRateOfChangeOfPower, 0);
  Serial.print ("- No of mains cycles per display line   = ");
  Serial.println (maxCycleCountForDisplay);
#ifdef DEBUG 
  Serial.print ("- simulation rate, mSec per mains cycle = ");
  Serial.println (perCycleDelayForDebugMode);
#endif  
  Serial.println ();  

  Serial.println("Once running, various parameters are repeatedly displayed:");
  Serial.println ("- selected level of surplus energy (W)");
  Serial.println ("- current level of surplus energy (W)");
  Serial.println ("- on/off state of the triac");
  Serial.print ("- level of the ROUTER'S energy bucket (W), with a cal error of >>> ");
  Serial.println (calErrorOfRouter);  
  Serial.println ("- any underflow events since the last display line");
  Serial.println ("- any overflow events since the last display line");
  Serial.println ("- energy offset of the meter's bucket");
  Serial.println ("- level of the METER's energy bucket");
  Serial.println ("- cumulative total of meter pulses");
  Serial.println ();
  Serial.println("The following commands can be entered at any time, followed by 'CR': ");
  Serial.println("- surplus PV power value, in Watts (can be +ve or -ve)");
  Serial.println("- character 'g', to start the simulation");
  Serial.println("- character 'p', to pause the simulation");
  Serial.println();
  pause();
}


void loop() // each loop is for one pair of V & I measurements
{
  checkForUserInput();
  noOfSamplePairs++;              // for stats only
  samplesDuringThisMainsCycle++;  // for power calculation at the start of each mains cycle

  // store values from previous loop
  lastSampleV=sampleV;            // for digital high-pass filter
  lastFilteredV = filteredV;      // for HPF, used to identify the start of each mains cycle
 
#ifdef DEBUG 
  sampleV = getNextVoltageSample(); // synthesised value
#else
// Get the next raw voltage samples (used for mains-cycle detection only)
  sampleV = analogRead(voltageSensorPin);   
#endif

  // a high-pass filter is used just for determining the start of each mains cycle  
  filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV);   

  // Establish the polarities of the latest and previous filtered voltage samples
  byte polarityOfLastReading = polarityNow;
  if(filteredV >= 0) 
    polarityNow = POSITIVE; 
  else 
    polarityNow = NEGATIVE;


  if (polarityNow == POSITIVE)
  {
    if (polarityOfLastReading != POSITIVE)
    {
      // This is the start of a new mains cycle (just after the +ve going z-c point)
      cycleCount++; // for stats only
      
#ifdef DEBUG
      delay (perCycleDelayForDebugMode);
#endif
      // Determine the energy contribution from the last mains cycle.  There
      // are two factors:
      //
      // 1) add any energy due to surplus PV for the last mains cycle
      // 
      updateSurplusPowerFromPV(); // allows power changes to be introduced gradually
      float latestEnergyContribution  = surplusPowerNow / cyclesPerSecond;
      
      
      // 2) subtract any energy consumed by the dump load during the last mains cycle
      //
      if (triacState == TRIAC_ON) {
        latestEnergyContribution -= powerRatingOfDumpLoad / cyclesPerSecond; }
      
      // A calibration error factor can also be applied, but only to the router's 
      // energy bucket, not to the meter's bucket!
      //
      energyInBucket_router += latestEnergyContribution * calErrorOfRouter;    

      // Apply max and min limits to the router's bucket level
      if (energyInBucket_router > oneWattHourInJoules)
      {
        energyInBucket_router = oneWattHourInJoules;  
        overflowDetected = true;
      }
      else
      if (energyInBucket_router < 0)
      {
        energyInBucket_router = 0;   
        underflowDetected = true;
      }

      // update the energy bucket by a similar amount (with no Calibration Error), and 
      // take appropriate action if either of its end-point limits are reached.
      //
      float lowerLimitOfMeter = energyOffsetOfMeter;
      float upperLimitOfMeter = energyOffsetOfMeter + oneWattHourInJoules;
      
      energyInBucket_meter += latestEnergyContribution; 
      
      if (energyInBucket_meter > upperLimitOfMeter)
      {
        // energy has overflowed, so adjust the meter's offset value by this amount 
        energyOffsetOfMeter += (energyInBucket_meter - upperLimitOfMeter);
      }
      else
      if (energyInBucket_meter < lowerLimitOfMeter)
      {
        // All pre-paid energy has been used up, so another 1 Whour's worth must be
        // purchased.  Alter the meter's offset value by this amount to reflect the 
        // new chunk of energy that is now available for use. 
        //
        chargeRegister++;
        energyOffsetOfMeter -= oneWattHourInJoules;  
      }
      

      // Display relevant data every N mains cycles
      cycleCountForDisplay ++;
      if (cycleCountForDisplay >= maxCycleCountForDisplay)
      { 
        displayData();
        cycleCountForDisplay = 0; 
        overflowDetected = false;
        underflowDetected = false;
      }      
      
      triggerNeedsToBeArmed = true;        
    } // end of processing that is specific to the first +ve Vsample in each new mains cycle
   
    // still processing POSITIVE Vsamples ...
    //
    
    if (triggerNeedsToBeArmed == true)
    {
      // check to see whether the trigger device can now be reliably armed
      if(filteredV > 50) // 20V min for Motorola trigger
      {
        if (energyInBucket_router > energyThreshold_upper)       
        {
          increaseLoadIfPossible(); // to reduce the level
        }
        else
        if (energyInBucket_router < energyThreshold_lower)       
        {
          decreaseLoadIfPossible(); // to increase the level
        }            
        
        // and clear the flag.
        triggerNeedsToBeArmed = false;
        
      }
    }    
  }  // end of processing that is specific to positive Vsamples
  else
  {
    if (polarityOfLastReading != NEGATIVE)
    {
      // Simulate the effect of the triac coming on or off at the -ve going
      // zero-crossing point that has just occurred. 
      //
      if (nextStateOfTriac == TRIAC_ON) {      
        triacState = TRIAC_ON; }
      else {  
        triacState = TRIAC_OFF; }          
 
      // then set the Arduino's output pin accordingly (only triac 0 is supported)
      digitalWrite(outputPinForTrigger, triacState);     
    }
  }
} // end of loop()


void increaseLoadIfPossible()
{
  // if permitted by anti-flicker restrictions, turn the triac on.
  if (cycleCount > cycleCountAtLastTransition + minCycleCountsBetweenTransitions) 
  {  
    if (triacState == TRIAC_OFF)
    {
      nextStateOfTriac = TRIAC_ON;
      cycleCountAtLastTransition = cycleCount;
    }
  }
}
 
byte decreaseLoadIfPossible()
{
  // if permitted by anti-flicker restrictions, turn the triac off.
  if (cycleCount > cycleCountAtLastTransition + minCycleCountsBetweenTransitions) 
  {  
    if (triacState == TRIAC_ON)
    {
      nextStateOfTriac = TRIAC_OFF;
      cycleCountAtLastTransition = cycleCount;
    }
  }
}
 
 
void updateSurplusPowerFromPV()
{
  if (surplusPowerNow < surplusPowerSetting)
    {
    //  power is to be increased 
    if ((surplusPowerSetting - surplusPowerNow) > maxPowerChangePerCycle) {
      surplusPowerNow += maxPowerChangePerCycle; }
    else {
      surplusPowerNow = surplusPowerSetting; }      
  }
  else
  if (surplusPowerNow > surplusPowerSetting) {
    //  power is to be decreased 
    if ((surplusPowerNow - surplusPowerSetting) > maxPowerChangePerCycle) {
      surplusPowerNow -= maxPowerChangePerCycle; }
    else {
      surplusPowerNow = surplusPowerSetting; }
  }
}



void displayData()
{
  cycleCountForDisplay = 0; 
  int intVal;
  char strVal[10];
  byte lenOfStrVal;
      
  // first, display the surplus PV setting as a right-justified integer
  intVal = surplusPowerSetting; // apply integer rounding 
  strVal[6];
  itoa(intVal, strVal, 10); // decimal conversion to string
  lenOfStrVal;
  lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
  for (int i = 0; i < (5 - lenOfStrVal); i++) 
  {
    Serial.print(' '); 
  }
  Serial.print(strVal);       
       
  //  display the current rate of surplus PV 
  intVal = surplusPowerNow; // apply integer rounding 
  itoa(intVal, strVal, 10); // decimal conversion to string
  lenOfStrVal;
  lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
  Serial.print("  "); 
  for (int i = 0; i < (5 - lenOfStrVal); i++) { 
    Serial.print(' '); }
  Serial.print(strVal);       
       
  // now display the state of the triac
  if (triacState == TRIAC_ON) {
    Serial.print ("   ON"); }
  else {
    Serial.print ("  off"); }
             
       
  //  display the current level in the energy bucket
  intVal = energyInBucket_router; // apply integer rounding 
  itoa(intVal, strVal, 10); // decimal conversion to string
  lenOfStrVal;
  lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
  Serial.print("  "); 
  for (int i = 0; i < (5 - lenOfStrVal); i++) {
    Serial.print(' '); }
  Serial.print(strVal);       
        
  // display whether underflow has been detected since the last display line
  Serial.print("   "); 
  if (underflowDetected) {
    Serial.print('U'); }
  else {
    Serial.print('.'); }
 
  // display whether overflow has been detected since the last display line
  Serial.print("  "); 
  if (overflowDetected) {
    Serial.print("O"); }
  else {
    Serial.print("."); }
        
  // display the energy offset of the meter
  intVal = energyOffsetOfMeter; // apply integer rounding 
  itoa(intVal, strVal, 10); // decimal conversion to string
  lenOfStrVal;
  lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
  Serial.print("  "); 
  for (int i = 0; i < (8 - lenOfStrVal); i++) {
    Serial.print(' '); }
  Serial.print(strVal);       
          
  // display the energy level of the meter
  intVal = energyInBucket_meter; // apply integer rounding 
  itoa(intVal, strVal, 10); // decimal conversion to string
  lenOfStrVal;
  lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
  Serial.print("  "); 
  for (int i = 0; i < (8 - lenOfStrVal); i++) {
    Serial.print(' '); }
  Serial.print(strVal);       
          
  // display the state of the charge register
  intVal = chargeRegister; // apply integer rounding 
  itoa(intVal, strVal, 10); // decimal conversion to string
  lenOfStrVal;
  lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
  Serial.print("  "); 
  for (int i = 0; i < (3 - lenOfStrVal); i++) {
    Serial.print(' '); }
  Serial.print(strVal);       
                  
  Serial.println(); 
}

void checkForUserInput()
{
  if (Serial.available() )
  {
    char inbuf[8] = {0,0,0,0,0,0,0,0}; 
    Serial.readBytesUntil('\n', inbuf, 8); 
    
    if (strcmp(inbuf, "p") == 0)
    {
      pause();
    }
    else
    {
      int value = atoi(inbuf);    
      if ((value == 0) && (inbuf[0] != '0'))
      {
//        Serial.println("Sorry, instruction not recognised :-( "); 
      }  
      else
      {
        float valFloat = atof(inbuf);
        if (valFloat > 32000)
        {
//          Serial.print("Too big, 32kW max please! "); 
        }
        else
        {
          surplusPowerSetting = valFloat;
          Serial.println("NewPower"); 
        }
      }
    }
  }
}

void pause()
{
  char inbuf[10];
  boolean done = false;
  
  Serial.println ("paused");
  while (!done)
  { 
    while (!Serial.available()) 
    {
      delay(100); 
    }
    for (int i = 0; i < 10; i++) 
    {  
      inbuf[i] = 0;
    }
    Serial.readBytesUntil(0, inbuf, 10); 
    if (strcmp(inbuf, "g") == 0)
    {
      done = true;     
    }
    else
    {
//      Serial.println("To re-start simulation, enter 'g' followed by CR");
    }
  }
}

int freeRam () {
  extern int __heap_start, *__brkval; 
  int v; 
  return (int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval); 
}


#ifdef DEBUG
// function to synthesise voltage samples for DEBUG mode only
float getNextVoltageSample()
{
  int voltageSample;

  voltageSample = voltageSamples_4debug[vsIndex_4debug];
  voltageSample+= 500.0; // not critical, approx mid-way in the ADC's input range.
  
  vsIndex_4debug++;  
  if (vsIndex_4debug >= noOfVoltageSamplesPerCycle_4debug) 
  {
    vsIndex_4debug = 0;
  }

  return voltageSample;
}
#endif

