// Flicker is the effect whereby repeated switching a high-power load on 
// and off can cause the illumination of nearby light bulbs to vary.  
//
// For this tool, all values of current and energy are simulated rather 
// than being measured.  The voltage sensor is only required so that the
// triac can be fired at the right time.  This allows a bulb to be used
// as a convenient output device. 
//
// In the _multiLoad version, it is not envisaged that a physical output 
// device will not be used.  However, the standard trigger pin (9) is still
// available for controlling the first triac.  To avoid the need for a 
// mains supply, a DEBUG mode has been added.  This allows the simulalation 
// to be run without any additional hardware.
//
// The on-board LED now shows whenever the energy bucket overflows or
// underflows.  When overflow occurs, there is likely to be some loss of 
// energy to the grid; underflow is likely to result in some cost to the 
// user for energy consumed.
//
// NORMAL mode is now just a special case of ANTI_FLICKER mode (when the 
// upper and lower threshold are co-incident and the minimum time between 
// transitions is zero). 
//
// This tool provides a convenient place for experimenting with anti-flicker
// algorithms because:
//
// - the energy bucket is in Joules (3600J), and it uses FP maths;
// - it's a simulation, so any value of 'surplus PV' can be arranged.
// - any number of triacs and loads can be supported
// - with a simulation, everything is repeatable and can be analysed
// - mimimal real energy is consumed during trials!
//
// Version 2 imposes a maximum rate of change whenever the amount of 
// (simulated) surplus PV is changed.  This allows the anti-flicker 
// performance to be more meaningfully assessed.
//
//                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
//                  January 2012

#define DEBUG

const byte noOfTriacs = 4; // <-- at least 1, otherwise nothing will happen!

enum polarities {NEGATIVE, POSITIVE};
enum polarities polarityNow;
enum triacStates {TRIAC_ON, TRIAC_OFF}; // the external trigger device is active low
enum triacStates nextStateOfTriac[noOfTriacs];
enum triacStates triacState[noOfTriacs];
enum LEDstates {LED_OFF, LED_ON}; // the LED on pin 13 is active high

byte outputPinForLED = 13;
byte outputPinForTrigger = 9;
byte voltageSensorPin = 2;
byte currentSensorPin = 1;

float surplusPowerSetting = 0; // < ------- now set from user interface
float surplusPowerNow = 0; // < ------- calculated
float maxRateOfChangeOfPower = 300; // in Watts per second
float maxPowerChangePerCycle; //  in Watts per mains cycle

int powerRatingOfDumpLoad[noOfTriacs]; // values are assigned in setup()

long cycleCount = 0;
long cycleCountAtLastOffToOnEvent = 0;
int samplesDuringThisMainsCycle = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths
long noOfSamplePairs = 0;

boolean triggerNeedsToBeArmed = false;
boolean beyondStartUpPhase = false;
boolean overflowDetected = false;
boolean underflowDetected = false;


float energyInBucket = 0;                                 
int capacityOfEnergyBucket = 3600; // 0.001 kWh = 3600 Joules
int sampleV;   // as output from the the ADC 0 - 1023 
int lastSampleV;     // stored value from the previous loop (for HP filter)         
float lastFilteredV,filteredV;  //  voltage values after HP-filtering to remove the DC offset

// anti-flicker parameters (NORMAL is now just a special case of ANTI-FLICKER mode)
//
int minCycleCountsBetweenTransitions = 75; // 0 for NORMAL 
                                           // greater than 0 for A/F (in 20mS units)
float offsetOfEnergyThresholds = 0.4;      // 0 for NORMAL
                                           // less than 0.5 for A/F for hysteresis
int energyThreshold_upper = capacityOfEnergyBucket * (0.5 + offsetOfEnergyThresholds);
int energyThreshold_lower = capacityOfEnergyBucket * (0.5 - offsetOfEnergyThresholds);
long cycleCountAtLastTransition = 0;

 // to control the rate of display 
int maxCycleCountForDisplay = 50; // in 20mS units
int cycleCountForDisplay = 0;

#ifdef DEBUG
// components for simulating voltages samples for DEBUG mode
byte noOfVoltageSamplesPerCycle_4debug = 45;
int voltageSamples_4debug[45];
byte vsIndex_4debug = 0;
int perCycleDelayForDebugMode = 20; // <-- nominally 20mS, but can be 
                                    // increased to slow the simulation rate
#endif



void setup()
{  
//  delay(7000); // <--- to avoid losing display with the emonTx 
  
  Serial.begin(9600);
  Serial.setTimeout(250); // the default delay of 1 second feels rather slow
  
  pinMode(outputPinForTrigger, OUTPUT);  
  pinMode(outputPinForLED, OUTPUT); 
 
  for (int i = 0; i < noOfTriacs; i++)
  {
    nextStateOfTriac[i] = TRIAC_OFF;
    triacState[i] = TRIAC_OFF;
    powerRatingOfDumpLoad[i] = 3000; // <-- in Watts (or loads can be set individually)
  }
  digitalWrite(outputPinForTrigger, triacState[0]); // keep lamp off until the action starts  

  maxPowerChangePerCycle = (maxRateOfChangeOfPower / cyclesPerSecond); 
  
  
 #ifdef DEBUG    
 /*  populate the voltage sample array for DEBUG use
  */
  float amplitude = 240 * sqrt(2); // in ADC units
  float angleIncrement = (2 * PI) / noOfVoltageSamplesPerCycle_4debug;
  float angleOffset = 0.01; // to avoid sample right at zero crossing point
  float voltage, angle;
  byte index;
  
  for (index = 0; index < noOfVoltageSamplesPerCycle_4debug; index++)
  {
    angle = (index * angleIncrement) + angleOffset;
    voltage = amplitude * sin(angle);
    voltageSamples_4debug[index] = voltage;
  } 
 #endif  
                      
   
  Serial.println(); 
  Serial.println();
  Serial.println ("-----------------------------------------------");
  Serial.println ("Welcome to the Flicker Demo tool for multiple loads. ");
  Serial.println ("Please note that various settings can be changed at compile time:  ");
  Serial.println ();
  Serial.println ("- include the #define DEBUG line to operate without a voltage sensor   ");
  Serial.println ("- set the number of triacs, noOfTriacs  ");
  Serial.println ("- set the rating of each 'dump load', powerRatingOfDumpLoad[i]  ");
  Serial.println ("- set the energy thresholds for a/f, offsetOfEnergyThresholds  ");
  Serial.println ("- set the max flick rate for a/f, minCycleCountsBetweenTransitions  ");
  Serial.println ("- set the max rate at which surplus power may change (Watts/sec)");
  Serial.println ("- apply 'normal' criteria, which is now a special case of anti-flicker");
  Serial.println ("- set the display rate, maxCycleCountForDisplay ");
  Serial.println ("- set the simulation rate, perCycleDelayForDebugMode (DEBUG mode only) ");
  Serial.println ();
  Serial.println("Once running, various parameters are repeatedly displayed:");
  Serial.println ("- The selected level of surplus energy (W)");
  Serial.println ("- The current level of surplus energy (W)");
  Serial.println ("- The on/off state of each triac");
  Serial.println ("- the level in the energy bucket");
  Serial.println ("- any overflow events since the last display line?");
  Serial.println ("- any underflow events since the last display line?");
  Serial.println ();
  Serial.println("The following commands can be entered at any time, followed by 'CR': ");
  Serial.println("- the surplus PV power value, in Watts (max 32kW)");
  Serial.println("- the character 'g', to start the simulation");
  Serial.println("- the character 'p', to pause the simulation");
  Serial.println();
  pause();
}


void loop() // each loop is for one pair of V & I measurements
{
  checkForUserInput();
  noOfSamplePairs++;              // for stats only
  samplesDuringThisMainsCycle++;  // for power calculation at the start of each mains cycle

  // store values from previous loop
  lastSampleV=sampleV;            // for digital high-pass filter
  lastFilteredV = filteredV;      // for HPF, used to identify the start of each mains cycle
 
#ifdef DEBUG 
  sampleV = getNextVoltageSample(); // synthesised value
#else
// Get the next raw voltage samples (used for mains-cycle detection only)
  sampleV = analogRead(voltageSensorPin);   
#endif

  // a high-pass filter is used just for determining the start of each mains cycle  
  filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV);   

  // Establish the polarities of the latest and previous filtered voltage samples
  byte polarityOfLastReading = polarityNow;
  if(filteredV >= 0) 
    polarityNow = POSITIVE; 
  else 
    polarityNow = NEGATIVE;


  if (polarityNow == POSITIVE)
  {
    if (polarityOfLastReading != POSITIVE)
    {
      // This is the start of a new mains cycle (just after the +ve going z-c point)
      cycleCount++; // for stats only
      
#ifdef DEBUG
      delay (perCycleDelayForDebugMode);
#endif
      
      // Update the level in the energy bucket.  For this simulation, there are 
      // just two factors:
      //
      // -> the amount of surplus PV during the last mains cycle 
      // -> the effect of any dump loads during the last mains cycle
      //
      float latestEnergyContribution = updateSurplusPowerFromPV();
      energyInBucket += latestEnergyContribution;    
      
      for (int i = 0; i < noOfTriacs; i++)
      {
        if (triacState[i] == TRIAC_ON)
        {
          energyInBucket -= (powerRatingOfDumpLoad[i] / cyclesPerSecond); 
        }
      }

      // Apply max and min limits to bucket's level
      if (energyInBucket > capacityOfEnergyBucket)
      {
        energyInBucket = capacityOfEnergyBucket;  
        digitalWrite(outputPinForLED, LED_ON); // overflow  
        overflowDetected = true;
      }
      else
      if (energyInBucket < 0)
      {
        energyInBucket = 0;   
        digitalWrite(outputPinForLED, LED_ON); // underflow   
        underflowDetected = true;
      }
      else
      {
        digitalWrite(outputPinForLED, LED_OFF); // mid-range     
      }

      triggerNeedsToBeArmed = true;   // the trigger is armed every mains cycle

      // -----------------------------------------------------
      // Code to display relevant data every N mains cycles
      //
      cycleCountForDisplay ++;
      if (cycleCountForDisplay >= maxCycleCountForDisplay)
      { 
        cycleCountForDisplay = 0; 
        int intVal;
        char strVal[6];
        byte lenOfStrVal;
        
        // first, display the surplus PV setting as a right-justified integer
        intVal = surplusPowerSetting; // apply integer rounding 
        strVal[6];
        itoa(intVal, strVal, 10); // decimal conversion to string
        lenOfStrVal;
        lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
        for (int i = 0; i < (5 - lenOfStrVal); i++) 
        {
          Serial.print(' '); 
        }
        Serial.print(strVal);       
       
        // now, display the current rate of surplus PV 
        intVal = surplusPowerNow; // apply integer rounding 
        strVal[6];
        itoa(intVal, strVal, 10); // decimal conversion to string
        lenOfStrVal;
        lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
        Serial.print(", "); 
        for (int i = 0; i < (5 - lenOfStrVal); i++) 
        {
          Serial.print(' '); 
        }
        Serial.print(strVal);       
       
        // now display the state of each triac
        for (int i = 0; i < noOfTriacs; i++)
        {
          if (triacState[i] == TRIAC_ON) {
            Serial.print (", ON "); }
          else {
            Serial.print (", off"); }
        }     
       
        // now display the current level in the energy bucket
        intVal = energyInBucket; // apply integer rounding 
        strVal[6];
        itoa(intVal, strVal, 10); // decimal conversion to string
        lenOfStrVal;
        lenOfStrVal = strlen(strVal); // determine length of 'energy as string'
             
        Serial.print(", "); 
        for (int i = 0; i < (5 - lenOfStrVal); i++) 
        {
          Serial.print(' '); 
        }
        Serial.print(strVal);       
        
        // display whether overflow has been detected since the last display line
        Serial.print("  "); 
        if (overflowDetected)
        {
          Serial.print("  O");
        }
        else
        {
          Serial.print("  .");
        }
        
       // display whether underflow has been detected since the last display line
        if (underflowDetected)
        {
          Serial.print("  U");
        }
        else
        {
          Serial.print("  .");
        }
 
        overflowDetected = false;
        underflowDetected = false;
        Serial.println(); 
      }
      
      // End of code for display every N cycles
      // --------------------------------
      
      
    } // end of processing that is specific to the first +ve Vsample in each new mains cycle
   
    // still processing POSITIVE Vsamples ...
    //
    
    if (triggerNeedsToBeArmed == true)
    {
      // check to see whether the trigger device can now be reliably armed
      if(filteredV > 50) // 20V min for Motorola trigger
      {
        if (energyInBucket > energyThreshold_upper)       
        {
          increaseLoadIfPossible(); // to reduce the level
        }
        else
        if (energyInBucket < energyThreshold_lower)       
        {
          decreaseLoadIfPossible(); // to increase the level
        }            
        
        // and clear the flag.
        triggerNeedsToBeArmed = false;
        
      }
    }    
  }  // end of processing that is specific to positive Vsamples
  else
  {
    if (polarityOfLastReading != NEGATIVE)
    {
      // Simulate the effect of the triac coming on or off at the -ve going
      // zero-crossing point that has just occurred. 
      //
      for (int i = 0; i < noOfTriacs; i++)
      {
        if (nextStateOfTriac[i] == TRIAC_ON) {      
          triacState[i] = TRIAC_ON; }
        else {  
          triacState[i] = TRIAC_OFF; }          
      }
 
      // then set the Arduino's output pin accordingly (only triac 0 is supported)
      digitalWrite(outputPinForTrigger, triacState[0]);     
    }
  }
} // end of loop()


void increaseLoadIfPossible()
{
  // if permitted by anti-flicker restrictions, turn on the lowest valued
  // triac that is not already on.
  boolean done = false;
  
  if (cycleCount > cycleCountAtLastTransition + minCycleCountsBetweenTransitions) 
  {  
    for (int i = 0; i < noOfTriacs; i++)
    {
      if (!done)
      {
        if (triacState[i] == TRIAC_OFF)
        {
          nextStateOfTriac[i] = TRIAC_ON;
          cycleCountAtLastTransition = cycleCount;
          done = true;
        }
      }
    }
  }
}
 
byte decreaseLoadIfPossible()
{
  // if permitted by anti-flicker restrictions, turn off the highest valued
  // triac that is not already off.
  boolean done = false;
  
  if (cycleCount > cycleCountAtLastTransition + minCycleCountsBetweenTransitions) 
  {  
    for (int i = (noOfTriacs -1); i >= 0; i--)
    {
      if (!done)
      {
        if (triacState[i] == TRIAC_ON)
        {
          nextStateOfTriac[i] = TRIAC_OFF;
          cycleCountAtLastTransition = cycleCount;
          done = true;
        }
      }
    }
  }
}
 
 
float updateSurplusPowerFromPV()
{
  if (surplusPowerNow < surplusPowerSetting)
    {
    //  power is to be increased 
    if ((surplusPowerSetting - surplusPowerNow) > maxPowerChangePerCycle)
    {
      surplusPowerNow += maxPowerChangePerCycle;
    }
    else
    {
      surplusPowerNow = surplusPowerSetting;
    }      
  }
  else
  if (surplusPowerNow > surplusPowerSetting)
  {
    //  power is to be decreased 
    if ((surplusPowerNow - surplusPowerSetting) > maxPowerChangePerCycle)
    {
      surplusPowerNow -= maxPowerChangePerCycle;
    }
    else
    {
      surplusPowerNow = surplusPowerSetting;
    }
  }
  else
  {
    // no change, surplus power matches the latest setting
  }
  
  float surplusEnergyPerCycle = surplusPowerNow / cyclesPerSecond;
  return (surplusEnergyPerCycle);
}


void checkForUserInput()
{
  if (Serial.available() )
  {
    char inbuf[8] = {0,0,0,0,0,0,0,0}; 
    Serial.readBytesUntil('\n', inbuf, 8); 
    
    if (strcmp(inbuf, "p") == 0)
    {
      pause();
    }
    else
    {
      int value = atoi(inbuf);    
      if ((value == 0) && (inbuf[0] != '0'))
      {
        Serial.println("Sorry, instruction not recognised :-( "); 
      }  
      else
      {
        float valFloat = atof(inbuf);
        if (valFloat > 32000)
        {
          Serial.print("Too big, 32kW max please! "); 
        }
        else
        {
          surplusPowerSetting = valFloat;
          Serial.println("Surplus power setting changed"); 
        }
      }
    }
  }
}

void pause()
{
  char inbuf[10];
  boolean done = false;
  
  Serial.println ("paused ...");
  while (!done)
  { 
    while (!Serial.available()) 
    {
      delay(100); 
    }
    for (int i = 0; i < 10; i++) 
    {  
      inbuf[i] = 0;
    }
    Serial.readBytesUntil(0, inbuf, 10); 
    if (strcmp(inbuf, "g") == 0)
    {
      done = true;     
    }
    else
    {
      Serial.println("To re-start simulation, enter 'g' followed by CR");
    }
  }
}

#ifdef DEBUG
// function to synthesive voltage samples for DEBUG mode only
float getNextVoltageSample()
{
  int voltageSample;

  voltageSample = voltageSamples_4debug[vsIndex_4debug];
  voltageSample+= 500.0; // not critical, approx mid-way in the ADC's input range.
  
  vsIndex_4debug++;  
  if (vsIndex_4debug >= noOfVoltageSamplesPerCycle_4debug) 
  {
    vsIndex_4debug = 0;
  }

  return voltageSample;
}
#endif

