//
//                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
//                  January 2013

#include <Arduino.h> // probably a good idea to include this

#include <TimerOne.h>
#define ADC_TIMER_PERIOD 1000 // uS

enum polarities {NEGATIVE, POSITIVE};

// general global variables
const byte voltageSensorPin = 2;  // analogue
const byte startUpPeriod = 5; // in seconds, to allow HP filters to settle
const int DCoffset_I = 512; // for calculating "real power", a nominal value is fine

boolean beyondStartUpPhase = false;
long DCoffset_V_long = 512L * 256; // nominal mid-point value of ADC @ x256 scale
long DCoffset_V_min;
long DCoffset_V_max;
long cumVdeltasThisCycle_long;   // <<--- for LPF 
enum polarities polarityOfLastSampleV; // for zero-crossing detection
int sampleV;
long noOfSamplesDuringLastPeriod; // for deferred printout

// for the interlock mechanism between the main processor and the ADC 
volatile boolean dataReady = false;

// for the display
byte days;
byte hours;
byte mins;
unsigned long startTime;


void setup()
{  
  Serial.begin(9600);
  Serial.setTimeout(250);
  Serial.println();
  Serial.println();
  Serial.println("-------------------------------------");
  Serial.println("Sketch ID:      MainsFreqChecker.ino");
     
  // define upper and lower limits to ensure correct startup of the LP filter which 
  // identifies DC offset in the stream of voltage samples
  DCoffset_V_min = (long)(512L - 100) * 256; // mid-point of ADC minus a working margin
  DCoffset_V_max = (long)(512L + 100) * 256; // mid-point of ADC plus a working margin

  // set up the ADC for fixed timer operation
  ADMUX = 0x40 + voltageSensorPin; // 
  ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2);  // ADC's clock is system clock / 128
  ADCSRA |= (1 << ADEN);                       // Enable the ADC 
  Timer1.initialize(ADC_TIMER_PERIOD);   // set Timer1 interval
  Timer1.attachInterrupt( timerIsr );    // declare timerIsr() as interrupt service routine

  Serial.print ("ADC mode:       ");
  Serial.print (ADC_TIMER_PERIOD);
  Serial.println ( "uS fixed timer");

// section to record the start time  
  char inbuf[10];
  char dateStr[20];
  byte noOfBytes;
  byte len;
  char hoursStr[3];
  char* colonPtr; 
  char minsStr[3];
  boolean finished;
  
  Serial.println("-----");  
  Serial.println ("Please enter the date (not checked)");
  while (!Serial.available()) 
  {
    delay(100); 
  }
  for (int i = 0; i < 20; i++) { dateStr[i] = 0; }
  Serial.readBytesUntil(0, dateStr, 20); 
  Serial.print ("Day 0 is: ");
  Serial.println (dateStr);
  
  finished = false;  
  while (!finished)
  {    
    Serial.println ("Please enter the time now in the format HH:MM ");
    while (!Serial.available()) 
    {
      delay(100); 
    }

    for (int i = 0; i < 10; i++) { inbuf[i] = 0; }
    Serial.readBytesUntil(0, inbuf, 10); 
    Serial.print ("Start time is: ");      
    Serial.println (inbuf);
    
    for (int i = 0; i < 3; i++) { hoursStr[i] = 0; }
    len = strcspn(inbuf, ":"); // returns the no of chars before the ':'
    strncpy(hoursStr, inbuf, len); // copy the same no of characters
    hours = atoi(hoursStr);
  
    for (int i = 0; i < 3; i++) { minsStr[i] = 0; } 
    colonPtr = strchr(inbuf, ':'); // returns a pointer to the ':' character
    colonPtr++;                // pointer now points to start of mins field
    strcpy(minsStr, colonPtr);
    mins = atoi(minsStr);  
    
    if (((hours >= 0) && (hours <= 23)) && ((mins >= 0) && (mins <= 59)))          
    {  
      finished = true;
    }
    else
    {
      Serial.print ("Sorry, that input has not been recognised, ");
      Serial.println ("please try again ");   
    }
  }
  
  days = 0;
  startTime = millis();
  Serial.println("Recording will start in 5 seconds ...");
  Serial.println();
}

void timerIsr(void)
{                                                                                
  sampleV = ADC;
  ADCSRA |= (1<<ADSC);             
  dataReady = true; 
}


void loop()             
{ 
  if (dataReady)   // flag is set after every ADC conversion
  {
    dataReady = false; // reset the flag
    allGeneralProcessing(); // executed once for each voltage sample
    if (dataReady)
    {
      Serial.print('$'); // out of time!
    }
  }   
} // end of loop()



void allGeneralProcessing()
{
  // remove DC offset from the raw voltage sample by subtracting the accurate value 
  // as determined by a LP filter.
  long sampleVminusDC_long = ((long)sampleV<<8) - DCoffset_V_long; 
  static long samplesDuringThisPeriod = 0; // accumulated vaule
  static int monitorPeriod_counter = 0;    // # of mains cycles
  static int OneMinute_counter = 0;      // 
  
  // determine polarity, to aid the logical flow
  enum polarities polarityNow;   
  if(sampleVminusDC_long > 0) { 
    polarityNow = POSITIVE; }
  else { 
    polarityNow = NEGATIVE; }

  if (polarityNow == POSITIVE) 
  {                           
    if (polarityOfLastSampleV != POSITIVE)
    {
      // This is the start of a new +ve half cycle (just after the zero-crossing point)
      
      if (beyondStartUpPhase)
      {   
        
        OneMinute_counter++;
        if (OneMinute_counter == 3000) // one minute is 3000 mains cycles
        {
          OneMinute_counter = 0;
          updateTimestampCounters();
        }
      
        monitorPeriod_counter++;
        if (monitorPeriod_counter == 750) // one minute is 3000 mains cycles
        {
          monitorPeriod_counter = 0;
          noOfSamplesDuringLastPeriod = samplesDuringThisPeriod; // for printing next time
          samplesDuringThisPeriod = 0;         
        }
        
        processDisplay(monitorPeriod_counter);
      }
      else
      {
        if((millis() - startTime) > 5000)
        {
          beyondStartUpPhase = true;
          OneMinute_counter = 0;
          monitorPeriod_counter = 0;
          samplesDuringThisPeriod = 0;  
//          printTimestamp();
//          Serial.println("(start)");       
        }
      }
    } // end of processing that is specific to the first Vsample in each +ve half cycle  
  }
  else // the polatity of this sample is negative
  {     
    if (polarityOfLastSampleV != NEGATIVE)
    {
      // This is the start of a new -ve half cycle (just after the zero-crossing point)
      // This is a convenient point to update the Low Pass Filter for DC-offset removal
      long previousOffset = DCoffset_V_long;
      DCoffset_V_long = previousOffset + (0.01 * cumVdeltasThisCycle_long); 
      cumVdeltasThisCycle_long = 0;
      
      // To ensure that the LPF will always start up correctly when 240V AC is available, its
      // output value needs to be prevented from drifting beyond the likely range of the 
      // voltage signal.  This avoids the need to use a HPF as was done for initial Mk2 builds.
      //
      if (DCoffset_V_long < DCoffset_V_min) {
        DCoffset_V_long = DCoffset_V_min; }
      else  
      if (DCoffset_V_long > DCoffset_V_max) {
        DCoffset_V_long = DCoffset_V_max; }
           
    } // end of processing that is specific to the first Vsample in each -ve half cycle
  } // end of processing that is specific to samples where the voltage is positive
  
  // processing for EVERY pair of samples
  //
  samplesDuringThisPeriod++;
  cumVdeltasThisCycle_long += sampleVminusDC_long; // for use with LP filter
  
  // store items for use during next loop
  polarityOfLastSampleV = polarityNow;  // for identification of half cycle boundaries
}


void updateTimestampCounters()
{
  mins++;
  if (mins == 60)
  {
    mins = 0;
    hours++;
  }
  if (hours == 24)
  {
    hours = 0;
    days++;
  }    
};


void processDisplay(int count)
{
  static boolean firstLine = true;
  char tempStr[5];
  
  switch(count)
  {
    case 1:
      Serial.print( "day "); 
      break;
    case 2:
      Serial.print (days);
      break;
    case 3:
      Serial.print (", ");
      break;
    case 4:
      Serial.print (hours);
      break;
    case 5:
      Serial.print (':');
      break;
    case 6:
      itoa(mins, tempStr, 10);
      if (strlen(tempStr) == 1) {
        Serial.print ('0'); }       
      Serial.print (mins);
      break;
    case 7:
      Serial.print (", ");
      break;
    case 8:
      if (firstLine)
      {
        Serial.println ("(start)");
        firstLine = false;
      }
      else
      { 
        Serial.println (noOfSamplesDuringLastPeriod);
      }
      break;
    default:
      break;  
  }
}

