/* 
 * A tool which monitors and displays the LED activity of a digital supply meter
 * using a Light Dependent Resistor (LDR).  LED events are categorised into 
 * PULSE, ON and OFF.  Also displays the energy flow at the meter using standard
 * input circuitry.
 *
 *   Robin Emley (calypso_rae on Open Energy Monitor Forum)
 *   July 2012
 *
                  ----------------> +5V
                  |
                  /
                  \ 8K2
                  /
                  |
             ---------------------> dig 2
             |       |
        -->  /       |
        -->  \       _
        LDR  /       -  0.01uF
             |       |
             ----------------------> GND
*/

#define POSITIVE 1
#define NEGATIVE 0
#define ON 0  // output from the LDR is active low
#define OFF 1

byte ledDetectorPin = 2;  // digital 
byte ledRepeaterPin = 13;  // digital 
byte voltageSensorPin = 2;  // analogue 
byte currentSensorPin = 1;  // analogue 

long cycleCount = 0;
int samplesDuringThisMainsCycle = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths

long noOfSamplePairs = 0;
byte polarityNow;
boolean beyondStartUpPhase = false;

// the'energy bucket' mimics the operation of a digital supply meter at the grid connection point.
float energyInBucket = 0;     
float prevDisplayedLevel;
int capacityOfEnergyBucket = 3600; // 0.001 kWh = 3600 Joules

int sampleV,sampleI;   // voltage & current samples are integers in the ADC's input range 0 - 1023 
int lastSampleV;     // stored value from the previous loop (HP filter is for voltage samples only)         

double lastFilteredV,filteredV;  //  voltage values after filtering to remove the DC offset,
                                 //   and the stored values from the previous loop             
double prevDCoffset;          // <<--- for LPF to quantify the DC offset
double DCoffset;              // <<--- for LPF 
double cumVdeltasThisCycle;   // <<--- for LPF 
double sumP;   //  cumulative sum of power calculations within this mains cycles

float POWERCAL;  // To convert the product of raw V & I samples into Joules.  
                  
byte ledState, prevLedState;
boolean ledRecentlyOnFlag;
unsigned long ledOnAt;


void setup()
{  
  Serial.begin(9600);
//  Serial.begin(115200);
  Serial.println(); 
  pinMode(ledDetectorPin, INPUT);
  pinMode(ledRepeaterPin, OUTPUT);  

  POWERCAL = 0.085; // Joules per ADC-unit squared.  Used for converting the product of 
                    // voltage and current samples into Joules.
                        
  ledRecentlyOnFlag = false;  
}


void loop() // each loop is for one pair of V & I measurements
{
  noOfSamplePairs++;              // for stats only
  samplesDuringThisMainsCycle++;  // for power calculation at the start of each mains cycle

  lastSampleV=sampleV;            // save previous voltage values for digital high-pass filter
  lastFilteredV = filteredV;      

  sampleV = analogRead(voltageSensorPin);                 //Read in raw voltage signal
  sampleI = analogRead(currentSensorPin);                 //Read in raw current signal

  // Apply digital high pass filter to remove 2.5V DC offset from voltage sample.
  // The HPF is used to group samples into mains cycles
  filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV);
   

  // Establish the polarities of the latest and previous filtered voltage samples
  byte polarityOfLastReading = polarityNow;
  if(filteredV >= 0) 
    polarityNow = POSITIVE; 
  else 
    polarityNow = NEGATIVE;
 
  if ((polarityNow == POSITIVE) && (polarityOfLastReading == NEGATIVE))
  {
    // This is the start of a new mains cycle, which occurs every 20mS
    cycleCount++;       
    checkLedStatus();

    // update the Low Pass Filter which is used for the removal of DC-offset
    prevDCoffset = DCoffset;
    DCoffset = prevDCoffset + (0.01 * cumVdeltasThisCycle); 

    //  Calculate the real power of all instantaneous measurements taken during the 
    //  previous mains cycle, and determine the gain (or loss) in energy.
    double realPower = POWERCAL * sumP / (float)samplesDuringThisMainsCycle;
    float realEnergy = realPower / cyclesPerSecond;

    if((cycleCount % 50) == 5) 
    {
      Serial.print(cycleCount /50); // time in seconds since startup
      Serial.print(",  ");
      Serial.println((long)energyInBucket); // Watts
    }      

    if (beyondStartUpPhase == true)
    {  
      // Providing that the DC-blocking filters have had sufficient time to settle,    
      // add this power contribution to the energy bucket
      energyInBucket += realEnergy;            
    }
    else
    {  
      // wait until the DC-blocking filters have had time to settle
      if(cycleCount > 100) // 100 mains cycles is 2 seconds
        beyondStartUpPhase = true;
    }      
      
    // clear the per-cycle accumulators for use in this new mains cycle.  
    sumP = 0;
    samplesDuringThisMainsCycle = 0;
    cumVdeltasThisCycle = 0;                  
  } // end of processing that is specific to the first +ve Vsample in each new cycle
  
  double sampleIminusDC = sampleI - DCoffset;
  double sampleVminusDC = sampleV - DCoffset;   
  double instP = sampleVminusDC * sampleIminusDC; //  power contribution for this pair of V&I samples 
  sumP +=instP;     // cumulative power values for this mains cycle 

  cumVdeltasThisCycle += (sampleV - DCoffset); // for use with LP filter
} // end of loop()


// helper function, to categorise LED events into:
//    LED PULSE (importing power);
//    LED ON (exporting power);
//    LED OFF (no longer exporting power)
// can be conveniently called every 20ms, at the start of each mains cycle
void checkLedStatus()
{
  ledState = digitalRead (ledDetectorPin);
  
  if (ledState != prevLedState)
  {
    // led has changed state
    if (ledState == ON)
    {
      // led has just gone on
      ledOnAt = millis();
      ledRecentlyOnFlag = true;
    }
    else
    {
      // led has just gone off   
      if (ledRecentlyOnFlag == true)
      {
        Serial.println ("** LED PULSE **");  // the meter has noted some energy consumption
        ledRecentlyOnFlag = false;
      }
      else
      {
        Serial.println ("** LED OFF **");   // the meter status has changed to "Not Exporting"     
      }      
    }    
  }
  else
  {
    // the LED state has not changed
    if (ledState == ON)
    {
      if (ledRecentlyOnFlag == true)
      { 
        // check to see if the known duration of a pulse has been exceeded    
        unsigned long timeNow = millis();      
        if ((timeNow - ledOnAt) > 50)
        {
          Serial.println ("** LED ON **");  // the meter status has changed to "Exporting"     
          ledRecentlyOnFlag = false;   
        }     
      }
    }
  } 
 
  if (ledState == ON) 
    digitalWrite(ledRepeaterPin, 1);
  else 
    digitalWrite(ledRepeaterPin, 0);  
    
  prevLedState = ledState;   
}


