
/* A meter simulation tool to support discussions on the Open Energy Monitor forum.
 *
 * Robin Emley (calypso_rae)
 * September 2012
 */

#define LED_OFF 0
#define LED_ON 1
#define LED_PULSE 2
#define MILLIS_PER_SECOND 1000
#define JOULES_PER_WATT_HOUR 3600

int level;
const int maxLevel = JOULES_PER_WATT_HOUR;
const int minLevel = -JOULES_PER_WATT_HOUR;
int levelChangePerLoop;

char blankLine[100];
char newLine[100];
int ledState = LED_OFF;
int Wh_register = 0;
unsigned long timeNow;
unsigned long timeOfLastPCE = 0;  
// A PCE is a "Potentially Chargeable Event" (my terminology).  I think there 
// are three types of PCE.  The first two types are chargeable, the third one is not.
//
// 1. In normal circumstances, pulses are seen at frequent intervals, each one 
// occurring after 0.001Wh of energy has been drawn from the grid.  Each pulse
// is a Chargeable Event for which the user will be billed for one thousandth of the 
// per-kWhour of consumption.  At each pulse, the register is incremented (by one) 
// and the level of the accumulator (energy bucket) is reduced by 3600J (i.e. 0.001kWh). 
//
// 2. When a significant amount of locally generated power flows back though the meter,
// the LED goes on to show "Reverse Energy Detected".  When import recommences, the 
// next Chargeable Event is when the LED goes off.  At this stage, the register is 
// again incremented (by one) and the level of the accumulator is reduced by 3600J. 
//
// 3. If minimal consumption is detected during a period of some minutes, digital meters 
// can go into an "anti-creep" mode.  With no firm information to go on, I've had 
// to guess at how this might work.  My feeling is that the criteria for this mode will
// be something simple.  
//   As a first attempt, I have simply arranged that the register will only be incremented
// if the amount of time that has elapsed since the previous Chargeable Event is less
// than a certain duration.  In practice, this timescale is of the order of ten minutes, but 
// this value can be usefully reduced for simulation purposes to, say, 60 seconds.
//    If more time that this has elapsed before the bucket is once more full, I regard the 
// meter as having entered "anti-creep" mode.  Although the level in the accumulator is 
// reduced by 1Whour's worth (3600J) as before, the register is not updated nor is a 
// pulse seen.  The time is, however, recorded so that interval to the next PCE can be 
// determined.  
// 


void setup()
{  
  Serial.begin(9600);
  Serial.println(); Serial.println(); Serial.println();
  
  Serial.println("Meter Simulation Tool"); Serial.println();
  Serial.println("Displays the behaviour of a virtual meter when different currents flow.");
  Serial.println("To change the current, just enter an integer value in the upper section"); 
  Serial.println("of the Serial Window then press Return or click 'Send' .  Positive and");
  Serial.println("negative values are allowed, as is the character'0'. For details of the" );
  Serial.println("virtual meter's operation, please consult the code.");
  Serial.println();
  Serial.println("Max recommended values are +/- 3000 watts");
  Serial.println();
  Serial.println("With the cursor in the 'Send' zone, press 'g' then [CR] to continue ..." );
  Serial.println();
  pause();
  
  displayTitleLine();
  level = 0;
  levelChangePerLoop = 0; // to be set by user from keyboard
}


void loop() 
{ 
  checkForUserInput();  
  level+= levelChangePerLoop;
  
  if (((millis()/1000) % 60) ==0) 
  {
    Serial.println("Simulation paused, press'g' then [CR] to continue ...");
    pause(); // useful to pause the display
  }  
    
  if (level >= maxLevel)
  {
    timeNow = millis();
    if ((timeNow - timeOfLastPCE) < 60 * MILLIS_PER_SECOND)
    {
      // a chargeable event has occurred  
      Wh_register ++;
      if (ledState == LED_ON)
        ledState = LED_OFF;
      else
        ledState = LED_PULSE;  
    }
    else  
    {
      // minimal rate of import, so no charge to user (anti-creep mode)
      ledState = LED_ON;
    }
    
    level -= JOULES_PER_WATT_HOUR;
    timeOfLastPCE = timeNow;
  }
  else
  if (level < 0)
  {
    ledState = LED_ON; // "Reverse Energy Detected"
  }
  
  if (level < minLevel)
  {
    level = minLevel; // energy loss has occurred
  }
  
  displayData();
  
  if (ledState == LED_PULSE)
  {
    ledState = LED_OFF;
  }

  delay(1000);
} 

void displayTitleLine()
{
  Serial.println   ("|<- -3600                     0                     +3600 ->|   ledState   WattHours"); 
  Serial.println   ("|                             .                             |   --------   ---------"); 
  strcpy(blankLine, "|                             .                             |      ");
  Serial.println(blankLine); 
}

void displayData()
{
  strcpy(newLine, blankLine);
  int index = map (level, -JOULES_PER_WATT_HOUR, JOULES_PER_WATT_HOUR, 0, 60);
  newLine[index] = '*'; 
  Serial.print(newLine);

  if(ledState == LED_OFF)   Serial.print("    off     ");
  else
  if(ledState == LED_ON)    Serial.print("   *ON*     ");
  else
  if(ledState == LED_PULSE) Serial.print("   PULSE    ");
  
  Serial.println(Wh_register);
}


void checkForUserInput()
{
  if (Serial.available() )
  {
    char inbuf[8] = {1,2,3,4,5,6,7,8}; // any characters will do 
    Serial.readBytesUntil('\n', inbuf, 8); 
    int value = atoi(inbuf);    
    if ((value == 0) && (inbuf[0] != '0'))
    {
      // invalid input
    }  
    else
    {
      levelChangePerLoop = value;
    }
  }
}

void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0)
    {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      if (dummyByte == 'g') 
        done++;
    }
  }    
}

