// Mk2i is a revised version of my stand-alone sketch for diverting suplus 
// PV power to a dump load using a triac.  The original Mk2 version, which includes 
// more explanation and 'debug' code to support off-line working, may be found at
// http://openenergymonitor.org/emon/node/841
// 
// General aspects of Mk2x:
// The Mk2x code is suitable for hardware that has multiple voltage references, such 
// as emonTx.  Integer maths is used for improved speed.  Low-level instructions are 
// also used for ADC operations which 'frees up' time that was previously unavailable 
// for general processing activities.  Interrupts, to control the operation of the ADC,
// were introduced for the Mk2i.  None of the Mk2a releases use interrupts.
//
// Specific releases of Mk2x: 
// The initial release of Mk2a had twin high-pass filters, as used in the standard OEM 
// V&I sketch.  Due to a couple of minor bugs, it was soon replaced by the _rev2 version. 
//
// The Mk2a_rev3 version marks a further change to the way that the raw sample streams are
// processed.  This has come about because, for the purpose of calculating "real power", 
// there is no need for DC offset to be pre-removed from the raw current samples. The 
// standard calculation for real power does this automatically when applied over each whole 
// cycle of the mains.  Half-cycle processing of power has been discontinued.
//   A single LPF, which is updated just once per mains cycle, was introduced for 
// the purpose of determining the DC offset of the voltage waveform.  This value is then
// subtracted from each raw voltage sample.  A nominal value of 512 is subtracted from 
// each current sample, this being to prevent the system becoming over-sensitive to any 
// imbalance conditions or random noise; the actual value is unimportant.
//
// The ANTI_FLICKER option was also introduced in Mk2a_rev3.  This mode uses two thresholds
// for the energy bucket rather than just one.  When surplus power is available,
// the power-diversion logic operates the triac between these two limits on a hysteresis 
// basis.  To ensure that the anti-flicker requirement is always met, a minimum amount of 
// time has to elapse between consecutive activations of the triac.  Under certain 
// conditions, this may cause a small amount of surplus power to be lost when the bucket 
// overflows.  The on-board LED (pin 13) shows whenever then this occurs. Accurate 
// calibration of the system is essential when operating in this mode.
//  
// TALLYMODE is a #define option which allows energy data to be collected for 
// subsequent display.  In Mk2a_rev2, energy values in positive and negative half-cycles was 
// recorded separately.  From Mk2a_rev3 onwards, energy is recorded for whole cycles only. 
// In Mk2i, the value of each tally (in Watts) is displayed rather than its index.
//   While running in TALLYMODE, the code is fully functional.  It pauses after a 
// user-selectable period so that stored data can be retrieved.  To avoid overflow, the 
// maximum recommended duration is 10 minutes.  To start again, press the 'g' key, then
// carriage-return.  [This works reliably on my Mac but not on my Windows laptop...]
//   TALLYMODE provides a very easy way to calibrate the system.  Clip the CT around
// one core of a cable that supplies a known load, and do a short run with appropriate 
// Min and Max values.  If the peak distribution (in Watts) is not where you expect it 
// to be, then just change the value of powerCal until it is.  
//   Full details about calibration are provided just before setup().
//
// SPEEDCHECK was a #define option that was introduced in Mk2a.  For Mk2i, it has been
// replaced by WORKLOAD_CHECK.  By use of the function delayMicroseconds(), this facility 
// introduces additional delay until all the available ADC conversion time has been 
// used up.  WORKLOAD_CHECK provides a quantitative assessment of the amount of spare 
// processing capacity that is available for doing oher things.  In normal operation, 
// this mode should be disabled. 
//
// Mk2i is interrupt based.  The ADC runs autonomously in either of two modes as described 
// in the code.  In both of these modes, the main processor waits in loop() until a dataReady
// flag is set; all of the available data is then processed in one go.  
//   In free-running mode, the system works faster than any previous variant of Mk2.  This
// would be a good choice for a system that just needs to divert surplus power as efficiently
// as possible.
//   In timer mode, the ADC is activated at fixed intervals via a hardware timer.  This mode
// would be a good choice for a system that has a significant amount of other processing to do.
//   For anyone who is unhappy about using interrupt-based code, Mk2a_rev3 would probably be
// be a better choice.
//
//                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
//                  January 2013


/* 
Circuit for monitoring pulses from a supply meter using checkLedStatus():
 
                  ----------------> +5V
                  |
                  /
                  \ 8K2
                  /
                  |
             ---------------------> dig 2
             |       |
        -->  /       |
        -->  \       _
        LDR  /       -  0.01uF
             |       |
             ----------------------> GND
*/

#include <Arduino.h> // it's probably a good idea to include this

// For Mk2i, two interrupt modes are supported, one of which must be active:
// -> With ADC_FREE enabled, the ADC is free-running with a cycle time of ~104uS.
// -> With ADC_TIMER active, the ADC is repeatedly triggered by a hardware timer.  Using
//      a slower timer reduces the base workload and makes more processing time available
//      for other purposes.
//
#define ADC_FREE   // <-- One of these lines must be included (but not both)
//#define ADC_TIMER  // <-- One of these lines must be included (but not both)

#ifdef ADC_TIMER
#include <TimerOne.h>
#define ADC_TIMER_PERIOD 125 // uS
  #ifdef ADC_FREE
  #error 'please select only one interrupt mode!
  #endif
#else
  #ifndef ADC_FREE
  #error 'please select an interrupt mode!
  #endif
#endif


// There are two "extra features", only one of which may be active at a time:
// -> TALLYMODE, for recording energy data, useful for calibration.
// -> WORKLOAD_CHECK, for determining how much spare processing time there is. 
//  
//#define TALLYMODE        // <-- Only one of these lines may be included 
#define WORKLOAD_CHECK  // <-- Only one of these lines may be included 

#ifdef TALLYMODE
  #ifdef WORKLOAD_CHECK
  #error 'please select only one of the extra features!
  #endif
#endif

// Power diversion can operate in either of two modes:
// -> NORMAL, where the triac switches rapidly on/off to maintain a constant energy level.
// -> ANTI_FLICKER, whereby the repetition rate is reduced to avoid rapid fluctuations
//    of the local mains voltage.
//
enum operatingModes {NORMAL, ANTI_FLICKER};
enum operatingModes operatingMode = NORMAL; // <-- the desired mode is selected here

#define CYCLES_PER_SECOND 50 
#define JOULES_PER_WATT_HOUR 3600 // 0.001 kWh = 3600 Joules
enum polarities {NEGATIVE, POSITIVE};
enum triacStates {TRIAC_ON, TRIAC_OFF}; // the external trigger device is active low
enum LEDstates {LED_ON, LED_OFF,};   // the LED detector is also active low

// general global variables
const byte outputPinForLed = 13;  // digital
const byte outputPinForTrigger = 9; // digital
const byte ledDetectorPin = 2;  // digital 
const byte ledRepeaterPin = 10;  // digital 
const byte voltageSensorPin = 2;  // analogue
const byte currentSensorPin = 1;   // analogue
const byte startUpPeriod = 5; // in seconds, to allow HP filters to settle
const int DCoffset_I = 512; // for calculating "real power", a nominal value is fine

long cycleCount = 0;
long cycleCountAtLastActivation = 0;
int samplesDuringThisCycle;
enum triacStates nextStateOfTriac;
enum triacStates triacState;
boolean triggerNeedsToBeArmed;
boolean beyondStartUpPhase = false;
float safetyMargin_WattsPerHalfCycle;
long triggerThreshold_long;
long energyInBucket_long = 0; 
int phaseCal_int; 
long capacityOfEnergyBucket_long;  
long energyThreshold_long;  
long antiFlicker_lowerEnergyThreshold;
long antiFlicker_upperEnergyThreshold;
long sumP;
long DCoffset_V_long = 512L * 256; // nominal mid-point value of ADC @ x256 scale
long DCoffset_V_min;
long DCoffset_V_max;
long cumVdeltasThisCycle_long;   // <<--- for LPF 

float minTimeBetweenActivations = 5; // <-- anti-flicker requirement (seconds)
int minCycleCountsBetweenActivations;

// values that need to be stored from one loop to the next
long lastSampleVminusDC_long;  //    for phaseCal algorithm
enum polarities polarityOfLastSampleV; // for zero-crossing detection
//int sampleV_forNextLoop; // for deferred processing 
//int sampleI_forNextLoop; // for deferred processing
int sampleI;
int sampleV;

// items for LED monitoring
byte ledState, prevLedState;
boolean ledRecentlyOnFlag = false;
unsigned long ledOnAt;
long energyInBucket_4led_long = 0; 

// for the interlock mechanism between the main processor and the ADC 
volatile boolean dataReady = false;

#ifdef TALLYMODE
#define NUMBER_OF_TALLIES 100
unsigned int tally[NUMBER_OF_TALLIES + 2]; // For recording the energy content in whole mains cycles.
unsigned int tallymode_maxCycleCount;  // the cycleCount value when recording should cease
int tallymode_maxVal; // the maximum power to be recorded (Watts)
int tallymode_minVal; // the minimum power to be recorded (Watts)
long tallymode_minVal_long; // x256 version for integer maths
float tallymode_stepVal; // the power increment between consecutive tallies (Watts)
long tallymode_stepVal_long; // x256 version for integer maths
boolean tallymode_firstLoop = true; 
unsigned int tallymode_noOfValuesTallied; // overflows after 10.9 minutes
unsigned long tallymode_noOfSamplePairs; // to show the average samples-per-mains-cycle rate
int tallymode_durationOfRecording; // in seconds
#endif

#ifdef WORKLOAD_CHECK
int del = 0; // delay, as passed to delayMicroseconds()
int res = 0; // result, to be displayed at the next opportunity
byte count = 0; // to allow multiple runs per setting
byte displayFlag = 0; // to determine when printing may occur
#endif

// Calibration values
//-------------------
// Three calibration values are required: powerCal, voltageCal and phaseCal. 
// With most hardware, the default values are likely to work fine without 
// need for change.  A full explanation of each of these values now follows:
//   
// powerCal is a floating point variable which is used for converting the 
// product of voltage and current samples into Watts.
//
// The correct value of powerCal is entirely dependent on the hardware that is 
// in use.  For best resolution, the hardware should be configured so that the 
// voltage and current waveforms each span most of the ADC's usable range.  For 
// many systems, the maximum power that will need to be measured is around 3kW. 
//
// My sketch "MinAndMaxValues.ino" provides a good starting point for 
// system setup.  First arrange for the CT to be clipped around either core of a  
// cable which supplies a suitable load; then run the tool.  The resulting values 
// should sit nicely within the range 0-1023.  To allow some room for safety, 
// a margin of around 100 levels should be left at either end.  This gives a 
// output range of around 800 ADC levels, which is 80% of its usable range.
//
// My sketch "RawSamplesTool.ino" provides a one-shot visual display of the
// voltage and current waveforms.  This provides an easy way for the user to be 
// confident that their system has been set up correctly for the power levels 
// that are to be measured.
//
// The ADC has an input range of 0-5V and an output range of 0-1023 levels.
// The purpose of each input sensor is to convert the measured parameter into a 
// low-voltage signal which fits nicely within the ADC's input range. 
//
// In the case of 240V mains voltage, the numerical value of the input signal 
// in Volts is likely to be fairly similar to the output signal in ADC levels.  
// 240V AC has a peak-to-peak amplitude of 679V, which is not far from the ideal 
// output range.  Stated more formally, the conversion rate of the overall system 
// for measuring VOLTAGE is likely to be around 1 ADC-step per Volt (RMS).
//
// In the case of AC current, however, the situation is very different.  At
// mains voltage, a power of 3kW corresponds to an RMS current of 12.5A which 
// has a peak-to-peak range of 35A.  This is smaller than the output signal by 
// around a factor of twenty.  The conversion rate of the overall system for 
// measuring CURRENT is therefore likely to be around 20 ADC-steps per Amp.
//
// When calculating power, which is what this code does, the individual 
// conversion rates for voltage and current are not of importance.  It is 
// only the conversion rate for POWER which is important.  This is the 
// product of the individual conversion rates for voltage and current.  It 
// therefore has the units of ADC-steps squared per Watt.  Most systems will
// have a power conversion rate of around 20 (ADC-steps squared per Watt).
// 
// powerCal is the RECIPR0CAL of the power conversion rate.  A good value 
// to start with is therefore 1/20 = 0.05 (Watts per ADC-step squared)
//
const float powerCal = 0.061;  // <---- powerCal value 
 
// As mentioned above, the conversion rate for AC voltage has units of  
// ADC-steps per Volt.  Athough not required for measuring power, this 
// conversion rate does need to be known in order to determine when the
// voltage level is suitable for arming the external trigger device.
//
// To determine the voltage conversion rate, note the Min and Max values that 
// are seen when measuring 240Vac via the voltage sensor.  Subtract one from 
// the other to find the range.  Then put that value into the voltageCal formula 
// below.  voltageCal has units of ADC-steps per Volt.
// 
// 679 is the peak-to-peak range of a 240V RMS signal.  The (float) typecast
// is to prevent integer rounding 
//
const float voltageCal = 800 / (float)679; // <-- the first number is the output range of 
                                           //     your ADC when 240V AC is being measured
                        
// phaseCal is used to alter the phase of the voltage waveform relative to the
// current waveform.  The algorithm interpolates between the most recent pair
// of voltage samples according to the value of phaseCal. 
//
//    With phaseCal = 1, the most recent sample is used.  
//    With phaseCal = 0, the previous sample is used
//    With phaseCal = 0.5, the mid-point (average) value in used
//
// Values ouside the 0 to 1 range involve extrapolation, rather than interpolation
// and are not recommended.  By altering the order in which V and I samples are 
// taken, and for how many loops they are stored, it should always be possible to
// arrange for the optimal value of phaseCal to lie within the range 0 to 1.  When 
// measuring a resistive load, the voltage and current waveforms should be perfectly 
// aligned.  In this situation, the Power Factor will be 1.
//
// My sketch "PhasecalChecker.ino" provides an easy way to determine the correct 
// value of phaseCal for any hardware configuration.  An index of my various sketches 
// is available at http://openenergymonitor.org/emon/node/1757
//
const float  phaseCal = 1.0;



void setup()
{  
  Serial.begin(9600);
  pinMode(outputPinForTrigger, OUTPUT);  
  pinMode(outputPinForLed, OUTPUT);  
  Serial.println();
  Serial.println();
  Serial.println();
  Serial.println("-------------------------------------");
  Serial.println("Sketch ID:      Mk2i_PV_Router_rev1.ino");
   
  // When using integer maths, calibration values that have supplied in floating point 
  // form need to be rescaled.  
  //
  phaseCal_int = phaseCal * 256; // for integer maths
  
  // When using integer maths, the SIZE of the ENERGY BUCKET is altered to match the
  // voltage conversion rate that is in use.  This avoids the need to re-scale every 
  // energy contribution, thus saving processing time.  To avoid integer rounding,
  // energy is also recorded at x50 scale.  
  //   This process is described in more detail in the function, allGeneralProcessing(), 
  // just before the bucket is updated at the start of each new whole-cycle of the mains.
  //  
  capacityOfEnergyBucket_long = (JOULES_PER_WATT_HOUR * 50L) * (1/powerCal);
/*  
  Serial.print("powerCal = ");
  Serial.print(powerCal,4);
  Serial.println(" Watts per ADCstep^2");
  Serial.print("energy bucket = 3600 * 50 * (1/");
  Serial.print(powerCal,4);
  Serial.print(") = ");
  Serial.print(capacityOfEnergyBucket_long);
  Serial.println(" energy measurement units");
//  Serial.println();  
*/                                                
  energyThreshold_long = capacityOfEnergyBucket_long * 0.5; // for normal operation 
  antiFlicker_lowerEnergyThreshold = capacityOfEnergyBucket_long * 0.2; // for anti-flicker mode 
  antiFlicker_upperEnergyThreshold = capacityOfEnergyBucket_long * 0.8; // for anti-flicker mode  
  minCycleCountsBetweenActivations = 
       minTimeBetweenActivations * 50; // cycleCount increments every 20mS

  if (operatingMode == NORMAL) {
    energyInBucket_long = 0.8 * energyThreshold_long; } // for faster start-up  
  
  triggerThreshold_long = 50 * 256L / voltageCal;
    // +50V is a suitable point in the rising waveform to arm the zero-crossing trigger.
    // The 256 is because filteredV is scaled at x256 when integer maths is used.
    // The reciprocal of voltageCal converts ADClevels into Volts, as described above.
    
  // define upper and lower limits to ensure correct startup of the LP filter which 
  // identifies DC offset in the stream of voltage samples
  DCoffset_V_min = (long)(512L - 100) * 256; // mid-point of ADC minus a working margin
  DCoffset_V_max = (long)(512L + 100) * 256; // mid-point of ADC plus a working margin

#ifdef ADC_FREE  
  Serial.println ("ADC mode:       free-running");
  
  // Set up the ADC to be free-running 
  ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2);  // Set the ADC's clock to system clock / 128
  ADCSRA |= (1 << ADEN);                 // Enable the ADC 
  
  ADCSRA |= (1<<ADATE);  // set the Auto Trigger Enable bit in the ADCSRA register.  Because 
                         // bits ADTS0-2 have not been set (i.e. they are all zero), the 
                         // ADC's trigger source is set to "free running mode".
                         
  ADCSRA |=(1<<ADIE);    // set the ADC interrupt enable bit. When this bit is written 
                         // to one and the I-bit in SREG is set, the 
                         // ADC Conversion Complete Interrupt is activated. 

  ADCSRA |= (1<<ADSC);   // start ADC manually first time 
  sei();                 // Enable Global Interrupts  
#endif 

#ifdef ADC_TIMER  
  Serial.print ("ADC mode:       ");
  Serial.print (ADC_TIMER_PERIOD);
  Serial.println ( "uS fixed timer");

  // Set up the ADC to be triggered by a hardware timer of fixed duration  
  ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2);  // Set the ADC's clock to system clock / 128
  ADCSRA |= (1 << ADEN);                 // Enable ADC

  Timer1.initialize(ADC_TIMER_PERIOD);   // set Timer1 interval
  Timer1.attachInterrupt( timerIsr );    // declare timerIsr() as interrupt service routine
#endif 

  Serial.print ( "Triac mode:     ");
  if (operatingMode == NORMAL) {
    Serial.println ( "normal"); }
  else {  
    Serial.println ( "anti-flicker"); }
    
  char flag = 0;
  Serial.print ( "Extra Features: ");  
#ifdef TALLYMODE  
  Serial.print ( "TALLYMODE ");
  flag++;
#endif    
#ifdef WORKLOAD_CHECK  
  Serial.print ( "WORKLOAD_CHECK ");
  flag++;
#endif
  if (flag == 0) {
    Serial.print ("none"); }
  Serial.println ();
        
  Serial.print ( "powerCal =      "); Serial.println (powerCal,4);
  Serial.print ( "phaseCal =      "); Serial.println (phaseCal);
  Serial.println ("----");    

#ifdef WORKLOAD_CHECK
   Serial.println ("WELCOME TO WORKLOAD_CHECK ");
   Serial.println ("  This mode of operation allows the spare processing capacity of the system");
   Serial.println ("to be analysed.  Additional delay is gradually increased until all spare time");
   Serial.println ("has been used up.  This value (in uS) is noted and the process is repeated.  ");
   Serial.println ("The delay setting is increased by 1uS at a time, and each value of delay is ");
   Serial.println ("checked several times before the delay is increased. "); 
   Serial.println ("  The displayed value is the amount of spare time, per pair of V & I samples, ");
   Serial.println ("that is available for doing additional processing.  For normal operation, the ");
   Serial.println ("#define WORKLOAD_CHECK statement should be commented out.");
   Serial.println ();
 #endif
}

// For each mode of operation, an appropriate Interrupt Service Routine is now defined.
// In both cases, the ADC is instructed to measure V and I alternately.  A "data ready"
// flag is set after each voltage conversion has been completed.  
//   For each pair of samples, this means that current is measured before voltage.  The 
// current sample is taken first because the phase of the waveform for current is generally 
// slightly advanced relative to the waveform for voltage.  The data ready flag is cleared 
// within loop().

#ifdef ADC_TIMER
// This is the Interrupt Service Routine for when the ADC is fixed timer mode.  It is 
// executed whenever the ADC timer expires.  In this mode, the next ADC conversion is 
// initiated from within this ISR.  
//
void timerIsr(void)
{                                         
  static unsigned char sample_index = 0;
   
  if( sample_index==0 )
  {
    sampleV = ADC;
    ADMUX = 0x40 + currentSensorPin;  // after a Voltage conversion, the next one is for Current
    ADCSRA |= (1<<ADSC);              // start the ADC
    sample_index = 1;                 // toggle the control flag
    dataReady = true; 
  }
  else
  {
    sampleI = ADC;
    ADMUX = 0x40 + voltageSensorPin;  // after a Current conversion, the next one is for Voltage
    ADCSRA |= (1<<ADSC);              // start the ADC
    sample_index = 0;                 // toggle the control flag
   }
}
#endif

#ifdef ADC_FREE
// This is the Interrupt Service Routine for when the ADC is in the free-running mode.
// It is executed whenever the ADC conversion has finished, approx every 104 us.  In this 
// mode, the ADC restarts itself automatically.  
// 
ISR(ADC_vect)  
{                                         
  static unsigned char sample_index = 0;
   
  if( sample_index==0 )
  {
    sampleI = ADC; 
    ADMUX = 0x40 + currentSensorPin; // current, because the next V conversion is already under way
    sample_index = 1;                // toggle the control flag
  }
  else
  {
    sampleV = ADC; 
    ADMUX = 0x40 + voltageSensorPin; // voltage, because the next I conversion is already under way
    sample_index = 0;                // toggle the control flag
    dataReady = true; 
  }
}
#endif


// When using interrupt-based logic, the main processor waits in loop() until the 
// dataReady flag has been set by the ADC.  Once this flag has been set, the main
// processor clears the flag and proceeds with all the processing for one pair of 
// V & I samples.  It then returns to loop() to wait for the next pair to become 
// available.
//   If the next pair of samples become available before the processing of the 
// previous pair has been completed, data could be lost.  This situation can be 
// avoided by prior use of the WORKLOAD_CHECK mode.  Using this facility, the amount
// of spare processing capacity per loop can be determined.  
//   If there is insufficient processing capacity to do all that is required, the 
// base workload can be reduced by increasing the duration of ADC_TIMER_PERIOD (but 
// only when in ACD_TIMER mode).
//
void loop()             
{ 
  if (dataReady)   // flag is set after every pair of ADC conversions
  {
    dataReady = false; // reset the flag
    allGeneralProcessing(); // executed once for each pair of V&I samples
    
#ifdef WORKLOAD_CHECK 
    delayMicroseconds(del); // <--- to assess how much spare time there is
    if (dataReady)       // if data is ready again, delay was too long
    { 
      res = del;             // note the exact value
      del = 1;               // and start again with 1us delay   
      count = 0;
      displayFlag = 0;   
    }
    else
    {
      count++;          // to give several runs with the same value
      if (count > 50)
      {
        count = 0;
        del++;                //  increase delay by 1uS
      } 
    }
#endif  

  }  // <-- this closing brace needs to be outside the WORKLOAD_CHECK blocks! 
  
#ifdef WORKLOAD_CHECK 
  switch (displayFlag) 
  {
    case 0: // the result is available now, but don't display until the next loop
      displayFlag++;
      break;
    case 1: // with minimum delay, it's OK display the result now
      Serial.print(res);
      Serial.println("uS");
      displayFlag++;
      break;
    default:; // for most of the time, displayFlag is 2           
  }
#endif
  
} // end of loop()


// This routine is called to process each pair of V & I samples.  Note that when using 
// interrupt-based code, it is not necessary to delay the processing of each pair of 
// samples as was done in Mk2a builds.  This is because there is no longer a strict 
// alignment between the obtaining of each sample by the ADC and the processing that can 
// be done by the main processor while the ADC conversion is in progress.  
//   When interrupts are used, the main processor and the ADC work autonomously, their
// operation being only linked via the dataReady flag.  As soon as data is made available
// by the ADC, the main processor can start to work on it immediately.  The system's 
// response time will therefore have improved by around 250uS.
//
void allGeneralProcessing()
{
#ifdef TALLYMODE
  tallymode_checks();
#endif

  // remove DC offset from the raw voltage sample by subtracting the accurate value 
  // as determined by a LP filter.
  long sampleVminusDC_long = ((long)sampleV<<8) - DCoffset_V_long; 

  // determine polarity, to aid the logical flow
  enum polarities polarityNow;   
  if(sampleVminusDC_long > 0) { 
    polarityNow = POSITIVE; }
  else { 
    polarityNow = NEGATIVE; }

  if (polarityNow == POSITIVE) 
  {                           
    if (polarityOfLastSampleV != POSITIVE)
    {
      // This is the start of a new +ve half cycle (just after the zero-crossing point)
      cycleCount++;  
      triggerNeedsToBeArmed = true; // the trigger is armed once during each +ve half-cycle 
    
      // Calculate the real power and energy during the last whole mains cycle.
      //
      // sumP contains the sum of many individual calculations of instantaneous power.  In  
      // order to obtain the average power during the relevant period, sumP must first be 
      // divided by the number of samples that have contributed to its value.
      //
      // The next stage would normally be to apply a calibration factor so that real power 
      // can be expressed in Watts.  That's fine for floating point maths, but it's not such
      // a good idea when integer maths is being used.  To keep the numbers large, and also 
      // to save time, calibration of power is omitted at this stage.  realPower_long is 
      // therefore (1/powerCal) times larger than the actual power in Watts.
      //
      long realPower_long = sumP / samplesDuringThisCycle; // proportional to Watts
   
      // Next, the energy content of this power rating needs to be determined.  Energy is 
      // power multiplied by time, so the next step is normally to multiply by the time over 
      // which the power was measured.
      //   Instanstaneous power is calculated once every mains cycle, so that's every fiftieth 
      // of a second.  When integer maths is being used, this routine power-to-energy conversion 
      // seems an unnecessary workload.  As all sampling periods are of similar duration (20mS), 
      // it is more efficient simply to add all the power samples together, and note that their 
      // sum is actually 50 times greater than it would otherwise be.
      //   Although the numerical value itself does not change, a new name is helpful so as  
      // to avoid any confusion.  The 'energy' variable below is 50 * (1/powerCal) times larger 
      // than the actual energy in Joules.
      //
      long realEnergy_long = realPower_long; 
    
      // Energy contributions are summed in an accumulator which is generally known as the 
      // energy bucket.  The purpose of the energy bucket is to mimic the operation of the
      // supply meter.  Most supply meters have a range of 0.001kWh within which energy can 
      // pass to and fro without loss or charge to the user.  The energy bucket in the Mk2 
      // Power Router was therefore set to 0.001kWh, or 3600 Joules.  By use of floating point 
      // maths, its contents were correctly scaled to be in Joules.
      //
      // When using integer maths, as described above, energy contributions are scaled somewhat 
      // differently.  The capacity of the energy bucket for Mk2x builds thereore needs to 
      // be 3600J * 50 * (1/powerCal).  This is the value that appears in setup().   
    
      if (beyondStartUpPhase)
      {  
        // Providing that the initial settling time has passed, add this latest contribution
        // to the energy bucket
        energyInBucket_long += realEnergy_long;   
        energyInBucket_4led_long += realEnergy_long;   
         
        // Apply max and min limits to bucket's level.  This is to ensure correct operation
        // when conditions change, i.e. when import changes to export, and vici versa
        //
        if (energyInBucket_long > capacityOfEnergyBucket_long) 
        {
          energyInBucket_long = capacityOfEnergyBucket_long; 
          digitalWrite(outputPinForLed, 1); // illuminate the on-board LED if bucket overflows
        } 
        else 
        {
          digitalWrite(outputPinForLed, 0); // clear the on-board LED if bucket is not overflowing
          if (energyInBucket_long < 0) 
          {
            energyInBucket_long = 0; 
          }  
        }
  
#ifdef TALLYMODE
        tallymode_updateData(realPower_long); // update the relevant tally 
#endif
      }
      else
      {  
        // check whether the system has had time to settle
        if(cycleCount > (startUpPeriod * CYCLES_PER_SECOND))
        {
          beyondStartUpPhase = true;
        }
      }   
      
      // clear the per-cycle accumulators for use in this new mains cycle.  
      samplesDuringThisCycle = 0;
      sumP = 0;

    } // end of processing that is specific to the first Vsample in each +ve half cycle 
  
    // still processing samples where the voltage is POSITIVE ...
    if (triggerNeedsToBeArmed == true)
    {
      // check to see whether the trigger device can now be reliably armed
      if(sampleVminusDC_long > triggerThreshold_long)
      {
        // the are two separate schemes for diverting power, each with its own rules:
        if (operatingMode == NORMAL)
        {
          // In NORMAL mode, it is only necessary to check whether the energy threshold 
          // has been reached, and then set the triac accordingly 
          if (energyInBucket_long > energyThreshold_long) 
          {
            nextStateOfTriac = TRIAC_ON;  
          } 
          else
          {
            nextStateOfTriac = TRIAC_OFF; 
          }
        } 
        else
        {
          // In anti-flicker mode, the control logic is more complex:
          if (energyInBucket_long < antiFlicker_lowerEnergyThreshold)       
          {
            // when below the lower threshold, always turn the triac off 
            nextStateOfTriac = TRIAC_OFF;  
          }
          else
          if (energyInBucket_long > antiFlicker_upperEnergyThreshold) // upper threshold      
          {
            // when above the upper threshold, ensure that the triac is on if permitted,
            // otherwise it has to remain off
            if (triacState == TRIAC_OFF)
            {
              if (cycleCount > cycleCountAtLastActivation + minCycleCountsBetweenActivations) 
              {
                nextStateOfTriac = TRIAC_ON;  
                cycleCountAtLastActivation = cycleCount;
              }
            }
          }
          else
          {
            // the energy level is between the upper and lower thresholds, so
            // the triac's state remains unchanged (hysteresis)
          }          
        } // end of anti-flcker mode logic
                  
        // set the Arduino's output pin accordingly, and clear the flag
        digitalWrite(outputPinForTrigger, nextStateOfTriac);   
        triggerNeedsToBeArmed = false;      
      }
    }
  } // end of processing that is specific to samples where the voltage is positive
  
  else // the polatity of this sample is negative
  {     
    if (polarityOfLastSampleV != NEGATIVE)
    {
      // This is the start of a new -ve half cycle (just after the zero-crossing point)
      //
      // The triac can change state at each -ve going zero crossing.  Update the state of a
      // flag which shows the current state of the triac (for anti-flicker mode logic)
      //
      if (nextStateOfTriac == TRIAC_ON) {
        triacState = TRIAC_ON; }
      else {  
        triacState = TRIAC_OFF; }        
      
      // This is a convenient point to update the Low Pass Filter for DC-offset removal
      long previousOffset = DCoffset_V_long;
      DCoffset_V_long = previousOffset + (0.01 * cumVdeltasThisCycle_long); 
      cumVdeltasThisCycle_long = 0;
      
      // To ensure that the LPF will always start up correctly when 240V AC is available, its
      // output value needs to be prevented from drifting beyond the likely range of the 
      // voltage signal.  This avoids the need to use a HPF as was done for initial Mk2 builds.
      //
      if (DCoffset_V_long < DCoffset_V_min) {
        DCoffset_V_long = DCoffset_V_min; }
      else  
      if (DCoffset_V_long > DCoffset_V_max) {
        DCoffset_V_long = DCoffset_V_max; }
           
    } // end of processing that is specific to the first Vsample in each -ve half cycle
  } // end of processing that is specific to samples where the voltage is positive
  
  // processing for EVERY pair of samples
  //
  // remove most of the DC offset from the current sample (the precise value does not matter)
  long sampleIminusDC_long = ((long)(sampleI-DCoffset_I))<<8;
  
  // phase-shift the voltage waveform so that it aligns with the current when a 
  // resistive load is used
  long  phaseShiftedSampleVminusDC_long = lastSampleVminusDC_long
         + (((sampleVminusDC_long - lastSampleVminusDC_long)*phaseCal_int)>>8);  
  
  // calculate the "real power" in this sample pair and add to the accumulated sum
  long filtV_div4 = phaseShiftedSampleVminusDC_long>>2;  // reduce to 16-bits (now x64, or 2^6)
  long filtI_div4 = sampleIminusDC_long>>2; // reduce to 16-bits (now x64, or 2^6)
  long instP = filtV_div4 * filtI_div4;  // 32-bits (now x4096, or 2^12)
  instP = instP>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
  sumP +=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
  samplesDuringThisCycle++;
  
  // store items for use during next loop
  cumVdeltasThisCycle_long += sampleVminusDC_long; // for use with LP filter
  lastSampleVminusDC_long = sampleVminusDC_long;  // required for phaseCal algorithm
  polarityOfLastSampleV = polarityNow;  // for identification of half cycle boundaries
}
//  ----- end of main Mk2i code -----


#ifdef TALLYMODE
void tallymode_checks()
{
  if (tallymode_firstLoop) { 
    tallymode_setup(); } // user-dialogue for recording energy data
  else
  if (cycleCount > tallymode_maxCycleCount) {
    tallymode_dispatchData(); // send recorded energy data to the Serial monitor  
    cycleCount = 0;
    tallymode_firstLoop = true;  // get ready for another run
    pause(); } // so that user can access data from the Serial monitor
  else
  if (beyondStartUpPhase) {
    tallymode_noOfSamplePairs++; }
}


void  tallymode_setup()
{
  char inbuf[10];
  int tempInt;
  byte noOfBytes;
  boolean done;

  Serial.println ("WELCOME TO TALLYMODE ");
  Serial.println ("This mode of operation allows the energy content of individual mains cycles");
  Serial.println ("to be analysed.  For nomal operation, the #define TALLYMODE statement");
  Serial.println ("should be commented out. ");
  Serial.println ();  
  Serial.println ("Tallymode setup:");  
  Serial.print ("Time to run (seconds)? ");
  done = false;
  while (!done) {
    noOfBytes = Serial.available();
    if (noOfBytes > 0) { done = true; } else { delay(100); }}
  for (tempInt = 0; tempInt < 10; tempInt++) { inbuf[tempInt] = 0; }
  Serial.readBytes(inbuf, noOfBytes); 
  tempInt = atoi(inbuf);  Serial.println (tempInt);
  tallymode_maxCycleCount = (tempInt + startUpPeriod) * CYCLES_PER_SECOND;
  tallymode_durationOfRecording = tempInt;
  Serial.print(" tallymode_maxCycleCount = "); Serial.print(tallymode_maxCycleCount);     
  Serial.print(", recording to start at cycleCount ");
  Serial.println(startUpPeriod * CYCLES_PER_SECOND);  
  
  Serial.print ("Min value to be recorded (Watts)? ");
  done = false;
  while (!done) {
    noOfBytes = Serial.available();
    if (noOfBytes > 0) { done = true; } else { delay(100); }}
  for (tempInt = 0; tempInt < 10; tempInt++) { inbuf[tempInt] = 0; }
  Serial.readBytes(inbuf, noOfBytes); 
  tempInt = atoi(inbuf);  Serial.println (tempInt);
  tallymode_minVal = tempInt;
  Serial.print(" tallymode_minVal = "); Serial.println(tallymode_minVal);   
  tallymode_minVal_long = tallymode_minVal * (1/powerCal); // used for power, not energy,
                                                           // hence there's no x100 term.
  
  Serial.print ("Max value to be recorded (Watts)? ");
  done = false;
  while (!done) {
    noOfBytes = Serial.available();
    if (noOfBytes > 0) { done = true; } else { delay(100); }}
  for (tempInt = 0; tempInt < 10; tempInt++) { inbuf[tempInt] = 0; }
  Serial.readBytes(inbuf, noOfBytes); 
  tempInt = atoi(inbuf);  Serial.println (tempInt);
  tallymode_maxVal = tempInt;
  Serial.print(" tallymode_maxVal = "); Serial.println(tallymode_maxVal);   

  if (tallymode_maxVal <= tallymode_minVal) {
    Serial.print(7); // beep
    Serial.println ("!!!!!!!!PROBLEM!!!!!!!!!!!"); }
  
  tallymode_stepVal = (float)(tallymode_maxVal - tallymode_minVal) / NUMBER_OF_TALLIES;
  Serial.print(" tallymode_stepVal = "); Serial.println(tallymode_stepVal);   
  tallymode_stepVal_long = tallymode_stepVal * (1/powerCal); // used for power, not energy,
                                                             // hence there's no x100 term.

  for (tempInt = 0; tempInt < NUMBER_OF_TALLIES + 2; tempInt++) {
    tally[tempInt] = 0; }
    
  energyInBucket_long = energyThreshold_long -100; // for quick startup of power distribution
  tallymode_noOfValuesTallied = 0;
  tallymode_noOfSamplePairs = 1;
  beyondStartUpPhase = false;
  tallymode_firstLoop = false; 

  Serial.print(" Data recording will start in ");
  Serial.print(startUpPeriod);   
  Serial.println(" seconds ... ");
};


void  tallymode_updateData(long power_long)
{
  int index = (power_long - tallymode_minVal_long) / tallymode_stepVal_long;
  if (index < 0) {
      index = 0; } // tally[0] is for underflow
  else 
  if (index > NUMBER_OF_TALLIES) {
    index = NUMBER_OF_TALLIES + 1; } // tally[N+1] is for overflow

  tally[index]++;      
  tallymode_noOfValuesTallied++; 
}


void  tallymode_dispatchData()
{
  Serial.println ();
  Serial.println ("Format of results: ");
  Serial.print ("Sorted data runs from tally[1] (min) to tally[");
  Serial.print (NUMBER_OF_TALLIES);
  Serial.println ("] (max).");
  Serial.print ("tally[0] is below range; tally[ ");
  Serial.print (NUMBER_OF_TALLIES + 1);
  Serial.println ("] is above range.");
  Serial.println("Tally results are displayed with max power first. For each one:");
  Serial.println("  mid-range power of tally (Watts), number of times recorded.");
  Serial.println();
  Serial.println("Results for this run:");
  Serial.print (tallymode_minVal);
  Serial.println (", <- min value of tally range (W)");
  Serial.print (tallymode_maxVal);
  Serial.println (", <- max value of tally range (W)");
  Serial.print (tallymode_stepVal);
  Serial.println (", <- step value between tallies (W)");
  Serial.print (NUMBER_OF_TALLIES);
  Serial.println (", <- number of tallies");
  Serial.print (tallymode_durationOfRecording);
  Serial.println (", <- duration of recording (sec)");
  Serial.print (tallymode_noOfValuesTallied);
  Serial.println (", <- no of values tallied");
  Serial.print ( tallymode_noOfSamplePairs / 
         (tallymode_durationOfRecording * CYCLES_PER_SECOND));
  Serial.println (", <- samples per mains cycle (average)");
  Serial.println("***");
 
  float powerVal;
  for (int index = NUMBER_OF_TALLIES + 1; index >= 0; index--)
  {
/*  
    // display the index value for this tally
    Serial.print (index);
    Serial.print (", ");
*/    
    // display the power value for this tally
    if (index == NUMBER_OF_TALLIES + 1)
    {
      Serial.print (">"); 
      Serial.print (tallymode_maxVal);
      Serial.print ("W"); 
    }
    else
    if (index == 0) 
    {
      Serial.print ("<"); 
      Serial.print (tallymode_minVal);
      Serial.print ("W"); 
    }
    else  
    {
      if (index == NUMBER_OF_TALLIES) 
      {
        powerVal = tallymode_maxVal - (tallymode_stepVal / 2); 
      }
      else 
      {
        powerVal -= tallymode_stepVal; 
      }
      
      if ((int)powerVal == powerVal)
      {
        Serial.print (powerVal, 0); // to suppress the decimal part for integers 
      }
      else
      {
        Serial.print (powerVal);  // non-integers are displayed to 2 dec places
      }
    }  
    
    // display the nimber of hits for this tally
    Serial.print (", ");
    Serial.println (tally[index]);  
  }
};


void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0) {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      if (dummyByte == 'g') done++; }
  }    
}
#endif


// helper function, to process LED events, can be conveniently called once per mains cycle
void checkLedStatus()
{
  ledState = digitalRead (ledDetectorPin);

  if (ledState != prevLedState)
  {
    // led has changed state
    if (ledState == LED_ON)
    {
      // led has just gone on
      ledOnAt = millis();
      ledRecentlyOnFlag = true;
    }
    else
    {
      // led has just gone off   
      if (ledRecentlyOnFlag == true)
      {
        ledRecentlyOnFlag = false;        
        Serial.print ("** LED PULSE ** "); // this is a chargeable event  
      }
      else
      {
        Serial.print ("** LED OFF ** "); // 'no longer exporting' is also a chargeable event
      }
      Serial.println(millis()/1000);
    }    
  }
  else
  {
    // the LED state has not changed
    if (ledState == LED_ON)
    {
      if (ledRecentlyOnFlag == true)
      { 
        // check to see if the known duration of a pulse has been exceeded    
        unsigned long timeNow = millis();      
        if ((timeNow - ledOnAt) > 50)
        {
          Serial.print ("** LED ON **");  // 'exporting' is a non-chargeable state     
          Serial.print (",  energy in bucket = "); 
          Serial.println((long)(energyInBucket_4led_long)); 
          ledRecentlyOnFlag = false;   
        }     
      }
    }
  } 
  prevLedState = ledState;   
}





