// Mk2i is a revised version of my stand-alone sketch for diverting suplus 
// PV power to a dump load using a triac.  The original Mk2 version, which includes 
// more explanation and 'debug' code to support off-line working, may be found at
// http://openenergymonitor.org/emon/node/841
// 
// General aspects of Mk2x:
// The Mk2x code is suitable for hardware that has multiple voltage references, such 
// as emonTx.  Integer maths is used for improved speed.  Low-level instructions are 
// used for ADC operations which 'frees up' time that was previously unavailable for
// general processing activities.  Interrupts, to control the operation of the ADC,
// were introduced for the Mk2i.  None of the Mk2a releases use interrupts.
//
// Specific releases of Mk2x: 
// The initial release of Mk2a had twin high-pass filters, as used in the standard OEM 
// V&I sketch.  Due to a couple of minor bugs, it was soon replaced by the _rev2 version. 
//
// The Mk2a_rev3 version marks a further change to the way that the raw sample streams are
// processed.  This has come about because, for the purpose of calculating "real power", 
// there is no need for DC offset to be pre-removed from the raw current samples. The 
// standard calculation for real power does this automatically when applied over each whole 
// cycle of the mains.  Half-cycle processing of power has been discontinued.
//   A single LPF, which is updated just once per mains cycle, was re-introduced for 
// the purpose of determining the DC offset of the voltage waveform.  This value is then
// subtracted from each raw voltage sample.  A nominal value of 512 is subtracted from 
// each current sample, this being to prevent the system becoming over-sensitive to any 
// imbalance conditions or random noise; the actual value is unimportant.
//
// An ANTI_FLICKER option was also introduced in Mk2a_rev3.  This mode used two thresholds
// for the energy bucket rather than just one.  Accurate calibration of the system was 
// essential when operating in this mode.  A single-threshold alternative algorthm has 
// since been adopted (in Mk2i_rev5).
//  
// TALLYMODE is a #define option which allows energy data to be collected for 
// subsequent display.  For Mk2a_rev2, energy values in positive and negative half-cycles was 
// recorded separately.  From Mk2a_rev3 onwards, energy is recorded for whole cycles only. 
// In Mk2i, the value of each tally (in Watts) is displayed rather than its index.
//   While running in TALLYMODE, the code is fully functional.  It pauses after a 
// user-selectable period so that stored data can be retrieved.  To avoid overflow, the 
// maximum recommended duration is 10 minutes.  To start again, press the 'g' key, then
// carriage-return.  [This works reliably on my Mac but not on my Windows laptop...]
//   TALLYMODE provides a very easy way to calibrate the system.  Clip the CT around
// one core of a cable that supplies a known load, and do a short run with appropriate 
// Min and Max values.  If the peak distribution (in Watts) is not where you expect it 
// to be, then just change the value of powerCal until it is.  
//   Full details about calibration are provided just before setup().
//
// SPEEDCHECK was a #define option that was introduced in Mk2a.  For Mk2i, it has been
// replaced by WORKLOAD_CHECK.  By use of the function delayMicroseconds(), this facility 
// introduces additional delay until all the available ADC conversion time has been 
// used up.  WORKLOAD_CHECK provides a quantitative assessment of the amount of spare 
// processing capacity that is available for doing oher things.  In normal operation, 
// this mode should be disabled. 
//
// Mk2i is interrupt based.  The ADC runs autonomously in either of two modes as described 
// in the code.  In both of these modes, the main processor waits in loop() until a dataReady
// flag is set; all of the available data is then processed in one go.  
//   In free-running mode, the system works faster than any previous variant of Mk2.  This
// would be a good choice for a system that just needs to divert surplus power as efficiently
// as possible.
//   In timer mode, the ADC is activated at fixed intervals via a hardware timer.  This mode
// would be a good choice for a system that has a significant amount of other processing to do.
//   For anyone who is unhappy about using interrupt-based code, Mk2a_rev3 would probably be
// a better choice.
//
// Mk2i_rev2 has the addition of 3 LEDs which are useful when running in anti-flicker mode.
// These show the presence of:
//   - any surplusPV, on a per mains cycle basis (this is also the on-board LED, at pin 13)
//   - any overflow of the energy bucket (likely to result in loss of surplus PV)
//   - any underflow of the energy bucket (likely to result in consumption charge)
//   
// Mk2i_rev3 combines the processing of normal and anti-flicker modes. 'Anti-flicker' measures 
// now are included as standard.  'Normal' mode is now just a special case in which the 
// A/F parameters are set to their "do nothing" values.
//   Mk2i_rev3 also allows the output mode mode, ie NORMAL or ANTI_FLICKER, to be selected at run-time. 
// To ensure compatibility with previous builds, this facility is disabled by default.  Run-time
// selection of the output mode can be enabled by setting a new flag to 'true'.  In this case,
// the sketch senses the user's selection by routinely checking the state of digital pin 4.  
// This pin is held high by means of an internal pullup resistor, and may be pulled low at any 
// time by means of a simple on/off switch between digital pin 4 and ground.  
//
// Mk2i_rev4.  Although the new aspects of Mk2i_rev3 appeared to behave as intended, that version
// failed to run in its calibration mode (Tallymode) due to lack of RAM.  This problem has been
// fixed in_rev4 by the wholesale removal of Serial statements.  
//   A routine has been added from http://jeelabs.org/2011/05/22/atmega-memory-use/  which
// determines the amount of spare RAM.  By calling this routine, the amount of spare RAM is now 
// displayed at startup and whenever the output mode switch is changed.
//   A long-standing half-step rounding error in Tallymode has been fixed.
//
// (work in progress ...)
//
// Mk2i_rev5.  Introduction of RF capability for remote load.  Also supports a switch so
// that the remote load can be given primary or secondary status. Uses a different type of
// energy routing algorithm which has a single energy threshold and requires no calibration.
// To avoid conflict with connections for the RFM12B module, the pins for all LEDs have had 
// been changed.
// >>> WARNING: This version will not run without an RF module!
// 
// The associated sketch for the receiving end is:   RF_for_Mk2_rx.ino
//
//                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
//                  March 2013


/* 
Circuit for monitoring pulses from a supply meter using checkLedStatus():
 
                  ----------------> +5V
                  |
                  /
                  \ 8K2
                  /
                  |
             ---------------------> dig 2
             |       |
        -->  /       |
        -->  \       _
        LDR  /       -  0.01uF
             |       |
             ----------------------> GND
*/

#include <Arduino.h> // may not be needed, but it's probably a good idea to include this
#include <JeeLib.h>     // JeeLib is available at from: http://github.com/jcw/jeelib

// ----------------------- ADC mode selection ----------------------
//
// For Mk2i, two interrupt modes are supported, one of which must be active:
// -> With ADC_FREE enabled, the ADC is free-running with a cycle time of ~104uS.
// -> With ADC_TIMER active, the ADC is repeatedly triggered by a hardware timer.  Using
//      a slower timer reduces the base workload and makes more processing time available
//      for other purposes.
//
//#define ADC_FREE   // <-- One of these lines must be included (but not both)
#define ADC_TIMER  // <-- One of these lines must be included (but not both)

#ifdef ADC_TIMER
#include <TimerOne.h>
#define ADC_TIMER_PERIOD 125 // uS
  #ifdef ADC_FREE
  #error 'please select only one interrupt mode!
  #endif
#else
  #ifndef ADC_FREE
  #error 'please select an interrupt mode!
  #endif
#endif

// ---------------- Tallymode & Workload Check selection ----------------------
//
// There are two "extra features", only one of which may be active at a time:
// - TALLYMODE, for recording energy data, useful for calibration.
// - WORKLOAD_CHECK, for determining how much spare processing time there is. 
//  
//#define TALLYMODE        // <-- Only one of these lines may be included 
//#define WORKLOAD_CHECK  // <-- Only one of these lines may be included 

#ifdef TALLYMODE
  #ifdef WORKLOAD_CHECK
  #error 'please select only one of the extra features!
  #endif
#endif

// ----------------- other #defines ---------------------
//
#define CYCLES_PER_SECOND 50 
#define JOULES_PER_WATT_HOUR 3600 // 0.001 kWh = 3600 Joules
#define REFRESH_PERIOD 1000 // max period (ms) between RF messages


// ----------------- RF setup for remote load ---------------------
//
// WARNING - if an RF module is not available, then the following flag
//           must be set to false, otherwise the code will not run!
//
boolean RF_moduleIsAvailable = true; // <<<<<< 

// frequency options are RF12_433MHZ, RF12_868MHZ or RF12_915MHZ
//
#define freq RF12_868MHZ // Use the freq to match the module you have.

const int nodeID = 10;  // emonTx RFM12B node ID
const int networkGroup = 210;  // emonTx RFM12B wireless network group - needs to be same as emonBase and emonGLCD 
const int UNO = 1;  // Set to 0 if you're not using the UNO bootloader (i.e using Duemilanove) 
                                               // - All Atmega's shipped from OpenEnergyMonitor come with Arduino Uno bootloader
const int noOfCyclesBeforeRefresh = 50;
long cycleCountAtLastRFtransmission = 0;
int msgNumber = 0;

typedef struct { byte dumpState; int msgNumber; } Tx_struct;    //  data for RF comms
Tx_struct tx_data;

boolean sendRFcommandNextTime = false;

// ---------  definition & use of enumerated types ----------

const byte noOfDumploads = 2; // The logic expects a minimum of 2 dumploads, 
                              // for local & remote loads, but neither has to
                              // be physically present. 
                              //   If no RF module is present, the code will 
                              // work with one local load only.  I've not tried  
                              // with more than one physical local load - there 
                              // are no spare pins on the processor!
                              
enum polarities {NEGATIVE, POSITIVE};

enum loadStates {LOAD_ON, LOAD_OFF}; // all loads are active low
enum loadStates logicalLoadState[noOfDumploads]; 
enum loadStates physicalLoadState[noOfDumploads]; 

enum LEDstates {LED_ON, LED_OFF,};   // the LED detector is also active low
enum outputModes {ANTI_FLICKER, NORMAL};
enum loadPriorityModes {REMOTE_HAS_PRIORITY, LOCAL_HAS_PRIORITY};

enum energyStates {LOWER_HALF, UPPER_HALF};
enum energyStates energyStateNow;


// ---------  Power-diversion and priority selection ----------
//
// The power-diversion logic can operate in either of two modes:
//
// - NORMAL, where the triac switches rapidly on/off to maintain a constant energy level.
// - ANTI_FLICKER, whereby the repetition rate is reduced to avoid rapid fluctuations
//    of the local mains voltage.
//
// Similarly, the power-routing priority logic can operate in either of two modes:
//
// - LOCAL_HAS_PRIORITY, where the local load has priority over the remote one;
// - REMOTE_HAS_PRIORITY, where the remote load has priority over the local one.
//
// Each of the above selections can be made in either of two ways:
//
// - at compile-time, in which case the flag which follows should be set to "false",
//     with the chosen modes being set on the lines immediately below; or,
// - at run-time, in which case the flag which follows should be set to "true".  
//     Values for the output and priority modes are then set according to external 
//     switches, and the values supplied here are ignored.
//
boolean runTimeSelectionIsEnabled = false;                                           
enum outputModes outputMode = ANTI_FLICKER;                                                    
enum loadPriorityModes loadPriorityMode = LOCAL_HAS_PRIORITY;                                                   

// ----------- Pinout assignments  -----------
//
// digital input pins:
// dig pin 0 is for Serial Rx
// dig pin 1 is for Serial Tx
// dig pin 2 is for the RFM12B module (IRQ) 
const byte LED_detectorpin = 3; // for meter pulse detection (polling or interrupts)
const byte outputModeSelectorPin = 4;   //  for normal or anti-flicker mode
const byte loadPrioritySelectorPin = 5; //  local or remote load to have priority
const byte underflow_LED = 6;           // non-essential, may have to be axed.       
const byte overflow_LED = 7;       
const byte surplusPV_LED = 8;   
const byte physicalLoad_0_pin = 9;      // for trigger (active low)
// dig pin 10 is for the RFM12B module (SEL) 
// dig pin 11 is for the RFM12B module (SDI) 
// dig pin 12 is for the RFM12B module (SDO) 
// dig pin 13 is for the RFM12B module (CLK) 

// analogue input pins:
const byte voltageSensorPin = 2;      // analogue
const byte currentSensorPin = 1;      // analogue


// --------------  general global variables -----------------
//
// Some of these variables are used in multiple blocks so cannot be static.
// For integer maths, many variables need to be 'long'
//
boolean beyondStartUpPhase = false;    // start-up delay, allows things to settle
unsigned long startTime = 0;       // is non-zero when in Tallymode 

const byte startUpPeriod = 3;      // in seconds, to allow LP filter to settle
const int DCoffset_I = 512;        // nominal mid-point value of ADC @ x1 scale

/* NB.  In all previous versions of the Mk2 code, cycleCount counted right from the very start. 
 *      Now, it only starts to operate after the start-up period has expired.
 */
long cycleCount = 0;               // counts mains cycles after the start-up period has expired.
long triggerThreshold_long;        // for determining when the trigger may be safely armed
long capacityOfEnergyBucket_long;  // an energy accumulator for use with integer maths 
long energyInBucket_long = 0;      // to record the present level in the energy accumulator
long energyInBucket_4led_long = 0; // a 'no limits version for use with checkLedStatus()
long energyThreshold_long;         // used for 'normal' and single-threshold 'AF' logic
int phaseCal_int;                  // to avoid the need for floating-point maths
long DCoffset_V_long;              // <--- for LPF
long DCoffset_V_min;               // <--- for LPF
long DCoffset_V_max;               // <--- for LPF


// anti-flicker parameters (NORMAL is now just a special case of ANTI-FLICKER mode)
//
int postMidPointCrossingDelay_cycles; // assigned in setup(), different for each output mode
const int postMidPointCrossingDelayForAF_cycles = 25; // in 20 ms counts 
const int interLoadSeparationDelay_cycles = 25; // in 20 ms cycle counts (for both output modes)
byte activeLoadID; // only one load may operate freely at a time.
long cycleCountAtLastMidPointCrossing = 0;
long cycleCountAtLastTransition = 0;
long energyAtLastOffTransition_long; 
long energyAtLastOnTransition_long; 

// for interaction between the main processor and the ISRs 
volatile boolean dataReady = false;
int sampleI;
int sampleV;

// Tallymode is a comprehensive facility for calibration & recording of energy data
// It uses several functions, so most of these values cannot easily be made 'static'
#ifdef TALLYMODE
#define NUMBER_OF_TALLIES 100
unsigned int tally[NUMBER_OF_TALLIES + 2]; // For recording the energy content in whole mains cycles.
unsigned int tallymode_maxCycleCount;  // the cycleCount value when recording should cease
int tallymode_maxVal; // the maximum power to be recorded (Watts)
int tallymode_minVal; // the minimum power to be recorded (Watts)
long tallymode_minVal_long; // x256 version for integer maths
float tallymode_stepVal; // the power increment between consecutive tallies (Watts)
long tallymode_stepVal_long; // x256 version for integer maths
boolean tallymode_firstLoop = true; // to support subsequent runs without needing to restart
unsigned int tallymode_noOfValuesTallied; // overflows after 10.9 minutes
unsigned long tallymode_noOfSamplePairs; // to show the average samples-per-mains-cycle rate
int tallymode_durationOfRecording; // in seconds
long tallymode_halfStep;
#endif


// Calibration values
//-------------------
// Three calibration values are used: powerCal, voltageCal and phaseCal. 
// With most hardware, the default values are likely to work fine without 
// need for change.  A full explanation of each of these values now follows:
//   
// powerCal is a floating point variable which is used for converting the 
// product of voltage and current samples into Watts.
//
// The correct value of powerCal is entirely dependent on the hardware that is 
// in use.  For best resolution, the hardware should be configured so that the 
// voltage and current waveforms each span most of the ADC's usable range.  For 
// many systems, the maximum power that will need to be measured is around 3kW. 
//
// My sketch "MinAndMaxValues.ino" provides a good starting point for 
// system setup.  First arrange for the CT to be clipped around either core of a  
// cable which supplies a suitable load; then run the tool.  The resulting values 
// should sit nicely within the range 0-1023.  To allow some room for safety, 
// a margin of around 100 levels should be left at either end.  This gives a 
// output range of around 800 ADC levels, which is 80% of its usable range.
//
// My sketch "RawSamplesTool.ino" provides a one-shot visual display of the
// voltage and current waveforms.  This provides an easy way for the user to be 
// confident that their system has been set up correctly for the power levels 
// that are to be measured.
//
// The ADC has an input range of 0-5V and an output range of 0-1023 levels.
// The purpose of each input sensor is to convert the measured parameter into a 
// low-voltage signal which fits nicely within the ADC's input range. 
//
// In the case of 240V mains voltage, the numerical value of the input signal 
// in Volts is likely to be fairly similar to the output signal in ADC levels.  
// 240V AC has a peak-to-peak amplitude of 679V, which is not far from the ideal 
// output range.  Stated more formally, the conversion rate of the overall system 
// for measuring VOLTAGE is likely to be around 1 ADC-step per Volt (RMS).
//
// In the case of AC current, however, the situation is very different.  At
// mains voltage, a power of 3kW corresponds to an RMS current of 12.5A which 
// has a peak-to-peak range of 35A.  This is smaller than the output signal by 
// around a factor of twenty.  The conversion rate of the overall system for 
// measuring CURRENT is therefore likely to be around 20 ADC-steps per Amp.
//
// When calculating power, which is what this code does, the individual 
// conversion rates for voltage and current are not of importance.  It is 
// only the conversion rate for POWER which is important.  This is the 
// product of the individual conversion rates for voltage and current.  It 
// therefore has the units of ADC-steps squared per Watt.  Most systems will
// have a power conversion rate of around 20 (ADC-steps squared per Watt).
// 
// powerCal is the RECIPR0CAL of the power conversion rate.  A good value 
// to start with is therefore 1/20 = 0.05 (Watts per ADC-step squared)
//
const float powerCal = 0.061;  // <---- powerCal value 
 
// As mentioned above, the conversion rate for AC voltage has units of  
// ADC-steps per Volt.  Athough not required for measuring power, this 
// conversion rate does need to be known in order to determine when the
// voltage level is suitable for arming the external trigger device.
//
// To determine the voltage conversion rate, note the Min and Max values that 
// are seen when measuring 240Vac via the voltage sensor.  Subtract one from 
// the other to find the range.  Then put that value into the voltageCal formula 
// below.  voltageCal has units of ADC-steps per Volt.
// 
// 679 is the peak-to-peak range of a 240V RMS signal.  The (float) typecast
// is to prevent integer rounding 
//
const float voltageCal = 800 / (float)679; // <-- the first number is the output range of 
                                           //     your ADC when 240V AC is being measured
                        
// phaseCal is used to alter the phase of the voltage waveform relative to the
// current waveform.  The algorithm interpolates between the most recent pair
// of voltage samples according to the value of phaseCal. 
//
//    With phaseCal = 1, the most recent sample is used.  
//    With phaseCal = 0, the previous sample is used
//    With phaseCal = 0.5, the mid-point (average) value in used
//
// Values ouside the 0 to 1 range involve extrapolation, rather than interpolation
// and are not recommended.  By altering the order in which V and I samples are 
// taken, and for how many loops they are stored, it should always be possible to
// arrange for the optimal value of phaseCal to lie within the range 0 to 1.  When 
// measuring a resistive load, the voltage and current waveforms should be perfectly 
// aligned.  In this situation, the Power Factor will be 1.
//
// My sketch "PhasecalChecker.ino" provides an easy way to determine the correct 
// value of phaseCal for any hardware configuration.  An index of my various Mk2-related
// exhibits is available at http://openenergymonitor.org/emon/node/1757
//
const float  phaseCal = 1.0;


void setup()
{  
  pinMode(underflow_LED, OUTPUT);  
  pinMode(overflow_LED, OUTPUT);  
  pinMode(surplusPV_LED, OUTPUT);
  
  pinMode(physicalLoad_0_pin, OUTPUT);  
  for(int i = 0; i< noOfDumploads; i++)
  {
    logicalLoadState[i] = LOAD_OFF;
    physicalLoadState[i] = LOAD_OFF;
  } 
  
  if (runTimeSelectionIsEnabled)
  {
    int pinState;
    pinMode(outputModeSelectorPin, INPUT);
    digitalWrite(outputModeSelectorPin, HIGH); // enable the internal pullup resistor
    delay (100); // allow time to settle
    pinState = digitalRead(outputModeSelectorPin);    // initial selection and
    outputMode = (enum outputModes)pinState;          //   assignment of output mode

    pinMode(loadPrioritySelectorPin, INPUT);
    digitalWrite(loadPrioritySelectorPin, HIGH); // enable the internal pullup resistor
    delay (100); // allow time to settle
    pinState = digitalRead(loadPrioritySelectorPin);       // initial selection and
    loadPriorityMode = (enum loadPriorityModes)pinState;   //   assignment of output mode
  }
  
  if (RF_moduleIsAvailable)
  {
    rf12_initialize(nodeID, freq, networkGroup);             // initialize RF
    rf12_sleep(RF12_SLEEP);
  }
  
  Serial.begin(9600);                                      // initialize Serial interface
  Serial.println();
  Serial.println();
  Serial.println();
  Serial.println("-------------------------------------");
  Serial.println("Sketch ID:      Mk2i_PV_Router_rev5.ino");
       
  // When using integer maths, calibration values that have supplied in floating point 
  // form need to be rescaled.  
  //
  phaseCal_int = phaseCal * 256; // for integer maths
  
  // When using integer maths, the SIZE of the ENERGY BUCKET is altered to match the
  // voltage conversion rate that is in use.  This avoids the need to re-scale every 
  // energy contribution, thus saving processing time.  To avoid integer rounding,
  // energy is also recorded at x50 scale.  
  //   This process is described in more detail in the function, allGeneralProcessing(), 
  // just before the bucket is updated at the start of each new whole-cycle of the mains.
  //  
  capacityOfEnergyBucket_long = (JOULES_PER_WATT_HOUR * 50L) * (1/powerCal);
  
  // for tidy start-up of energy routing logic (single threshold version)
  energyAtLastOffTransition_long = capacityOfEnergyBucket_long; 
  energyAtLastOnTransition_long = 0; 

  triggerThreshold_long = 50 * 256L / voltageCal;
    // +50V is a suitable point in the rising waveform to arm the zero-crossing trigger.
    // The 256 is because filteredV is scaled at x256 when integer maths is used.
    // The reciprocal of voltageCal converts ADClevels into Volts, as described above.
    
  // Define operating limits for the LP filter which identifies DC offset in the voltage 
  // sample stream.  By limiting the output range, the filter always should start up 
  // correctly.
  DCoffset_V_long = 512L * 256; // nominal mid-point value of ADC @ x256 scale  
  DCoffset_V_min = (long)(512L - 100) * 256; // mid-point of ADC minus a working margin
  DCoffset_V_max = (long)(512L + 100) * 256; // mid-point of ADC plus a working margin

#ifdef ADC_FREE  
  Serial.println ("ADC mode:       free-running");
  
  // Set up the ADC to be free-running 
  ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2);  // Set the ADC's clock to system clock / 128
  ADCSRA |= (1 << ADEN);                 // Enable the ADC 
  
  ADCSRA |= (1<<ADATE);  // set the Auto Trigger Enable bit in the ADCSRA register.  Because 
                         // bits ADTS0-2 have not been set (i.e. they are all zero), the 
                         // ADC's trigger source is set to "free running mode".
                         
  ADCSRA |=(1<<ADIE);    // set the ADC interrupt enable bit. When this bit is written 
                         // to one and the I-bit in SREG is set, the 
                         // ADC Conversion Complete Interrupt is activated. 

  ADCSRA |= (1<<ADSC);   // start ADC manually first time 
  sei();                 // Enable Global Interrupts  
#endif 

#ifdef ADC_TIMER  
  Serial.print ("ADC mode:       ");
  Serial.print (ADC_TIMER_PERIOD);
  Serial.println ( "uS fixed timer");

  // Set up the ADC to be triggered by a hardware timer of fixed duration  
  ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2);  // Set the ADC's clock to system clock / 128
  ADCSRA |= (1 << ADEN);                 // Enable ADC

  Timer1.initialize(ADC_TIMER_PERIOD);   // set Timer1 interval
  Timer1.attachInterrupt( timerIsr );    // declare timerIsr() as interrupt service routine
#endif 

  Serial.print ( "Output mode:    ");
  if (outputMode == NORMAL) {
    Serial.println ( "normal"); }
  else 
  {  
    Serial.println ( "anti-flicker");
  }

  Serial.print ( "Priority load:  ");
  if (loadPriorityMode == LOCAL_HAS_PRIORITY) {
    Serial.println ( "local"); }
  else 
  {  
    Serial.println ( "remote");
  }
    
  Serial.print ( "Auto-select?:   ");
  if(runTimeSelectionIsEnabled) {
    Serial.println ( "yes"); }
  else {  
    Serial.println ( "no"); }
      
  char flag = 0;
  Serial.print ( "Extra Features: ");  
#ifdef TALLYMODE  
  Serial.print ( "TALLYMODE ");
  flag++;
#endif    
#ifdef WORKLOAD_CHECK  
  Serial.print ( "WORKLOAD_CHECK ");
  flag++;
#endif
  if (flag == 0) {
    Serial.print ("none"); }
  Serial.println ();
        
  Serial.print ( "powerCal =      "); Serial.println (powerCal,4);
  Serial.print ( "phaseCal =      "); Serial.println (phaseCal);
  
  configureParamsForSelectedOutputMode(); 
  Serial.println ("----");    

#ifdef WORKLOAD_CHECK
   Serial.println ("WELCOME TO WORKLOAD_CHECK ");
  
//   <<- start of commented out section, to save on RAM space!
/*   
   Serial.println ("  This mode of operation allows the spare processing capacity of the system");
   Serial.println ("to be analysed.  Additional delay is gradually increased until all spare time");
   Serial.println ("has been used up.  This value (in uS) is noted and the process is repeated.  ");
   Serial.println ("The delay setting is increased by 1uS at a time, and each value of delay is ");
   Serial.println ("checked several times before the delay is increased. "); 
 */ 
//  <<- end of commented out section, to save on RAM space!

   Serial.println ("  The displayed value is the amount of spare time, per pair of V & I samples, ");
   Serial.println ("that is available for doing additional processing.");
   Serial.println ();
 #endif
}

// For each mode of operation, an appropriate Interrupt Service Routine is now defined.
// In both cases, the ADC is instructed to measure V and I alternately.  A "data ready"
// flag is set after each voltage conversion has been completed.  
//   For each pair of samples, this means that current is measured before voltage.  The 
// current sample is taken first because the phase of the waveform for current is generally 
// slightly advanced relative to the waveform for voltage.  The data ready flag is cleared 
// within loop().

#ifdef ADC_TIMER
// This is the Interrupt Service Routine for when the ADC is fixed timer mode.  It is 
// executed whenever the ADC timer expires.  In this mode, the next ADC conversion is 
// initiated from within this ISR.  
//
void timerIsr(void)
{                                         
  static unsigned char sample_index = 0;
   
  if( sample_index==0 )
  {
    sampleV = ADC;
    ADMUX = 0x40 + currentSensorPin;  // after a Voltage conversion, the next one is for Current
    ADCSRA |= (1<<ADSC);              // start the ADC
    sample_index = 1;                 // toggle the control flag
    dataReady = true; 
  }
  else
  {
    sampleI = ADC;
    ADMUX = 0x40 + voltageSensorPin;  // after a Current conversion, the next one is for Voltage
    ADCSRA |= (1<<ADSC);              // start the ADC
    sample_index = 0;                 // toggle the control flag
   }
}
#endif

#ifdef ADC_FREE
// This is the Interrupt Service Routine for when the ADC is in the free-running mode.
// It is executed whenever the ADC conversion has finished, approx every 104 us.  In this 
// mode, the ADC restarts itself automatically.  
// 
ISR(ADC_vect)  
{                                         
  static unsigned char sample_index = 0;
   
  if( sample_index==0 )
  {
    sampleI = ADC; 
    ADMUX = 0x40 + currentSensorPin; // current, because the next V conversion is already under way
    sample_index = 1;                // toggle the control flag
  }
  else
  {
    sampleV = ADC; 
    ADMUX = 0x40 + voltageSensorPin; // voltage, because the next I conversion is already under way
    sample_index = 0;                // toggle the control flag
    dataReady = true; 
  }
}
#endif


// When using interrupt-based logic, the main processor waits in loop() until the 
// dataReady flag has been set by the ADC.  Once this flag has been set, the main
// processor clears the flag and proceeds with all the processing for one pair of 
// V & I samples.  It then returns to loop() to wait for the next pair to become 
// available.
//   If the next pair of samples become available before the processing of the 
// previous pair has been completed, data could be lost.  This situation can be 
// avoided by prior use of the WORKLOAD_CHECK mode.  Using this facility, the amount
// of spare processing capacity per loop can be determined.  
//   If there is insufficient processing capacity to do all that is required, the 
// base workload can be reduced by increasing the duration of ADC_TIMER_PERIOD (but 
// only when in ACD_TIMER mode).
//
void loop()             
{ 
#ifdef WORKLOAD_CHECK
  static int del = 0; // delay, as passed to delayMicroseconds()
  static int res = 0; // result, to be displayed at the next opportunity
  static byte count = 0; // to allow multiple runs per setting
  static byte displayFlag = 0; // to determine when printing may occur
#endif
  
  if (dataReady)   // flag is set after every pair of ADC conversions
  {
    dataReady = false; // reset the flag
    allGeneralProcessing(); // executed once for each pair of V&I samples
    
#ifdef WORKLOAD_CHECK 
    delayMicroseconds(del); // <--- to assess how much spare time there is
    if (dataReady)       // if data is ready again, delay was too long
    { 
      res = del;             // note the exact value
      del = 1;               // and start again with 1us delay   
      count = 0;
      displayFlag = 0;   
    }
    else
    {
      count++;          // to give several runs with the same value
      if (count > 50)
      {
        count = 0;
        del++;          //  increase delay by 1uS
      } 
    }
#endif  

  }  // <-- this closing brace needs to be outside the WORKLOAD_CHECK blocks! 
  
#ifdef WORKLOAD_CHECK 
  switch (displayFlag) 
  {
    case 0: // the result is available now, but don't display until the next loop
      displayFlag++;
      break;
    case 1: // with minimum delay, it's OK to print now
      Serial.print(res);
      displayFlag++;
      break;
    case 2: // with minimum delay, it's OK to print now
      Serial.println("uS");
      displayFlag++;
      break;
    default:; // for most of the time, displayFlag is 3           
  }
#endif
  
} // end of loop()


// This routine is called to process each pair of V & I samples.  Note that when using 
// interrupt-based code, it is not necessary to delay the processing of each pair of 
// samples as was done in Mk2a builds.  This is because there is no longer a strict 
// alignment between the obtaining of each sample by the ADC and the processing that can 
// be done by the main processor while the ADC conversion is in progress.  
//   When interrupts are used, the main processor and the ADC work autonomously, their
// operation being only linked via the dataReady flag.  As soon as data is made available
// by the ADC, the main processor can start to work on it immediately.  
//
void allGeneralProcessing()
{
  static boolean triggerNeedsToBeArmed = false;  // once per mains cycle (+ve half)
  static int samplesDuringThisCycle;             // for normalising the power in each mains cycle
  static long sumP;                              // for per-cycle summation of 'real power' 
  static enum polarities polarityOfLastSampleV;  // for zero-crossing detection
  static long cumVdeltasThisCycle_long;    // for the LPF which determines DC offset (voltage)
  static long lastSampleVminusDC_long;     //    for the phaseCal algorithm

#ifdef TALLYMODE
  tallymode_checks();
#endif

  // remove DC offset from the raw voltage sample by subtracting the accurate value 
  // as determined by a LP filter.
  long sampleVminusDC_long = ((long)sampleV<<8) - DCoffset_V_long; 

  // determine polarity, to aid the logical flow
  enum polarities polarityNow;   
  if(sampleVminusDC_long > 0) { 
    polarityNow = POSITIVE; }
  else { 
    polarityNow = NEGATIVE; }

  if (polarityNow == POSITIVE) 
  {                           
    if (beyondStartUpPhase)
    {  
      if (polarityOfLastSampleV != POSITIVE)
      {
        // This is the start of a new +ve half cycle (just after the zero-crossing point)
        cycleCount++;  
        triggerNeedsToBeArmed = true; // the trigger is armed once during each +ve half-cycle 
    
        // Calculate the real power and energy during the last whole mains cycle.
        //
        // sumP contains the sum of many individual calculations of instantaneous power.  In  
        // order to obtain the average power during the relevant period, sumP must first be 
        // divided by the number of samples that have contributed to its value.
        //
        // The next stage would normally be to apply a calibration factor so that real power 
        // can be expressed in Watts.  That's fine for floating point maths, but it's not such
        // a good idea when integer maths is being used.  To keep the numbers large, and also 
        // to save time, calibration of power is omitted at this stage.  realPower_long is 
        // therefore (1/powerCal) times larger than the actual power in Watts.
        //
        long realPower_long = sumP / samplesDuringThisCycle; // proportional to Watts
   
        // Next, the energy content of this power rating needs to be determined.  Energy is 
        // power multiplied by time, so the next step is normally to multiply by the time over 
        // which the power was measured.
        //   Instanstaneous power is calculated once every mains cycle, so that's every fiftieth 
        // of a second.  When integer maths is being used, this routine power-to-energy conversion 
        // seems an unnecessary workload.  As all sampling periods are of similar duration (20mS), 
        // it is more efficient simply to add all the power samples together, and note that their 
        // sum is actually 50 times greater than it would otherwise be.
        //   Although the numerical value itself does not change, a new name is helpful so as  
        // to avoid any confusion.  The 'energy' variable below is 50 * (1/powerCal) times larger 
        // than the actual energy in Joules.
        //
        long realEnergy_long = realPower_long; 
    
        // Energy contributions are summed in an accumulator which is generally known as the 
        // energy bucket.  The purpose of the energy bucket is to mimic the operation of the
        // supply meter.  Most supply meters have a range of 1 Wh within which energy can 
        // pass to and fro without loss or charge to the user.  The energy bucket in the Mk2 
        // Power Router was therefore set to 1 Wh, or 3600 Joules.  By use of floating point 
        // maths, its contents were correctly scaled to be in Joules.
        //
        // When using integer maths, as described above, energy contributions are scaled somewhat 
        // differently.  The capacity of the energy bucket for Mk2x builds thereore needs to 
        // be 3600J * 50 * (1/powerCal).  This is the value that appears in setup().   
    
        // Providing that the initial settling time has passed, add this latest contribution
        // to the energy bucket
        energyInBucket_long += realEnergy_long;   // the version for diverting power has limits 
        energyInBucket_4led_long += realEnergy_long; // ... but this one does not
         
        // Apply max and min limits to bucket's level.  This is to ensure correct operation
        // when conditions change, i.e. when import changes to export, and vici versa.
        // WARNING - While diverting surplus power, Overflow and Underflow are 
        //           not desirable conditions, hence the LEDs
        //
        if (energyInBucket_long > capacityOfEnergyBucket_long) 
        { 
          // overflow is likely to result in the loss of surplus energy to the grid 
          energyInBucket_long = capacityOfEnergyBucket_long; 
          digitalWrite(overflow_LED, 1);  //   illuminate the overflow LED ...
          digitalWrite(underflow_LED, 0); //   ... and ensure the underflow LED is off
        } 
        else         
        if (energyInBucket_long < 0) 
        {
          // underflow is likely to result in charge for consumption from the grid 
          energyInBucket_long = 0; 
          digitalWrite(underflow_LED, 1); //   illuminate the underflow LED ...
          digitalWrite(overflow_LED, 0);  //   ... and ensure the underflow LED is off
        }  
        else
        {
          digitalWrite(underflow_LED, 0); //   ensure that ...
          digitalWrite(overflow_LED, 0);  //      ... both LEDs are off
        }
        
        if (realEnergy_long > 0) {             // Update the LED which shows whether there is
          digitalWrite(surplusPV_LED, 1); }    // any surplus power.  This is particularly useful 
        else {                                 // in anti-flicker mode when 'on' periods
          digitalWrite(surplusPV_LED, 0); }    // of the triac can be quite infrequent
  
#ifdef TALLYMODE
        tallymode_updateData(realPower_long); // update the relevant tally 
#endif
    
/*
        if ((cycleCount % 50) == 7) // a convenient place to display something occasionally
        {
          Serial.println(energyInBucket_long);  
        }
*/ 

        // update the mid-point crossing variable for the single-threshold algorithm
        enum energyStates energyStateOnLastLoop = energyStateNow;

        if (energyInBucket_long > energyThreshold_long) {
          energyStateNow = UPPER_HALF; }
        else {
          energyStateNow = LOWER_HALF; }
        
        if (energyStateNow != energyStateOnLastLoop) {
          cycleCountAtLastMidPointCrossing = cycleCount; }


        // clear the per-cycle accumulators for use in this new mains cycle.  
        samplesDuringThisCycle = 0;
        sumP = 0;

      } // end of processing that is specific to the first Vsample in each +ve half cycle 
  
      // still processing samples where the voltage is POSITIVE ...
      if (triggerNeedsToBeArmed == true)
      {     
        // check to see whether the trigger device can now be reliably armed
        if(sampleVminusDC_long > triggerThreshold_long)
        {
          boolean OKtoSendRFcommandNow = false; 
          boolean changeOfLoadState = false;
        
          // a pending RF command takes priority over the normal logic
          if (sendRFcommandNextTime)
          {
            OKtoSendRFcommandNow = true;
            sendRFcommandNextTime = false;
          } 
          else
          {   
            /* Now it's time to determine whether any of the the loads need to be changed.  
             * This is a 2-stage process:
             *   First, change the LOGICAL loads as necessary, then update the PHYSICAL
             * loads according to the mapping that exists between them.  The mapping is 
             * 1:1 by default but can be altered by a hardware switch which allows the 
             * priority of the remote load to be altered.
             *   This code uses a single-threshold algorithm which relies on regular switching 
             * of the load.  The same code is used for NORMAL and ANTI_FLICKER modes, the
             * only difference being these modes being amount of time that can elapse after 
             * the level in the energy bucket has crossed the mid-point.
             */
            if (cycleCount > cycleCountAtLastMidPointCrossing + postMidPointCrossingDelay_cycles)
            {           
              if (energyInBucket_long > energyThreshold_long) 
              {
                increaseLoadIfPossible();  // to reduce the level in the bucket
              }  
              else
              {
                decreaseLoadIfPossible();  // to increase the level in the bucket       
              }
            }
          
          
            /* Update the state of all physical loads and determine whether the  
             * state of the remote load has changed. The remote load is hard-coded 
             * as Physical Load 1 (Physical Load 0 is a local load)
             */
            boolean remoteLoadHasChangedState = false;
            byte prevStateOfRemoteLoad = (byte)physicalLoadState[1];
            updatePhysicalLoadStates();
            if ((byte)physicalLoadState[1] != prevStateOfRemoteLoad)
            {
              remoteLoadHasChangedState = true;
            }
                
            /* Now determine whether an RF command should be sent.  This can be for 
             * either of two reasons:
             *
             * - the on/off state of the remote load has changed
             * - a refresh command is due ('cos no change of state has occurred recently)
             *
             * If the on/off state has changed, but a refresh command was sent on the
             * previous cycle, the 'change of state' command is deferred until the next 
             * cycle.  A refresh command can always be sent straight away because it can 
             * be guaranteed that no command will have been sent immediately beforehand.
             */
           
            // determine how long it's been since the last RF command was sent 
            long cycleCountSinceLastRFtransmission = cycleCount - cycleCountAtLastRFtransmission;
                     
            if (remoteLoadHasChangedState)
            {
              // ensure that RF commands are not sent on consecutive cycles
              if (cycleCountSinceLastRFtransmission > 1)
              {
                // the "change of state" can be acted on immediately
                OKtoSendRFcommandNow = true; // local flag
              }
              else
              {
                // the "change of state" must be deferred until next cycle
                sendRFcommandNextTime = true; // global flag
              }
            }
            else
            {
              // no "change of state", so check whether a refresh command is due
              if (cycleCountSinceLastRFtransmission >= noOfCyclesBeforeRefresh)
              {
                // a refresh command is due (which can always be sent immediately)
                OKtoSendRFcommandNow = true;
              }
            }
          }
     
          if (RF_moduleIsAvailable)
          {   
            // deal with the remote load, which is physical load 1 
            if (OKtoSendRFcommandNow)
            {
              cycleCountAtLastRFtransmission = cycleCount;
              tx_data.msgNumber = msgNumber++;
              tx_data.dumpState = physicalLoadState[1];
              send_rf_data();
   //         
              Serial.print(tx_data.dumpState);    
              Serial.print(", ");    
              Serial.print(tx_data.msgNumber); 
            
              Serial.print(";   ");          
              Serial.println(activeLoadID); // useful for keeping track of different priority loads
   //         
            }       
          }
          else
          { 
            if ((cycleCount % 25) == 0)
            {
              Serial.println(activeLoadID); // useful for keeping track of different priority loads
            }
          }
          
          
          // update all other physical loads
          digitalWrite(physicalLoad_0_pin, physicalLoadState[0]);       
        
          // clear the flag which ensures that loads are only updated once per mains cycle
          triggerNeedsToBeArmed = false;           
        }
      }
    }
    else
    {  
      // wait until the DC-blocking filters have had time to settle
      if(millis() > startTime + (startUpPeriod * 1000)) 
      {
        beyondStartUpPhase = true;
        sumP = 0;
        samplesDuringThisCycle = 0;
        Serial.println ("Go!");
      }
    }
  } // end of processing that is specific to samples where the voltage is positive
  
  else // the polarity of this sample is negative
  {     
    if (polarityOfLastSampleV != NEGATIVE)
    {
      // This is the start of a new -ve half cycle (just after the zero-crossing point)
      //
      // This is a convenient point to update the Low Pass Filter for DC-offset removal,
      // which needs to be done right from the start.
      long previousOffset = DCoffset_V_long;
      DCoffset_V_long = previousOffset + (cumVdeltasThisCycle_long>>6); // faster than * 0.01
      cumVdeltasThisCycle_long = 0;
      
      // To ensure that the LPF will always start up correctly when 240V AC is available, its
      // output value needs to be prevented from drifting beyond the likely range of the 
      // voltage signal.  This avoids the need to use a HPF as was done for initial Mk2 builds.
      //
      if (DCoffset_V_long < DCoffset_V_min) {
        DCoffset_V_long = DCoffset_V_min; }
      else  
      if (DCoffset_V_long > DCoffset_V_max) {
        DCoffset_V_long = DCoffset_V_max; }
        
      if (runTimeSelectionIsEnabled)
      {
        checkOutputModeSelection(); // updates outputMode if switch is changed
        checkLoadPrioritySelection(); // updates load priorities if switch is changed
      }
      
    } // end of processing that is specific to the first Vsample in each -ve half cycle
  } // end of processing that is specific to samples where the voltage is positive
  
  // Processing for EVERY pair of samples. Most of this code is not used during the 
  // start-up period, but it does no harm to leave it in place.  Accumulated values 
  // are cleared when beyondStartUpPhase is set to true.
  //
  // remove most of the DC offset from the current sample (the precise value does not matter)
  long sampleIminusDC_long = ((long)(sampleI-DCoffset_I))<<8;
  
  // phase-shift the voltage waveform so that it aligns with the current when a 
  // resistive load is used
  long  phaseShiftedSampleVminusDC_long = lastSampleVminusDC_long
         + (((sampleVminusDC_long - lastSampleVminusDC_long)*phaseCal_int)>>8);  
  
  // calculate the "real power" in this sample pair and add to the accumulated sum
  long filtV_div4 = phaseShiftedSampleVminusDC_long>>2;  // reduce to 16-bits (now x64, or 2^6)
  long filtI_div4 = sampleIminusDC_long>>2; // reduce to 16-bits (now x64, or 2^6)
  long instP = filtV_div4 * filtI_div4;  // 32-bits (now x4096, or 2^12)
  instP = instP>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
  sumP +=instP; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
  samplesDuringThisCycle++;
  
  // store items for use during next loop
  cumVdeltasThisCycle_long += sampleVminusDC_long; // for use with LP filter
  lastSampleVminusDC_long = sampleVminusDC_long;  // required for phaseCal algorithm
  
  polarityOfLastSampleV = polarityNow;  // for identification of half cycle boundaries
}
// end of loop()


//boolean increaseLoadIfPossible()
void increaseLoadIfPossible()
{
  /* if permitted by A/F rules, turn on the highest priority logical load that is not already on.
   */
  boolean changed = false;
  
  // Only one load may operate freely at a time.  Other loads are prevented from 
  // switching until a sufficient period has elapsed since the last transition. 
  // Also, the energy level must not have risen since the previous ON transition.  
  // These measures allow a lower priority load to contribute if a higher priority
  // load is not having the desired effect, but not immediately.
  // 
  if (energyInBucket_long >= energyAtLastOnTransition_long)
  {     
    boolean timeout = (cycleCount > cycleCountAtLastTransition + interLoadSeparationDelay_cycles); 
    for (int i = 0; i < noOfDumploads && !changed; i++)
    {
      if (logicalLoadState[i] == LOAD_OFF)
      {
        if ((i == activeLoadID) || timeout)
        {
          logicalLoadState[i] = LOAD_ON;
          cycleCountAtLastTransition = cycleCount;
          energyAtLastOnTransition_long = energyInBucket_long;
          energyAtLastOffTransition_long = 3600; // reset the 'opposite' mechanism.
          activeLoadID = i;
          changed = true; 
        }
      }
    }
  }
  else
  {
    // energy level has not risen so there's no need to apply any more load
  }
//  return (changed);
}
 
//boolean decreaseLoadIfPossible()
void decreaseLoadIfPossible()
{
  /* if permitted by A/F rules, turn ooo the highest priority logical load that is not already off.
   */
  boolean changed = false;
  
  // Only one load may operate freely at a time.  Other loads are prevented from 
  // switching until a sufficient period has elapsed since the last transition. 
  // Also, the energy level must not have fallen since the previous ON transition.  
  // These measures allow a lower priority load to contribute if a higher priority
  // load is not having the desired effect, but not immediately.
  // 
  if (energyInBucket_long <= energyAtLastOnTransition_long)
  {     
    boolean timeout = (cycleCount > cycleCountAtLastTransition + interLoadSeparationDelay_cycles); 
//    for (int i = 0; i < noOfDumploads && !done; i++)
    for (int i = (noOfDumploads -1); i >= 0 && !changed; i--)
    {
      if (logicalLoadState[i] == LOAD_ON)
      {
        if ((i == activeLoadID) || timeout)
        {
          logicalLoadState[i] = LOAD_OFF;
          cycleCountAtLastTransition = cycleCount;
          energyAtLastOnTransition_long = energyInBucket_long;
          energyAtLastOffTransition_long = 0; // reset the 'opposite' mechanism.
          activeLoadID = i;
          changed = true; 
        }
      }
    }
  }
  else
  {
    // energy level has not fallen so there's no need to apply any more load
  }
//  return (changed);
}
 


void updatePhysicalLoadStates()
/*
 * This function provides the link between the logical and physical loads.  The 
 * array, logicalLoadState[], contains the on/off state of all logical loads, with 
 * element 0 being for the one with the highest priority.  The array, 
 * physicalLoadState[], contains the on/off state of all physical loads. 
 * 
 * By default, the association between the physical and logical loads is 1:1.  If
 * the remote load is set to have priority, the logical-to-physical association for
 * loads 0 and 1 are swapped.
 *
 * Background info: 
 * - Physical load 0 is local, and is controlled by use of an output pin.  
 * - Physical load 1 is remote, and is controlled via the associated RFM12B module.
 */
{
  for (int i = 0; i < noOfDumploads; i++)
  {
    physicalLoadState[i] = logicalLoadState[i]; 
  }
   
  if (loadPriorityMode == REMOTE_HAS_PRIORITY)
  {
    // swap physical loads 0 & 1 if remote load has priority 
    physicalLoadState[0] = logicalLoadState[1];
    physicalLoadState[1] = logicalLoadState[0];
  } 
}


// this function changes the value of outputMode if the state of the external switch is altered 
void checkOutputModeSelection()  
{
  static byte outputModeSwitchcount = 0;
  int pinState = digitalRead(outputModeSelectorPin);
  if (pinState != outputMode)
  {
    outputModeSwitchcount++;
  }  
  if (outputModeSwitchcount >= 20)
  {
    outputModeSwitchcount = 0;
    outputMode = (enum outputModes)pinState;  // change the global variable
    Serial.print ("outputMode selection changed to ");
    if (outputMode == NORMAL) {
      Serial.println ( "normal"); }
    else {  
      Serial.println ( "anti-flicker"); }
    
    configureParamsForSelectedOutputMode();
  }
}

// this function changes the value of the load priorities if the state of the external switch is altered 
void checkLoadPrioritySelection()  
{
  static byte loadPrioritySwitchCcount = 0;
  int pinState = digitalRead(loadPrioritySelectorPin);
  if (pinState != loadPriorityMode)
  {
    loadPrioritySwitchCcount++;
  }  
  if (loadPrioritySwitchCcount >= 20)
  {
    loadPrioritySwitchCcount = 0;
    loadPriorityMode = (enum loadPriorityModes)pinState;  // change the global variable
    Serial.print ("loadPriority selection changed to ");
    if (loadPriorityMode == LOCAL_HAS_PRIORITY) {
      Serial.println ( "local"); }
    else {  
      Serial.println ( "remote"); }
  }
}


void configureParamsForSelectedOutputMode()
{
  energyThreshold_long = capacityOfEnergyBucket_long * 0.5;
       
  if (outputMode == ANTI_FLICKER)
  {
    postMidPointCrossingDelay_cycles = postMidPointCrossingDelayForAF_cycles; 
  }
  else
  { 
    postMidPointCrossingDelay_cycles = 0;
  }
  
  // display relevant settings for selected output mode
  Serial.print("  capacityOfEnergyBucket_long = ");
  Serial.println(capacityOfEnergyBucket_long);
  Serial.print("  energyThreshold_long   = ");
  Serial.println(energyThreshold_long);
  Serial.print("  postMidPointCrossingDelay_cycles = ");
  Serial.println(postMidPointCrossingDelay_cycles); 
  Serial.print("  interLoadSeparationDelay_cycles = ");
  Serial.println(interLoadSeparationDelay_cycles); 
  
  Serial.print(">>free RAM = ");
  Serial.println(freeRam());  // a useful value to keep an eye on
}

void send_rf_data()
{
  rf12_sleep(RF12_WAKEUP);
  // if ready to send + exit route if it gets stuck 
  int i = 0; 
  while (!rf12_canSend() && i<10)
  { 
    rf12_recvDone(); 
    i++;
  }
  rf12_sendStart(0, &tx_data, sizeof tx_data);
  rf12_sendWait(2);
  rf12_sleep(RF12_SLEEP);
}



int freeRam () {
  extern int __heap_start, *__brkval; 
  int v; 
  return (int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval); 
}


#ifdef TALLYMODE
void tallymode_checks()
{
  if (tallymode_firstLoop) { 
    tallymode_setup(); } // user-dialogue for recording energy data
  else
  if (cycleCount >= tallymode_maxCycleCount) {
    tallymode_dispatchData(); // send recorded energy data to the Serial monitor  
//    cycleCount = 0;
    tallymode_firstLoop = true;  // get ready for another run
    pause(); } // so that user can access data from the Serial monitor

  else
  if (beyondStartUpPhase) {
      tallymode_noOfSamplePairs++; }
}


void  tallymode_setup()
{
  char inbuf[10];
  int tempInt;
  byte noOfBytes;
  boolean done;

 // <-- start of commented out section to save on RAM space.
 /*
   Serial.println ("WELCOME TO TALLYMODE "); 
  Serial.println ("This mode of operation allows the energy content of individual mains cycles");
  Serial.println ("to be analysed.  For nomal operation, the #define TALLYMODE statement");
  Serial.println ("should be commented out. ");
*/ 
   // <-- end of commented out section to save on RAM space.

  Serial.println ();  
  Serial.println ("Tallymode setup:");  
  Serial.print ("Time to run (secs)? ");
  done = false;
  while (!done) {
    noOfBytes = Serial.available();
    if (noOfBytes > 0) { done = true; } else { delay(100); }}
  for (tempInt = 0; tempInt < 10; tempInt++) { inbuf[tempInt] = 0; }
  Serial.readBytes(inbuf, noOfBytes); 
  tempInt = atoi(inbuf);  Serial.println (tempInt);
  tallymode_maxCycleCount = tempInt * CYCLES_PER_SECOND;
  tallymode_durationOfRecording = tempInt;
  
  Serial.print ("Min value (Watts)? ");
  done = false;
  while (!done) {
    noOfBytes = Serial.available();
    if (noOfBytes > 0) { done = true; } else { delay(100); }}
  for (tempInt = 0; tempInt < 10; tempInt++) { inbuf[tempInt] = 0; }
  Serial.readBytes(inbuf, noOfBytes); 
  tempInt = atoi(inbuf);  Serial.println (tempInt);
  tallymode_minVal = tempInt;
//  Serial.print(" tallymode_minVal = "); Serial.println(tallymode_minVal);   
  tallymode_minVal_long = tallymode_minVal * (1/powerCal); // used for power, not energy,
                                                           // hence there's no x100 term.
  
  Serial.print ("Max value (Watts)? ");
  done = false;
  while (!done) {
    noOfBytes = Serial.available();
    if (noOfBytes > 0) { done = true; } else { delay(100); }}
  for (tempInt = 0; tempInt < 10; tempInt++) { inbuf[tempInt] = 0; }
  Serial.readBytes(inbuf, noOfBytes); 
  tempInt = atoi(inbuf);  Serial.println (tempInt);
  tallymode_maxVal = tempInt;
//  Serial.print(" tallymode_maxVal = "); Serial.println(tallymode_maxVal);   

  if (tallymode_maxVal <= tallymode_minVal) {
    Serial.print(7); // beep
    Serial.println ("ERROR!"); }
  
  tallymode_stepVal = (float)(tallymode_maxVal - tallymode_minVal) / NUMBER_OF_TALLIES;
//  Serial.print(" tallymode_stepVal = "); Serial.println(tallymode_stepVal);   
  tallymode_stepVal_long = tallymode_stepVal * (1/powerCal); // used for power, not energy,
                                                             // hence there's no x50 term.
  tallymode_halfStep = (tallymode_stepVal_long / 2); // to avoid integer-rounding 

  for (tempInt = 0; tempInt < NUMBER_OF_TALLIES + 2; tempInt++) {
    tally[tempInt] = 0; }
    
  energyInBucket_long = (capacityOfEnergyBucket_long / 2); // for rapid startup of power distribution
  cycleCount = 0;
  tallymode_noOfValuesTallied = 0;
  tallymode_noOfSamplePairs = 0; 
  beyondStartUpPhase = false; // to allow LPF to re-settle for next run
  tallymode_firstLoop = false; 

  Serial.print(" Recording will start in ");
  Serial.print(startUpPeriod);   
  Serial.println(" seconds ... ");
  
  startTime = millis();
};


void  tallymode_updateData(long power_long)
{ 
  if (power_long >= 0) { // for correct operation with 'long -> integer' rounding
    power_long += tallymode_halfStep; }
  else { 
    power_long -= tallymode_halfStep; }
//
  int index = (power_long - tallymode_minVal_long) / tallymode_stepVal_long;
  
  if (index < 0) {
    index = 0; } // tally[0] is for underflow
  else 
  if (index > NUMBER_OF_TALLIES) {
    index = NUMBER_OF_TALLIES + 1; } // tally[N+1] is for overflow
  else {
    tally[index]++; }
    
  tallymode_noOfValuesTallied++; 
}


void  tallymode_dispatchData()
{
  
//   <<- start of commented out section, to save on RAM space!
/*   
  Serial.println ();
  Serial.println ("Format of results: ");
  Serial.print ("Sorted data runs from tally[1] (min) to tally[");
  Serial.print (NUMBER_OF_TALLIES);
  Serial.println ("] (max).");
  Serial.print ("tally[0] is below range; tally[ ");
  Serial.print (NUMBER_OF_TALLIES + 1);
  Serial.println ("] is above range.");
  Serial.println("Tally results are displayed with max power first. For each one:");
  Serial.println("  mid-range power of tally (Watts), number of times recorded.");
  Serial.println();
*/   
//   <<- end of commented out section, to save on RAM space!
  
  Serial.println("Results for this run:");
  Serial.print (tallymode_minVal);
  Serial.println (", <- min value of tally range (W)");
  Serial.print (tallymode_maxVal);
  Serial.println (", <- max value of tally range (W)");
  Serial.print (tallymode_stepVal);
  Serial.println (", <- step value between tallies (W)");
  Serial.print (NUMBER_OF_TALLIES);
  Serial.println (", <- No. of tallies");
  Serial.print (tallymode_durationOfRecording);
  Serial.println (", <- duration (secs)");
  Serial.print (tallymode_noOfValuesTallied);
  Serial.println (", <- no of values tallied");
  Serial.print ( tallymode_noOfSamplePairs / 
    ((float)tallymode_durationOfRecording * CYCLES_PER_SECOND), 1);
  Serial.println (", <- loops per mains cycle (av)");
  Serial.println("***");
 
  float powerVal;
  for (int index = NUMBER_OF_TALLIES + 1; index >= 0; index--)
  {
/*  
    // display the index value for this tally
    Serial.print (index);
    Serial.print (", ");
*/    
    // display the power value for this tally
    if (index == NUMBER_OF_TALLIES + 1)
    {
      Serial.print (">"); 
      Serial.print (tallymode_maxVal);
      Serial.print ("W"); 
    }
    else
    if (index == 0) 
    {
      Serial.print ("<"); 
      Serial.print (tallymode_minVal);
      Serial.print ("W"); 
    }
    else  
    {
      if (index == NUMBER_OF_TALLIES) 
      {
        powerVal = tallymode_maxVal - (tallymode_stepVal / 2); 
      }
      else 
      {
        powerVal -= tallymode_stepVal; 
      }
      
      if ((int)powerVal == powerVal)
      {
        Serial.print (powerVal, 0); // to suppress the decimal part for integers 
      }
      else
      {
        Serial.print (powerVal);  // non-integers are displayed to 2 dec places
      }
    }  
    
    // display the nimber of hits for this tally
    Serial.print (", ");
    Serial.println (tally[index]);  
  }
};


void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0) {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      if (dummyByte == 'g') done++; }
  }    
}
#endif


// helper function, to process LED events, can be conveniently called once per mains cycle
void checkLedStatus()
{
/* .. pin conflict with RFM12B !!!
 *
  static byte ledState, prevLedState;
  static boolean ledRecentlyOnFlag = false;
  static unsigned long ledOnAt;

  ledState = digitalRead (ledDetectorPin);

  if (ledState != prevLedState)
  {
    // led has changed state
    if (ledState == LED_ON)
    {
      // led has just gone on
      ledOnAt = millis();
      ledRecentlyOnFlag = true;
    }
    else
    {
      // led has just gone off   
      if (ledRecentlyOnFlag == true)
      {
        ledRecentlyOnFlag = false;        
        Serial.print ("** LED PULSE ** "); // this is a chargeable event  
      }
      else
      {
        Serial.print ("** LED OFF ** "); // 'no longer exporting' is also a chargeable event
      }
      Serial.println(millis()/1000);
    }    
  }
  else
  {
    // the LED state has not changed
    if (ledState == LED_ON)
    {
      if (ledRecentlyOnFlag == true)
      { 
        // check to see if the known duration of a pulse has been exceeded    
        unsigned long timeNow = millis();      
        if ((timeNow - ledOnAt) > 50)
        {
          Serial.print ("** LED ON **");  // 'exporting' is a non-chargeable state     
          Serial.print (",  energy in bucket = "); 
          Serial.println((long)(energyInBucket_4led_long)); 
          ledRecentlyOnFlag = false;   
        }     
      }
    }
  } 
  prevLedState = ledState;   
*/  
}

/*
      if (beyondStartUpPhase)
      {  

      }
      else
      {  
        // check whether the system has had time to settle
        if(cycleCount > (startUpPeriod * CYCLES_PER_SECOND))
        {
          beyondStartUpPhase = true;
        }
      } 
        
*/



