// This sketch provides continuous monitoring of real power on four channels, 
// which are split across two phases.  The interrupt-based kernel was kindly 
// provided by Jorg Becker.
//
//      Robin Emley (calypso_rae on Open Energy Monitor Forum)
//      October 2013

#include <Arduino.h> // may not be needed, but it's probably a good idea to include this
#include <JeeLib.h>     // JeeLib is available at from: http://github.com/jcw/jeelib

// In this sketch, the ADC is free-running with a cycle time of ~104uS.

//  WORKLOAD_CHECK is available for determining how much spare processing time there 
//  is.  To activate this mode, the #define line below should be included: 
//#define WORKLOAD_CHECK  

// ----------------- RF setup  ---------------------
#define freq RF12_868MHZ // Use the freq to match the module you have.

const int nodeID = 10;  // emonTx RFM12B node ID
const int networkGroup = 210;  // emonTx RFM12B wireless network group - needs to be same as emonBase and emonGLCD 
const int UNO = 1;  // Set to 0 if you're not using the UNO bootloader (i.e using Duemilanove) 
                                               // - All Atmega's shipped from OpenEnergyMonitor come with Arduino Uno bootloader
 typedef struct { 
   int msgNumber; 
   int realPower_Ia1;
   int realPower_Ia2;
   int realPower_Ib1;
   int realPower_Ib2; } Tx_struct;    // revised data for RF comms
Tx_struct tx_data;

// -----------------------------------------------------

enum polarities {NEGATIVE, POSITIVE};


// ----------- Pinout assignments  -----------
//
// digital pins:
// dig pin 0 is for Serial Rx
// dig pin 1 is for Serial Tx
// dig pin 2 is for the RFM12B module (IRQ) 
// dig pin 10 is for the RFM12B module (SEL) 
// dig pin 11 is for the RFM12B module (SDI) 
// dig pin 12 is for the RFM12B module (SDO) 
// dig pin 13 is for the RFM12B module (CLK) 

// analogue input pins 
const byte sensor_Va = 0;      
const byte sensor_Ia1 = 1;  
const byte sensor_Ia2 = 2;  
const byte sensor_Vb = 3;      
const byte sensor_Ib1 = 4;  
const byte sensor_Ib2 = 5;  


// --------------  general global variables -----------------
//
// Some of these variables are used in multiple blocks so cannot be static.
// For integer maths, many variables need to be 'long'
//
boolean beyondStartUpPhase = false;    // start-up delay, allows things to settle
const byte startUpPeriod = 3;      // in seconds, to allow LP filter to settle
const int DCoffset_I = 512;        // nominal mid-point value of ADC @ x1 scale

int phaseCal_int_Ia1;                  // to avoid the need for floating-point maths
int phaseCal_int_Ia2;                  // to avoid the need for floating-point maths
int phaseCal_int_Ib1;                  // to avoid the need for floating-point maths
int phaseCal_int_Ib2;                  // to avoid the need for floating-point maths
long DCoffset_Va_long;              // <--- for LPF
long DCoffset_Vb_long;              // <--- for LPF
long DCoffsetV_min;               // <--- for LPF
long DCoffsetV_max;               // <--- for LPF

// for interaction between the main processor and the ISR 
volatile boolean dataReady = false;
int sample_Va;
int sample_Ia1;
int sample_Ia2;
int sample_Vb;
int sample_Ib1;
int sample_Ib2;


// Calibration values
//-------------------
// Two calibration values are used in this sketch: powerCal, and phaseCal. 
// With most hardware, the default values are likely to work fine without 
// need for change.  A compact explanation of each of these values now follows:

// When calculating real power, which is what this code does, the individual 
// conversion rates for voltage and current are not of importance.  It is 
// only the conversion rate for POWER which is important.  This is the 
// product of the individual conversion rates for voltage and current.  It 
// therefore has the units of ADC-steps squared per Watt.  Most systems will
// have a power conversion rate of around 20 (ADC-steps squared per Watt).
// 
// powerCal is the RECIPR0CAL of the power conversion rate.  A good value 
// to start with is therefore 1/20 = 0.05 (Watts per ADC-step squared)
//
const float powerCal_Ia1 = 0.053;  
const float powerCal_Ia2 = 0.053;  
const float powerCal_Ib1 = 0.053;  
const float powerCal_Ib2 = 0.053;  
                        
// phaseCal is used to alter the phase of the voltage waveform relative to the
// current waveform.  The algorithm interpolates between the most recent pair
// of voltage samples according to the value of phaseCal. 
//
//    With phaseCal = 1, the most recent sample is used.  
//    With phaseCal = 0, the previous sample is used
//    With phaseCal = 0.5, the mid-point (average) value in used
//
// NB. Any tool which determines the optimal value of phaseCal must have a similar 
// scheme for taking sample values as does this sketch!
//
const float  phaseCal_Ia1 = 1.0;
const float  phaseCal_Ia2 = 1.0;
const float  phaseCal_Ib1 = 1.0;
const float  phaseCal_Ib2 = 1.0;


void setup()
{  
  rf12_initialize(nodeID, freq, networkGroup);             // initialize RF
  rf12_sleep(RF12_SLEEP);
  
  Serial.begin(9600);                                      // initialize Serial interface
  Serial.println();
  Serial.println();
  Serial.println();
  Serial.println("----------------------------------");
  Serial.println("Sketch ID:  TwoPhaseFourChannel_RPmonitor_1.ino");
       

       
  // When using integer maths, calibration values that have supplied in floating point 
  // form need to be rescaled.  
  //
  phaseCal_int_Ia1 = phaseCal_Ia1 * 256; // for integer maths
  phaseCal_int_Ia2 = phaseCal_Ia2 * 256; // for integer maths
  phaseCal_int_Ib1 = phaseCal_Ib1 * 256; // for integer maths
  phaseCal_int_Ib2 = phaseCal_Ib2 * 256; // for integer maths
    
  // Define operating limits for the LP filters which identify DC offset in the voltage 
  // sample streams.  By limiting the output range, these filters always should start up 
  // correctly.
  DCoffset_Va_long = 512L * 256; // nominal mid-point value of ADC @ x256 scale  
  DCoffset_Vb_long = 512L * 256; // nominal mid-point value of ADC @ x256 scale  

  DCoffsetV_min = (long)(512L - 100) * 256; // mid-point of ADC minus a working margin
  DCoffsetV_max = (long)(512L + 100) * 256; // mid-point of ADC plus a working margin

  
  Serial.println ("ADC mode:       free-running");
  
  // Set up the ADC to be free-running 
  ADCSRA  = (1<<ADPS0)+(1<<ADPS1)+(1<<ADPS2);  // Set the ADC's clock to system clock / 128
  ADCSRA |= (1 << ADEN);                 // Enable the ADC 
  
  ADCSRA |= (1<<ADATE);  // set the Auto Trigger Enable bit in the ADCSRA register.  Because 
                         // bits ADTS0-2 have not been set (i.e. they are all zero), the 
                         // ADC's trigger source is set to "free running mode".
                         
  ADCSRA |=(1<<ADIE);    // set the ADC interrupt enable bit. When this bit is written 
                         // to one and the I-bit in SREG is set, the 
                         // ADC Conversion Complete Interrupt is activated. 

  ADCSRA |= (1<<ADSC);   // start ADC manually first time 
  sei();                 // Enable Global Interrupts  

     
  char flag = 0;
  Serial.print ( "Extra Features: ");  
#ifdef WORKLOAD_CHECK  
  Serial.print ( "WORKLOAD_CHECK ");
  flag++;
#endif
  if (flag == 0) {
    Serial.print ("none"); }
  Serial.println ();
        
  Serial.print ( "powerCal_Ia1 =      "); Serial.println (powerCal_Ia1,4);
  Serial.print ( "phaseCal_Ia1 =      "); Serial.println (phaseCal_Ia1);
  Serial.print ( "powerCal_Ia2 =      "); Serial.println (powerCal_Ia2,4);
  Serial.print ( "phaseCal_Ia2 =      "); Serial.println (phaseCal_Ia2);
  Serial.print ( "powerCal_Ib1 =      "); Serial.println (powerCal_Ib1,4);
  Serial.print ( "phaseCal_Ib1 =      "); Serial.println (phaseCal_Ib1);
  Serial.print ( "powerCal_Ib2 =      "); Serial.println (powerCal_Ib2,4);
  Serial.print ( "phaseCal_Ib2 =      "); Serial.println (phaseCal_Ib2);
  
  Serial.println ("----");    

#ifdef WORKLOAD_CHECK
   Serial.println ("WELCOME TO WORKLOAD_CHECK ");
  
//   <<- start of commented out section, to save on RAM space!
/*   
   Serial.println ("  This mode of operation allows the spare processing capacity of the system");
   Serial.println ("to be analysed.  Additional delay is gradually increased until all spare time");
   Serial.println ("has been used up.  This value (in uS) is noted and the process is repeated.  ");
   Serial.println ("The delay setting is increased by 1uS at a time, and each value of delay is ");
   Serial.println ("checked several times before the delay is increased. "); 
 */ 
//  <<- end of commented out section, to save on RAM space!

   Serial.println ("  The displayed value is the amount of spare time, per pair of V & I samples, ");
   Serial.println ("that is available for doing additional processing.");
   Serial.println ();
 #endif
}

// An Interrupt Service Routine is now defined in which the ADC is instructed to perform 
// a conversion of the voltage signal and each of the signals for current.  A "data ready" 
// flag is set after each voltage conversion has been completed, it being the last one in
// the sequence.  
//   Samples for current are taken first because the phase of the waveform for current is 
// generally slightly advanced relative to the waveform for voltage.  The data ready flag 
// is cleared within loop().

// This Interrupt Service Routine is for use when the ADC is in the free-running mode.
// It is executed whenever an ADC conversion has finished, approx every 104 us.  In 
// free-running mode, the ADC has already started its next conversion by the time that
// the ISR is executed.  The ISR therefore needs to "look ahead". 
//   At the end of conversion Type N, conversion Type N+1 will start automatically.  The ISR 
// which runs at this point therefore needs to capture the results of conversion Type N , 
// and set up the conditions for conversion Type N+2, and so on.  
// 
ISR(ADC_vect)  
{                                         
  static unsigned char sample_index = 0;
  
  switch(sample_index)
  {
    case 0:
      sample_Va = ADC; 
      ADMUX = 0x40 + sensor_Ia2; // set up the next-but-one conversion
      sample_index++; // advance the control flag             
      dataReady = true; 
      break;
    case 1:
      sample_Ia1 = ADC; 
      ADMUX = 0x40 + sensor_Vb; // for the next-but-one conversion
      sample_index++; // advance the control flag                
      break;
    case 2:
      sample_Ia2 = ADC; 
      ADMUX = 0x40 + sensor_Ib1; // for the next-but-one conversion
      sample_index++; // advance the control flag                
      break;
    case 3:
      sample_Vb = ADC; 
      ADMUX = 0x40 + sensor_Ib2; // for the next-but-one conversion
      sample_index++; // advance the control flag                 
      break;
    case 4:
      sample_Ib1 = ADC; 
      ADMUX = 0x40 + sensor_Va; // for the next-but-one conversion
      sample_index++; // advance the control flag                 
      break;
    case 5:
      sample_Ib2 = ADC; 
      ADMUX = 0x40 + sensor_Ia1; // for the next-but-one conversion
      sample_index = 0; // reset the control flag                
      break;
    default:
      sample_index = 0;                 // to prevent lockup (should never get here)      
  }  
}


// When using interrupt-based logic, the main processor waits in loop() until the 
// dataReady flag has been set by the ADC.  Once this flag has been set, the main
// processor clears the flag and proceeds with all the processing for one pair of 
// V & I samples.  It then returns to loop() to wait for the next pair to become 
// available.
//   If the next pair of samples become available before the processing of the 
// previous pair has been completed, data could be lost.  This situation can be 
// avoided by prior use of the WORKLOAD_CHECK mode.  Using this facility, the amount
// of spare processing capacity per loop can be determined.  
//
void loop()             
{ 
#ifdef WORKLOAD_CHECK
  static int del = 0; // delay, as passed to delayMicroseconds()
  static int res = 0; // result, to be displayed at the next opportunity
  static byte count = 0; // to allow multiple runs per setting
  static byte displayFlag = 0; // to determine when printing may occur
#endif
  
  if (dataReady)   // flag is set after every pair of ADC conversions
  {
    dataReady = false; // reset the flag
    allGeneralProcessing(); // executed once for each pair of V&I samples
    
#ifdef WORKLOAD_CHECK 
    delayMicroseconds(del); // <--- to assess how much spare time there is
    if (dataReady)       // if data is ready again, delay was too long
    { 
      res = del;             // note the exact value
      del = 1;               // and start again with 1us delay   
      count = 0;
      displayFlag = 0;   
    }
    else
    {
      count++;          // to give several runs with the same value
      if (count > 50)
      {
        count = 0;
        del++;          //  increase delay by 1uS
      } 
    }
#endif  

  }  // <-- this closing brace needs to be outside the WORKLOAD_CHECK blocks! 
  
#ifdef WORKLOAD_CHECK 
  switch (displayFlag) 
  {
    case 0: // the result is available now, but don't display until the next loop
      displayFlag++;
      break;
    case 1: // with minimum delay, it's OK to print now
      Serial.print(res);
      displayFlag++;
      break;
    case 2: // with minimum delay, it's OK to print now
      Serial.println("uS");
      displayFlag++;
      break;
    default:; // for most of the time, displayFlag is 3           
  }
#endif
  
} // end of loop()


// This routine is called to process each pair of V & I samples.  Note that when using 
// interrupt-based code, it is not necessary to delay the processing of each pair of 
// samples as was done in Mk2a builds.  This is because there is no longer a strict 
// alignment between the obtaining of each sample by the ADC and the processing that can 
// be done by the main processor while the ADC conversion is in progress.  
//   When interrupts are used, the main processor and the ADC work autonomously, their
// operation being only linked via the dataReady flag.  As soon as data is made available
// by the ADC, the main processor can start to work on it immediately.  
//
void allGeneralProcessing()
{
  static long sumP_Ia1;                         
  static long sumP_Ia2;                              
  static long sumP_Ib1;                           
  static long sumP_Ib2;                            
  static enum polarities polarityOfLastSampleV;  // for zero-crossing detection
  static long cumVa_deltasThisCycle_long;    // for the LPF which determines DC offset (voltage)
  static long cumVb_deltasThisCycle_long;    // for the LPF which determines DC offset (voltage)
  static long lastSampleVa_minusDC_long;     //    for the phaseCal algorithm
  static long lastSampleVb_minusDC_long;     //    for the phaseCal algorithm
  static int msgNumber = 0;
  
  static int sequenceCount = 0;
  static int samplesDuringThisWindow_Ia1;            
  static int samplesDuringThisWindow_Ia2;            
  static int samplesDuringThisWindow_Ib1;            
  static int samplesDuringThisWindow_Ib2;            
  
  // remove DC offset from the raw voltage sample by subtracting the accurate value 
  // as determined by a LP filter.
  long sampleVa_minusDC_long = ((long)sample_Va<<8) - DCoffset_Va_long; 
  long sampleVb_minusDC_long = ((long)sample_Vb<<8) - DCoffset_Vb_long; 

  // determine polarity, to aid the logical flow
  enum polarities polarityNow;   
  if(sampleVa_minusDC_long > 0) { 
    polarityNow = POSITIVE; }
  else { 
    polarityNow = NEGATIVE; }

  if (polarityNow == POSITIVE) 
  {                           
    if (beyondStartUpPhase)
    {  
      if (polarityOfLastSampleV != POSITIVE)
      {
        // This is the start of a new +ve half cycle (just after the zero-crossing point)
        sequenceCount++;
        static char separatorString[5] = ", ";
        long realPower_long;
        
        switch(sequenceCount)
        {
          case 20: 
            realPower_long = sumP_Ia1 / samplesDuringThisWindow_Ia1; 
            tx_data.realPower_Ia1 = realPower_long * powerCal_Ia1;
            Serial.print("samples per mains cycle = " );           
            Serial.println(samplesDuringThisWindow_Ia1 /100.0);           
            sumP_Ia1 = 0;
            samplesDuringThisWindow_Ia1 = 0;
            Serial.print(tx_data.realPower_Ia1);
            break;
          case 40: 
            realPower_long = sumP_Ia2 / samplesDuringThisWindow_Ia2; 
            tx_data.realPower_Ia2 = realPower_long * powerCal_Ia2;
            sumP_Ia2 = 0;
            samplesDuringThisWindow_Ia2 = 0;
            Serial.print(separatorString);
            Serial.print(tx_data.realPower_Ia2);           
            break;
          case 60: 
            realPower_long = sumP_Ib1 / samplesDuringThisWindow_Ib1; 
            tx_data.realPower_Ib1 = realPower_long * powerCal_Ib1;
            sumP_Ib1 = 0;
            samplesDuringThisWindow_Ib1 = 0;
            Serial.print(separatorString);
            Serial.print(tx_data.realPower_Ib1);           
           break;
          case 80: 
            realPower_long = sumP_Ib2 / samplesDuringThisWindow_Ib2; 
            tx_data.realPower_Ib2 = realPower_long * powerCal_Ib2;
            sumP_Ib2 = 0;
            samplesDuringThisWindow_Ib2 = 0;
            Serial.print(separatorString);
            Serial.print(tx_data.realPower_Ib2);           
           break;
          case 100: 
            tx_data.msgNumber = msgNumber++;
            send_rf_data();  //  dispatch the RF message
            sequenceCount = 0;
            Serial.print(separatorString);
            Serial.println(tx_data.msgNumber++);           
            break;
          default: 
            if (sequenceCount > 100) {
              // should never get here
              sequenceCount = 0; }         
        }
        
        // Calculate the average real power during the measurement period.
        //
        // sumP contains the sum of many individual calculations of instantaneous power.  In  
        // order to obtain the average power during the relevant period, sumP must be divided 
        // by the number of samples that have contributed to its value.
        //
        
        // The next stage is to apply a calibration factor so that real power can be expressed 
        // in Watts.  That's fine for floating point maths, but it's not such
        // a good idea when integer maths is being used.  To keep the numbers large, and also 
        // to save time, calibration of power is omitted at this stage.  realPower_long is 
        // therefore (1/powerCal) times larger than the actual power in Watts.
        //
        long realPower_long_Ia1 = sumP_Ia1 / samplesDuringThisWindow_Ia1; 
        long realPower_long_Ia2 = sumP_Ia2 / samplesDuringThisWindow_Ia2; 
        long realPower_long_Ib1 = sumP_Ib1 / samplesDuringThisWindow_Ib1; 
        long realPower_long_Ib2 = sumP_Ib2 / samplesDuringThisWindow_Ib2; 
   
 
      } // end of processing that is specific to the first Vsample in each +ve half cycle   
    }
    else
    {  
      // wait until the DC-blocking filters have had time to settle
      if(millis() > startUpPeriod * 1000) 
      {
        beyondStartUpPhase = true;
        Serial.println ("Go!");
      }
    }
  } // end of processing that is specific to samples where the voltage is positive
  
  else // the polarity of this sample is negative
  {     
    if (polarityOfLastSampleV != NEGATIVE)
    {
      // This is the start of a new -ve half cycle (just after the zero-crossing point)
      //
      // This is a convenient point to update the twin Low Pass Filters for DC-offset removal,
      // one on each voltage channel.  This needs to be done right from the start.
      long previousOffset; 
      
      previousOffset = DCoffset_Va_long; // for voltage source Va
      DCoffset_Va_long = previousOffset + (cumVa_deltasThisCycle_long>>6); // faster than * 0.01
      cumVa_deltasThisCycle_long = 0;
      previousOffset = DCoffset_Vb_long;  // for voltage source Vb
      DCoffset_Vb_long = previousOffset + (cumVb_deltasThisCycle_long>>6); // faster than * 0.01
      cumVb_deltasThisCycle_long = 0;
      
      // To ensure that each of these LP filters will always start up correctly when 240V AC is 
      // available, its output value needs to be prevented from drifting beyond the likely range 
      // of the voltage signal.  This avoids the need to include a HPF as is often used for 
      // sketches of this type.
      //
      if (DCoffset_Va_long < DCoffsetV_min) {  // for voltage source Va
        DCoffset_Va_long = DCoffsetV_min; }
      else  
      if (DCoffset_Va_long > DCoffsetV_max) {
        DCoffset_Va_long = DCoffsetV_max; }
        
      if (DCoffset_Vb_long < DCoffsetV_min) {  // for voltage source Va
        DCoffset_Vb_long = DCoffsetV_min; }
      else  
      if (DCoffset_Vb_long > DCoffsetV_max) {
        DCoffset_Vb_long = DCoffsetV_max; }
        
    } // end of processing that is specific to the first Vsample in each -ve half cycle
  } // end of processing that is specific to samples where the voltage is positive
  
  // Processing for EVERY pair of samples. Most of this code is not used during the 
  // start-up period, but it does no harm to leave it in place.  Accumulated values 
  // are cleared when beyondStartUpPhase is set to true.
  //
  // remove most of the DC offset from the current sample (the precise value does not matter)
  long sampleIminusDC_long_Ia1 = ((long)(sample_Ia1 - DCoffset_I))<<8;
  long sampleIminusDC_long_Ia2 = ((long)(sample_Ia2 - DCoffset_I))<<8;
  long sampleIminusDC_long_Ib1 = ((long)(sample_Ib1 - DCoffset_I))<<8;
  long sampleIminusDC_long_Ib2 = ((long)(sample_Ib2 - DCoffset_I))<<8;
  
  // phase-shift the voltage waveform so that it aligns with the current when a 
  // resistive load is used
  long  phaseShiftedSampleVa_minusDC_long_Ia1 = lastSampleVa_minusDC_long
         + (((sampleVa_minusDC_long - lastSampleVa_minusDC_long)*phaseCal_int_Ia1)>>8);  
  long  phaseShiftedSampleVa_minusDC_long_Ia2 = lastSampleVa_minusDC_long
         + (((sampleVa_minusDC_long - lastSampleVa_minusDC_long)*phaseCal_int_Ia2)>>8);  
  long  phaseShiftedSampleVb_minusDC_long_Ib1 = lastSampleVb_minusDC_long
         + (((sampleVb_minusDC_long - lastSampleVb_minusDC_long)*phaseCal_int_Ib1)>>8);  
  long  phaseShiftedSampleVb_minusDC_long_Ib2 = lastSampleVb_minusDC_long
         + (((sampleVb_minusDC_long - lastSampleVb_minusDC_long)*phaseCal_int_Ib2)>>8);  
  
  // calculate the "real power" in this sample pair and add to the accumulated sum
  long filtV_div4_Ia1 = phaseShiftedSampleVa_minusDC_long_Ia1>>2;  // reduce to 16-bits (now x64, or 2^6)
  long filtI_div4_Ia1 = sampleIminusDC_long_Ia1>>2; // reduce to 16-bits (now x64, or 2^6)
  long instP_Ia1 = filtV_div4_Ia1 * filtI_div4_Ia1;  // 32-bits (now x4096, or 2^12)
  instP_Ia1 = instP_Ia1>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
  sumP_Ia1 +=instP_Ia1; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
  
  long filtV_div4_Ia2 = phaseShiftedSampleVa_minusDC_long_Ia2>>2;  // reduce to 16-bits (now x64, or 2^6)
  long filtI_div4_Ia2 = sampleIminusDC_long_Ia2>>2; // reduce to 16-bits (now x64, or 2^6)
  long instP_Ia2 = filtV_div4_Ia2 * filtI_div4_Ia2;  // 32-bits (now x4096, or 2^12)
  instP_Ia2 = instP_Ia2>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
  sumP_Ia2 +=instP_Ia2; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
  
  long filtV_div4_Ib1 = phaseShiftedSampleVb_minusDC_long_Ib1>>2;  // reduce to 16-bits (now x64, or 2^6)
  long filtI_div4_Ib1 = sampleIminusDC_long_Ib1>>2; // reduce to 16-bits (now x64, or 2^6)
  long instP_Ib1 = filtV_div4_Ib1 * filtI_div4_Ib1;  // 32-bits (now x4096, or 2^12)
  instP_Ib1 = instP_Ib1>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
  sumP_Ib1 +=instP_Ib1; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
  
  long filtV_div4_Ib2 = phaseShiftedSampleVb_minusDC_long_Ib2>>2;  // reduce to 16-bits (now x64, or 2^6)
  long filtI_div4_Ib2 = sampleIminusDC_long_Ib2>>2; // reduce to 16-bits (now x64, or 2^6)
  long instP_Ib2 = filtV_div4_Ib2 * filtI_div4_Ib2;  // 32-bits (now x4096, or 2^12)
  instP_Ib2 = instP_Ib2>>12;     // scaling is now x1, as for Mk2 (V_ADC x I_ADC)       
  sumP_Ib2 +=instP_Ib2; // cumulative power, scaling as for Mk2 (V_ADC x I_ADC)
  
  samplesDuringThisWindow_Ia1++;
  samplesDuringThisWindow_Ia2++;
  samplesDuringThisWindow_Ib1++;
  samplesDuringThisWindow_Ib2++;
  
  // store items for use during next loop
  cumVa_deltasThisCycle_long += sampleVa_minusDC_long; // for use with LP filter
  cumVb_deltasThisCycle_long += sampleVb_minusDC_long; // for use with LP filter
  lastSampleVa_minusDC_long = sampleVa_minusDC_long;  // required for phaseCal algorithm
  lastSampleVb_minusDC_long = sampleVb_minusDC_long;  // required for phaseCal algorithm
  
  polarityOfLastSampleV = polarityNow;  // for identification of half cycle boundaries
}
// end of allGeneralProcessing()


void send_rf_data()
{
  rf12_sleep(RF12_WAKEUP);
  // if ready to send + exit route if it gets stuck 
  int i = 0; 
  while (!rf12_canSend() && i<10)
  { 
    rf12_recvDone(); 
    i++;
  }
  rf12_sendStart(0, &tx_data, sizeof tx_data);
  rf12_sendWait(2);
  rf12_sleep(RF12_SLEEP);
}

int freeRam () {
  extern int __heap_start, *__brkval; 
  int v; 
  return (int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval); 
}




