/*
 * Tool to capture the raw samples generated by an Arduino during several mains
 * cycles.  This data is displayed on the Serial Monitor, and is also available
 * for subsequent processing using a spreadsheet. 
 *
 * Pauses after each set of measurements has been taken.  Press 'g', then [cr], 
 * to repeat.
 *
 *      Robin Emley (calypso_rae on Open Energy Monitor Forum)
 *      December 2012
 */
 
#define POSITIVE 1
#define NEGATIVE 0
#define ON 0  // the external trigger device is active low
#define OFF 1

byte sensorPin_Va = 0;
byte sensorPin_Ia1 = 1;
byte sensorPin_Ia2 = 2;
byte sensorPin_Vb = 3;
byte sensorPin_Ib1 = 4;
byte sensorPin_Ib2 = 5;

long cycleCount = 0;
int samplesRecorded = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths

byte polarityNow; 
boolean beyondStartUpPhase = false;
byte currentStateOfTriac;

int lastSample_Va;     // stored value from the previous loop (HP filter is for voltage samples only)         
float lastFiltered_Va;  //  voltage values after HP-filtering to remove the DC offset
byte polarityOfLastSample_Va; // for zero-crossing detection
  
boolean recordingNow;
boolean recordingComplete;
byte cycleNumberBeingRecorded;
byte noOfCyclesToBeRecorded;

unsigned long recordingMayStartAt;
boolean firstLoop = true;
int settlingDelay = 5; // <<---  settling time (seconds) for HPF 

char blankLine[82];
char newLine[82];
int storedSample_Va[50]; 
int storedSample_Ia1[50];
int storedSample_Ia2[50];
int storedSample_Vb[50]; 
int storedSample_Ib1[50];
int storedSample_Ib2[50];

void setup()
{  
  Serial.begin(9600);
 
  // initialise each character of the display line
  blankLine[0] = '|';
  blankLine[80] = '|';
  
  for (int i = 1; i < 80; i++)
  {
    blankLine[i] = ' ';
  }
  
  blankLine[40] = '.';
  
  Serial.print(">>free RAM = ");
  Serial.println(freeRam());  // a useful value to keep an eye on

}


/*  Allow the system to run for several seconds so that the filtered 
 *  voltage waveform can settle down.  This info is needed for determining 
 *  the start of each new mains cycle.  During this period, a countdown 
 *  is displayed.
 *
 *  After the settling period has expired, raw samples taken during 
 *  one complete mains cycle are stored in an array.  The capacity of the 
 *  array needs to be sufficient for the number of sample pairs that may
 *  appear.  A 100 x 2 integer array will probably suffice.
 *
 *  At the start of the following cycle, the data collected during the 
 *  previous cycle data is sent to the Serial window.  
 */
void loop() // each iteration of loop is for one pair of measurements only
{
  if(firstLoop)
  {
    unsigned long timeNow = millis();
    Serial.print ("millis() now = ");
    Serial.println (timeNow);
    
    recordingMayStartAt = timeNow + (settlingDelay * 1000);
    Serial.print ("recordingMayStartAt ");
    Serial.println (recordingMayStartAt);
    
    recordingNow = false;
    firstLoop = false;
    recordingComplete = false;
    noOfCyclesToBeRecorded = 1;
    cycleNumberBeingRecorded = 0;
    samplesRecorded = 0;    
  }
  
  int sample_Va = analogRead(sensorPin_Va);                 //Read in raw voltage signal
  int sample_Ia1 = analogRead(sensorPin_Ia1);                 //Read in raw current signal
  int sample_Ia2 = analogRead(sensorPin_Ia2);                 //Read in raw current signal
  int sample_Vb = analogRead(sensorPin_Vb);                 //Read in raw voltage signal
  int sample_Ib1 = analogRead(sensorPin_Ib1);                 //Read in raw current signal
  int sample_Ib2 = analogRead(sensorPin_Ib2);                 //Read in raw current signal

  float filtered_Va = 0.996*(lastFiltered_Va + sample_Va - lastSample_Va); 

  byte polarityOfThisSample_Va;
  if(filtered_Va > 0)
  {
    polarityOfThisSample_Va = POSITIVE;
    
    if (polarityOfLastSample_Va != POSITIVE) 
    {
      // This is the start of a new mains cycle
      cycleCount++; 
         
      if (recordingNow == true) {
        if (cycleNumberBeingRecorded >= noOfCyclesToBeRecorded) {
          Serial.print ("No of cycles recorded = ");
          Serial.println (cycleNumberBeingRecorded);      
          dispatch_recorded_data(); } 
        else {
          cycleNumberBeingRecorded++; } }    

      else
      if((cycleCount % 50) == 1) {  
        unsigned long timeNow = millis();   
        if (timeNow > recordingMayStartAt) {
           recordingNow = true;
           cycleNumberBeingRecorded++; } 
        else  {
          Serial.println((int)(recordingMayStartAt - timeNow) / 1000); } }    
    } // end of specific processing for first +ve reading in each mains cycle
    
  } // end of specific processing of +ve cycles
  else
  {
    polarityOfThisSample_Va = NEGATIVE; 
    
    if (polarityOfLastSample_Va != NEGATIVE) 
    {
      // at the start of a new negative half cycle
    }
  }
  
  if (recordingNow == true)
  {
    storedSample_Va[samplesRecorded] = sample_Va;
    storedSample_Ia1[samplesRecorded] = sample_Ia1;
    storedSample_Ia2[samplesRecorded] = sample_Ia2;
    storedSample_Vb[samplesRecorded] = sample_Vb;
    storedSample_Ib1[samplesRecorded] = sample_Ib1;
    storedSample_Ib2[samplesRecorded] = sample_Ib2;
    samplesRecorded++;
  }
    
  polarityOfLastSample_Va = polarityOfThisSample_Va;  
  lastSample_Va = sample_Va;                       
  lastFiltered_Va = filtered_Va;                  
} // end of loop()


void dispatch_recorded_data()
{      
  // display raw samples via the Serial Monitor
  // ------------------------------------------ 

  Serial.print("cycleCount ");
  Serial.print(cycleCount);
  Serial.print(",  samplesRecorded ");
  Serial.println(samplesRecorded);

  int Va, Ia1, Ia2;
  int min_Va = 1023, min_Ia1 = 1023, min_Ia2 = 1023;
  int max_Va = 0, max_Ia1 = 0, max_Ia2 = 0;
  int Vb, Ib1, Ib2;
  int min_Vb = 1023, min_Ib1 = 1023, min_Ib2 = 1023;
  int max_Vb = 0, max_Ib1 = 0, max_Ib2 = 0;
      
  for (int index = 0; index < samplesRecorded; index++) 
  {
    strcpy(newLine, blankLine);
    Va = storedSample_Va[index]; 
    Ia1 = storedSample_Ia1[index]; 
    Ia2 = storedSample_Ia2[index]; 
    Vb = storedSample_Vb[index]; 
    Ib1 = storedSample_Ib1[index]; 
    Ib2 = storedSample_Ib2[index]; 
 
    if (Va < min_Va){min_Va = Va;}
    if (Va > max_Va){max_Va = Va;}
    if (Ia1 < min_Ia1){min_Ia1 = Ia1;}
    if (Ia1 > max_Ia1){max_Ia1 = Ia1;}
    if (Ia2 < min_Ia2){min_Ia2 = Ia2;}
    if (Ia2 > max_Ia2){max_Ia2 = Ia2;}
    
    if (Vb < min_Vb){min_Vb = Vb;}
    if (Vb > max_Vb){max_Vb = Vb;}
    if (Ib1 < min_Ib1){min_Ib1 = Ib1;}
    if (Ib1 > max_Ib1){max_Ib1 = Ib1;}
    if (Ib2 < min_Ib2){min_Ib2 = Ib2;}
    if (Ib2 > max_Ib2){max_Ib2 = Ib2;}
    
    newLine[map(Va, 0, 1023, 0, 80)] = '0'; 
    newLine[map(Ia1, 0, 1023, 0, 80)] = '1'; 
    newLine[map(Ia2, 0, 1023, 0, 80)] = '2'; 
    newLine[map(Vb, 0, 1023, 0, 80)] = '3'; 
    newLine[map(Ib1, 0, 1023, 0, 80)] = '4'; 
    newLine[map(Ib2, 0, 1023, 0, 80)] = '5'; 
              
    if ((index % 1) == 0) // change this to "% 1" for full resolution
    {
      Serial.println(newLine);
    }
  }
    
  Serial.print("min_Va ");  Serial.print(min_Va);
  Serial.print(",  max_Va ");  Serial.print(max_Va);
  Serial.print(",  min_Ia1 ");  Serial.print(min_Ia1);
  Serial.print(",  max_Ia1 ");  Serial.println(max_Ia1);
  Serial.print(",  min_Ia2 ");  Serial.print(min_Ia2);
  Serial.println(",  max_Ia2 ");  Serial.println(max_Ia2);
  
  Serial.print("min_Vb ");  Serial.print(min_Vb);
  Serial.print(",  max_Vb ");  Serial.print(max_Vb);
  Serial.print(",  min_Ib1 ");  Serial.print(min_Ib1);
  Serial.print(",  max_Ib1 ");  Serial.println(max_Ib1);
  Serial.print(",  min_Ib2 ");  Serial.print(min_Ib2);
  Serial.print(",  max_Ib2 ");  Serial.println(max_Ib2);
  
  Serial.println();
  Serial.println();
      
  // despatch raw samples via the Serial Monitor 
  // ------------------------------------------- 
      
  Serial.println("Raw data from stored cycle: <Vsample>,<Isample>[cr]");
  Serial.print(samplesRecorded);
  Serial.println(", <<< No of sample pairs");

  for (int index = 0; index < samplesRecorded; index++) 
  {
    Serial.print (storedSample_Va[index]); 
    Serial.print(','); 
    Serial.println (storedSample_Ia1[index]);  
  }

  recordingNow = false;
  firstLoop = true;
  pause();
}      

void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0)
    {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      if (dummyByte == 'g') done++;
    }
  }    
}

int freeRam () {
  extern int __heap_start, *__brkval; 
  int v; 
  return (int) &v - (__brkval == 0 ? (int) &__heap_start : (int) __brkval); 
}


