/* 
 * Tool to display raw samples in graphical form.  Starts in' debug' mode
 * which uses synthesized values for voltage and current.  For normal operation, 
 * commment out the '#define DEBUG' statement a few lines down.
 *
 * Pauses after each set of measurements has been taken.  Press 'g', then [cr], 
 * to repeat.  
 *
 * 22/7/2012  Updated to display phase diagrams
 *
 *      Robin Emley (calypso_rae on Open Energy Monitor Forum)
 *      July 2012
 */

#include "EmonLib.h"      // Include Emon Library
EnergyMonitor emon1;      // Create an instance of the EnergyMonitor class
                          //  (only used here for the function, readVcc() )

// for normal operation, the next line should be commented out
// -----------------------------------------------------------
//
#define DEBUG
 
#define POSITIVE 1
#define NEGATIVE 0

// define the input pins
byte voltageSensorPin = 2;
byte currentSensorPin = 1;

long cycleCount = 0;
int samplesDuringThisMainsCycle = 0;
float cyclesPerSecond = 50; // use float to ensure accurate maths

long noOfSamplePairs = 0;
byte polarityNow = NEGATIVE; // probably not important, but better than being indeterminate
boolean recordingNow;

unsigned long timeNow;
unsigned long recordingMayStartAt;

int settlingDelay = 3; // <<---  settling time (seconds)for dc-blocking filters 
boolean firstLoop = true;

int SUPPLYVOLTAGE; // measured once in setup().  

// Local declarations of items for code that's been lifted from EmonLib. 
//
#ifdef DEBUG
// to ensure accurate maths for synthesised operation
float lastSampleV,sampleV;   // raw Voltage and Current samples, 
#else
int lastSampleV,sampleV, sampleI;   // raw Voltage and Current samples, 
#endif

double lastFilteredV,filteredV;  // raw V & I values after filtering to remove the DC offset,
double sumP;   //  cumulative sum of power calculations within this mains cycles
float phaseShiftedV;


double VCAL;
double ICAL;
double PHASECAL; 
double V_RATIO;
double I_RATIO;

char blankLine[82];
char newLine[82];

int storedSampleV[100];
int storedSampleI[100];
int storedPhaseShiftedV[100];

#ifdef DEBUG
// additional components for simulating waveforms in DEBUG mode
byte noOfVoltageSamplesPerCycle_4debug = 65;
int voltageSamples_4debug[65];
byte vsIndex_4debug = 0;
#endif


void setup()
{  
  Serial.begin(9600);
    
#if defined DEBUG 
  Serial.println("In DEBUG mode ..."); 
  Serial.println(); 
  
  // clear these calibration values when in DEBUG mode
  V_RATIO = 1.0;
  I_RATIO = 1.0;
  PHASECAL = 1.0; // <<<<---in debug mode, change PHASECAL here <<<<<<<
  
  /*  populate the voltage sample array for DEBUG use
   */
  float amplitude = 240 * sqrt(2); // in ADC units
  float angleIncrement = (2 * PI) / noOfVoltageSamplesPerCycle_4debug;
  float angleOffset = 0.01; // to avoid sample right at zero crossing point
  float angle, voltage;
  
  byte index;
  
  for (index = 0; index < noOfVoltageSamplesPerCycle_4debug; index++)
  {
    angle = (index * angleIncrement) + angleOffset;
    voltage = amplitude * sin(angle);
    voltageSamples_4debug[index] = (int)voltage;
  } 
  
#else
  Serial.println("In NORMAL mode ..."); 
  Serial.println(); 

  SUPPLYVOLTAGE = emon1.readVcc();
  V_RATIO = VCAL *((SUPPLYVOLTAGE/1000.0) / 1023.0);
  I_RATIO = ICAL *((SUPPLYVOLTAGE/1000.0) / 1023.0);

  Serial.print("SUPPLYVOLTAGE = "); 
  Serial.print(SUPPLYVOLTAGE); 
  Serial.print(", V_RATIO = "); 
  Serial.print(V_RATIO); 
  Serial.print(", I_RATIO = "); 
  Serial.println(I_RATIO);
 
  VCAL = 295;
  ICAL = 78;
  PHASECAL = 2.3;  // for 'normal' mode only
#endif

  blankLine[0] = '|';
  blankLine[80] = '|';
  
  // initialise each character
  for (int i = 1; i < 80; i++)
  {
    blankLine[i] = ' ';
  }
}


/*  Allow the system to run for several seconds so that the filtered 
 *  voltage waveform can settle down.  This info is needed for determining 
 *  the start of each new mains cycle.  During this period, a countdown 
 *  is displayed.
 *
 *  After the settling period has expired, raw samples taken during 
 *  one complete mains cycle are stored in an array.  The capacity of the 
 *  array needs to be sufficient for the number of sample pairs that may
 *  appear.  A 100 x 2 integer array will probably suffice.
 *
 *  At the start of the following cycle, the data collected during the 
 *  previous cycle data is sent to the Serial window.  
 */
void loop() // each iteration of loop is for one pair of measurements only
{
  lastSampleV=sampleV;                          //Used for digital high pass filter
  lastFilteredV = filteredV;                    //Used for offset removal

  if(firstLoop)
  {
    timeNow = millis();
    Serial.print ("millis() now = ");
    Serial.println (timeNow);
    
    recordingMayStartAt = timeNow + (settlingDelay * 1000);
    Serial.print ("recordingMayStartAt ");
    Serial.println (recordingMayStartAt);
    
    recordingNow = false;
    firstLoop = false;
  }

  //-----------------------------------------------------------------------------
  //  Read in the raw voltage and current sample 
  //     (values are synthesised when in DEBUG mode)
  //-----------------------------------------------------------------------------
#ifdef DEBUG
  sampleV = getVoltageSample_4debug(vsIndex_4debug); 
 
  // An easy way to obtain phase-shift in the current samples is to access the array of
  // voltage samples using an offset. With 65 samples per cycle in debug mode, 
  // each increase or decrease in offset correcponds to approx 5 degrees of phase-shift.
  //                                                       !
  //                                                      \!/    
  float sampleI = getVoltageSample_4debug(vsIndex_4debug + 0);  
  
  // Now reduce the amplitude so that the V & I waveforms are of different amplitude  
  sampleI -= 500;
  sampleI *= 0.75;
  sampleI += 500;
  
  vsIndex_4debug++;  
  if (vsIndex_4debug >= noOfVoltageSamplesPerCycle_4debug) 
  {
    vsIndex_4debug = 0;
  }
  
  
#else    
  sampleV = analogRead(voltageSensorPin);                 //Read in raw voltage signal
  sampleI = analogRead(currentSensorPin);                 //Read in raw current signal
#endif

  //------------------------------------------------------------
  // B1) Apply digital high pass filter to remove 2.5V DC offset  
    //----------------------------------------------------------
  filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV);
//  phaseShiftedV = lastFilteredV + PHASECAL * (filteredV - lastFilteredV); 
  phaseShiftedV = lastSampleV + PHASECAL * (sampleV - lastSampleV); 
  
  // Establish the polarities of the latest and previous filtered voltage samples
  byte polarityOfLastReading = polarityNow;
  if(filteredV >= 0) 
    polarityNow = POSITIVE; 
  else 
    polarityNow = NEGATIVE;
 

  if ((polarityNow == POSITIVE) && (polarityOfLastReading == NEGATIVE))
  {
    // This is the start of a new mains cycle
    cycleCount++; 
#ifdef DEBUG
    delay(14); //  <<----- to make the 'debug' rate match 50Hz    
#endif
    timeNow = millis();
    
    if (recordingNow == true)
    {
      // this is the start of the cycle AFTER the one where samples were collected, so
      // the data that has already been collected can now be displayed.
      recordingNow = false;
      firstLoop = true;
      
      Serial.print("cycleCount ");
      Serial.print(cycleCount);
      Serial.print(",  samplesDuringThisMainsCycle ");
      Serial.println(samplesDuringThisMainsCycle);

      // First, display the stored data as waveforms
      // -------------------------------------------
      int voltage, current, psV;
      int minVoltage = 1023, minCurrent = 1023, minPSV = 1023;
      int maxVoltage = 0, maxCurrent = 0, maxPSV = 0;
      
      int scaledV, scaledI, scaledPSV;
      int scaledVoltage[100];
      int scaledCurrent[100];
      int scaledPSVoltage[100];
      
      for (int index = 0; index < samplesDuringThisMainsCycle; index++) 
      {
        strcpy(newLine, blankLine);
        voltage = storedSampleV[index]; 
        current = storedSampleI[index]; 
        psV = storedPhaseShiftedV[index]; 

        if (voltage < minVoltage){minVoltage = voltage;}
        if (voltage > maxVoltage){maxVoltage = voltage;}
        if (current < minCurrent){minCurrent = current;}
        if (current > maxCurrent){maxCurrent = current;}
        if (psV < minPSV){minPSV = psV;}
        if (psV > maxPSV){maxPSV = psV;}
    
        newLine[40] = '.';
        scaledV = map(voltage, 0, 1023, 0, 80);
        newLine[scaledV] = 'v';
        scaledI = map(current, 0, 1023, 0, 80);
        newLine[scaledI] = 'c';
        scaledPSV = map(psV, 0, 1023, 0, 80);
 //       newLine[scaledPSV] = 'p';
       
        scaledVoltage[index] = scaledV;  // retain these scaled values 
        scaledCurrent[index] = scaledI;  // for use in phase diagram       
        scaledPSVoltage[index] = scaledPSV;  // for use in phase diagram       
        
        if ((index % 1) == 0) // change this to "% 1" for full resolution
        {
          Serial.println(newLine);
        }
      }    
      
      Serial.print("PHASECAL = ");  Serial.println(PHASECAL);
      Serial.print("minVoltage ");  Serial.print(minVoltage);
      Serial.print(",  maxVoltage ");  Serial.println(maxVoltage);
      Serial.print("minCurrent ");  Serial.print(minCurrent);
      Serial.print(",  maxCurrent ");  Serial.println(maxCurrent);
//      Serial.print("minPSV ");  Serial.print(minPSV);
//      Serial.print(",  maxPSV ");  Serial.println(maxPSV);


      // Next, display the same data as a phase diagram
      // ----------------------------------------------
      
      Serial.println();
      Serial.println();
      
//      for (int row = (samplesDuringThisMainsCycle - 1); row >= 0; row--) 
      for (int row = 80; row >= 0; row--) 
      {
        strcpy(newLine, blankLine);
        newLine[40] = '.';
        
        if (row == 40) 
        {
          for (int temp = 1; temp < 80; temp++) 
          {
            newLine[temp] = '.';
          }
        }    
/*    
        Serial.print ("row# = ");
        Serial.print (row);         
        Serial.print (",  scaledV = ");
        Serial.print (scaledVoltage[row]);          
        Serial.print (",  scaledI = ");
        Serial.println (scaledCurrent[row]);          
*/
        // Lissajous diagram
        for (int index_v = 0; index_v < (samplesDuringThisMainsCycle - 1); index_v++)
        {
          if (scaledVoltage[index_v] == row)
 //         if (scaledPSVoltage[index_v] == row)
          {
            int charNo = scaledCurrent[index_v];
            newLine[charNo] = 'o';            
          }
        }
        
        
        if ((row % 1) == 0) // change this to "% 1" for full resolution
        {
          Serial.println(newLine);
          delay (20);
        }
      }
      


      pause();
      
    }
    
    if((cycleCount % 50) == 1)
    {     
      if (timeNow > recordingMayStartAt)    
      {
        // we're off!
//        Serial.println( " <recording data ...> ");
//        Serial.println();
        recordingNow = true;
      } 
      else
      {
        Serial.println((int)(recordingMayStartAt - timeNow) / 1000);
//        Serial.print("  ");
//        Serial.println(cycleCount);
      }  
    }
    
    samplesDuringThisMainsCycle = 0;
    
  } // end of processing for first +ve reading in each mains cycle
  
  if (recordingNow == true)
  {
    storedSampleV[samplesDuringThisMainsCycle] = sampleV;
    storedSampleI[samplesDuringThisMainsCycle] = sampleI;
    storedPhaseShiftedV[samplesDuringThisMainsCycle] = (int)phaseShiftedV;

    if ((polarityNow == NEGATIVE) && (polarityOfLastReading == POSITIVE))
    {
    // uncomment this block to show the effect of Serial operations (approx 2mS delay)
/*
      Serial.print ("I've recorded ");
      Serial.print (samplesDuringThisMainsCycle);
      Serial.println (" samples so far ... ");
*/
    }    
  }
  samplesDuringThisMainsCycle++;
} // end of loop()


#ifdef DEBUG
// function to synthesize voltage samples for DEBUG mode only
float getVoltageSample_4debug(int index)
{
  if (index >= noOfVoltageSamplesPerCycle_4debug) {
    index -= noOfVoltageSamplesPerCycle_4debug; }
  if (index < 0) {
    index += noOfVoltageSamplesPerCycle_4debug; }
  
  return voltageSamples_4debug[index] + 500;
}
#endif

// When called, this next function will cause the code to pause 
// until any key, followed by [C/R], is pressed.  
//

void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0)
    {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      if (dummyByte == 'g') {
        done++; }
    }
  }    
}

