/* This tool performs several runs using a different value of PHASECAL each
 * time.  Power factor is displayed on each occasion.  When measuring current 
 * through a resistive load, this allows the optimal value to be determined.
 *
 * The code is loosly based on Mk2_PV_Router.  Note that current is now 
 * measured before voltage.
 *
 *                  Robin Emley (calypso_rae on Open Energy Monitor Forum)
 *                  August 2012
 */


#define POSITIVE 1
#define NEGATIVE 0

byte voltageSensorPin = 2;
byte currentSensorPin = 1;

long cycleCount = 0;
int samplesDuringThisMainsCycle;
int samplesDuringPFwindow;
float cyclesPerSecond = 50; // use float to ensure accurate maths

long noOfSamplePairs = 0;
byte polarityNow = NEGATIVE; // probably not important, but better than being indeterminate

boolean beyondStartUpPhase = false;

float energyInBucket = 0;                                                 

int sampleV,sampleI;   // voltage & current samples are integers in the ADC's input range 0 - 1023 
int lastSampleV;     // stored value from the previous loop (HP filter is for voltage samples only)         

float lastFilteredV,filteredV;  //  voltage values after filtering to remove the DC offset,
                                 //   and the stored values from the previous loop             
float prevDCoffset;          // <<--- for LPF to quantify the DC offset
float DCoffset;              // <<--- for LPF 
float cumVdeltasThisCycle;   // <<--- for LPF 
float sumP;   //  cumulative sum of power calculations within this mains cycles
float sumV_forPF, sumI_forPF, sumP_forPF;
float sampleVminusDC;
float lastSampleVminusDC;

float PHASECAL;
float PHASECAL_MIN;
float PHASECAL_MAX;
float PHASECAL_INCREMENT;

float POWERCAL;  // To convert the product of raw V & I samples into Joules.  


void setup()
{  
  Serial.begin(9600);
  
  char inbuf[10];
 
      Serial.println ("value of PHASECAL_MAX?");
      while (Serial.available() == 0)  
        delay(100);
      Serial.readBytesUntil('\n', inbuf, 10);
      PHASECAL_MAX = atof(inbuf);
      Serial.print(" PHASECAL_MAX = "); 
      Serial.println(PHASECAL_MAX);     
      Serial.flush();

      Serial.println ("value of PHASECAL_MIN?");
      while (Serial.available() == 0)  
        delay(100);
      Serial.readBytesUntil('\n', inbuf, 10);
      PHASECAL_MIN = atof(inbuf);
      Serial.print(" PHASECAL_MIN = "); 
      Serial.println(PHASECAL_MIN);     
      Serial.flush();

      Serial.println ("value of PHASECAL_INCREMENT?");
      while (Serial.available() == 0)  
        delay(100);
      Serial.readBytesUntil('\n', inbuf, 10);
      PHASECAL_INCREMENT = atof(inbuf);
      Serial.print(" PHASECAL_INCREMENT = "); 
      Serial.println(PHASECAL_INCREMENT);     
      Serial.flush();

      PHASECAL = PHASECAL_MIN;
    

  Serial.println("In NORMAL mode ..."); 
  Serial.println(); 

  POWERCAL = 0.1; // not needed for this application
                  
  Serial.print("The initial value of PHASECAL is ");     
  Serial.println(PHASECAL);     
}


void loop() // each loop is for one pair of V & I measurements
{
  noOfSamplePairs++;              // for stats only
  samplesDuringThisMainsCycle++;  // for power calculation at the start of each mains cycle
  samplesDuringPFwindow++;
  lastSampleV=sampleV;            // for digital high-pass filter
  lastSampleVminusDC=sampleVminusDC;  // for Power Factor calculation
  lastFilteredV = filteredV;   
  
  sampleI = analogRead(currentSensorPin);                 //Read in raw current signal
  sampleV = analogRead(voltageSensorPin);                 //Read in raw voltage signal

  sampleVminusDC = sampleV - DCoffset; // This value is needed more than once, so is best 
                                               // determined near the top of the loop. The equivalent
                                               // value for current is deferred until it is needed.  

  filteredV = 0.996*(lastFilteredV+sampleV-lastSampleV);
   

  // Establish the polarities of the latest and previous filtered voltage samples
  byte polarityOfLastReading = polarityNow;
  if(filteredV >= 0) 
    polarityNow = POSITIVE; 
  else 
    polarityNow = NEGATIVE;
 
  if (polarityNow == POSITIVE)
  {
    // --------------------------------------------------------
    // Start of processing that is specific to positive Vsamples
    // --------------------------------------------------------
     
    if (polarityOfLastReading != POSITIVE)
    {
      // This is the start of a new mains cycle
      cycleCount++; // for stats only

      /* update the Low Pass Filter for DC-offset removal
       */
      prevDCoffset = DCoffset;
      DCoffset = prevDCoffset + (0.01 * cumVdeltasThisCycle); 

      //  Calculate the real power of all instantaneous measurements taken during the 
      //  previous mains cycle, and determine the gain (or loss) in energy.
      float realPower = POWERCAL * sumP / (float)samplesDuringThisMainsCycle;
      float realEnergy = realPower / cyclesPerSecond;

      if (beyondStartUpPhase == true)
      {  
        // Providing that the DC-blocking filters have had sufficient time to settle,    
        // add this power contribution to the energy bucket
        energyInBucket += realEnergy;              
      }
      else
      {  
        // wait until the DC-blocking filters have had time to settle
        if(cycleCount > 100) // 100 mains cycles is 2 seconds
          beyondStartUpPhase = true;
      }
      /* clear the per-cycle accumulators for use in this new mains cycle.  
       */
      sumP = 0;
      samplesDuringThisMainsCycle = 0;
      cumVdeltasThisCycle = 0;
      
      if (((cycleCount % 50) == 0) && (cycleCount > 0))
      {
         float realPower = sumP_forPF / (float)samplesDuringPFwindow;
         float Vrms =  sqrt(sumV_forPF / (float)samplesDuringPFwindow); 
         float Irms =  sqrt(sumI_forPF / (float)samplesDuringPFwindow); 
         float apparentPower = Vrms * Irms;
         float powerFactor = realPower / apparentPower;
         Serial.print("real power = ");
         Serial.print(realPower);
         Serial.print(", apparent power = ");
         Serial.print(apparentPower);
         Serial.print(", power factor = ");
         Serial.println(powerFactor);
         sumV_forPF = 0;
         sumI_forPF = 0;
         sumP_forPF = 0;
         samplesDuringPFwindow = 0;
      }
      
      if (((cycleCount % 250) == 0) && (cycleCount > 0))
      {
         if (PHASECAL < PHASECAL_MAX)
         {
           PHASECAL += PHASECAL_INCREMENT;
           Serial.print ("PHASECAL is now ");
           Serial.println (PHASECAL);          
         }
         else
         {
           Serial.print("End of run, please restart to repeat");
           pause();
         }
         
      }
      
    } // end of processing that is specific to the first +ve Vsample in each new cycle
  }  // end of processing that is specific to positive Vsamples
  
  
  // Processing for ALL Vsamples, both positive and negative
  //------------------------------------------------------------
  
  float  sampleIminusDC = sampleI - DCoffset;
  float  phaseShiftedVminusDC = 
                lastSampleVminusDC + PHASECAL * (sampleVminusDC - lastSampleVminusDC);   
  float instP = phaseShiftedVminusDC * sampleIminusDC; //  power contribution for this pair of V&I samples 
  sumP +=instP;     // cumulative power values for this mains cycle 

  cumVdeltasThisCycle += (sampleV - DCoffset); // for use with LP filter

  // Additional items to support power factor calculation
  sumP_forPF += instP;
  float sqV = phaseShiftedVminusDC * phaseShiftedVminusDC;
  sumV_forPF += sqV;
  float sqI = sampleIminusDC * sampleIminusDC;
  sumI_forPF += sqI;
  
} // end of loop()


/* A function which causes the code to pause until any key, followed by [C/R], is pressed.  
 */
void pause()
{
  byte done = false;
  byte dummyByte;
   
  while (done != true)
  {
    if (Serial.available() > 0)
    {
      dummyByte = Serial.read(); // to 'consume' the incoming byte
      done++;
    }
  }    
}

